/**
 * NBA market discovery and transformation
 */

import type { KalshiApiClient, KalshiMarket, KalshiOrderbook, KalshiSeries } from '../types';
import type { NBAMarketRow } from '../types';

const NBA_KEYWORDS = ['NBA', 'basketball', 'basket ball'];
// NBA games can be listed weeks ahead; keep a wider default window.
const DAYS_AHEAD = 30;

/**
 * Extract team names from market title
 * Handles formats like "Lakers @ Warriors", "Lakers vs Warriors", etc.
 */
function extractTeams(title: string, subtitle?: string): { away: string; home: string } | null {
  const text = `${title} ${subtitle || ''}`.toUpperCase();

  // Common patterns
  const patterns = [
    /(\w+)\s+@\s+(\w+)/,
    /(\w+)\s+VS\s+(\w+)/,
    /(\w+)\s+VS\.\s+(\w+)/,
    /(\w+)\s+V\s+(\w+)/,
    /(\w+)\s+AT\s+(\w+)/,
  ];

  for (const pattern of patterns) {
    const match = text.match(pattern);
    if (match) {
      return { away: match[1], home: match[2] };
    }
  }

  // Fallback: try to split by common delimiters
  const parts = text.split(/[@VS|VS.|V|AT]/);
  if (parts.length === 2) {
    return {
      away: parts[0].trim(),
      home: parts[1].trim(),
    };
  }

  return null;
}

/**
 * Compute implied probability from bid/ask or yes_price/no_price
 */
function computeProbability(market: KalshiMarket, side: 'yes' | 'no'): number | null {
  // Try yes_price/no_price first
  if (side === 'yes' && market.yes_price !== undefined) {
    return market.yes_price;
  }
  if (side === 'no' && market.no_price !== undefined) {
    return market.no_price;
  }

  // Fallback to mid-price from bid/ask
  if (side === 'yes') {
    if (market.yes_bid !== null && market.yes_ask !== null) {
      return (market.yes_bid + market.yes_ask) / 2;
    }
    if (market.yes_bid !== null) return market.yes_bid;
    if (market.yes_ask !== null) return market.yes_ask;
    if (market.last_price !== null) return market.last_price;
  } else {
    if (market.no_bid !== null && market.no_ask !== null) {
      return (market.no_bid + market.no_ask) / 2;
    }
    if (market.no_bid !== null) return market.no_bid;
    if (market.no_ask !== null) return market.no_ask;
    if (market.last_price !== null) return 1 - market.last_price;
  }

  return null;
}

/**
 * Compute liquidity from orderbook or market data
 *
 * API returns orderbook as { yes: [[price, qty], ...], no: [[price, qty], ...] }
 * where each side contains bids only (arrays of [price, quantity] tuples)
 */
async function computeLiquidity(
  market: KalshiMarket,
  orderbook: KalshiOrderbook | null,
  side: 'yes' | 'no'
): Promise<number> {
  // If we have orderbook, sum depth near best price
  if (orderbook) {
    // Get the bids for the requested side (as [price, qty] tuples)
    const bids = side === 'yes' ? orderbook.yes : orderbook.no;

    if (!Array.isArray(bids) || bids.length === 0) {
      return market.liquidity || market.open_interest || 0;
    }

    // Find best bid (highest price)
    const bestBid = bids[0]?.[0] || 0;
    const band = 5; // 5 cents band (prices are in cents 0-100)

    let liquidity = 0;

    // Sum bids within band of best price
    for (const [price, qty] of bids) {
      if (price >= bestBid - band) {
        liquidity += qty;
      } else {
        break; // Bids are sorted high to low
      }
    }

    return liquidity;
  }

  // Fallback: use open_interest or volume as proxy
  if (market.open_interest) {
    return market.open_interest;
  }

  if (market.volume) {
    return market.volume;
  }

  // Use liquidity field if available
  if (market.liquidity) {
    return market.liquidity;
  }

  return 0;
}

/**
 * Cached sports series to avoid redundant getSeries() calls within the same session.
 */
let cachedSportsSeries: KalshiSeries[] | null = null;

/**
 * Cached market discovery results to avoid duplicate API calls.
 * Both createNbaStream and createConsolidatedNbaStream call discoverNBAMarkets();
 * this cache lets the second call reuse the first result within a short window.
 */
let cachedMarkets: { markets: KalshiMarket[]; expiresAt: number } | null = null;
const MARKET_CACHE_TTL_MS = 30_000; // 30 seconds

async function getSportsSeries(api: KalshiApiClient): Promise<KalshiSeries[]> {
  if (cachedSportsSeries) return cachedSportsSeries;
  cachedSportsSeries = await api.getSeries({ category: 'Sports' });
  return cachedSportsSeries;
}

/** Prime the series cache with an already-fetched result. */
export function primeSeriesCache(series: KalshiSeries[]): void {
  cachedSportsSeries = series;
}

/** Clear the series cache (e.g., on disconnect). */
export function clearSeriesCache(): void {
  cachedSportsSeries = null;
  cachedMarkets = null;
}

/**
 * Discover NBA markets
 */
export async function discoverNBAMarkets(
  api: KalshiApiClient,
  daysAhead: number = DAYS_AHEAD
): Promise<KalshiMarket[]> {
  // Return cached results if still fresh
  if (cachedMarkets && Date.now() < cachedMarkets.expiresAt) {
    return cachedMarkets.markets;
  }

  const cacheAndReturn = (results: KalshiMarket[]): KalshiMarket[] => {
    cachedMarkets = { markets: results, expiresAt: Date.now() + MARKET_CACHE_TTL_MS };
    return results;
  };

  try {
    // Strategy 1: Get series filtered by Sports category
    const series = await getSportsSeries(api);

    // Filter for NBA-related series
    const nbaSeries = series.filter((s) => {
      const title = s.title?.toUpperCase?.() ?? '';
      const category = s.category?.toUpperCase?.() ?? '';
      const subcategory = s.subcategory?.toUpperCase?.() ?? '';
      const ticker = s.series_ticker?.toUpperCase?.() ?? '';

      // Many NBA game series don't include "NBA" in the title; rely on the ticker prefix.
      if (ticker.startsWith('KXNBAGAME')) return true;

      return NBA_KEYWORDS.some(
        (keyword) =>
          title.includes(keyword) || category.includes(keyword) || subcategory.includes(keyword)
      );
    });

    // Strategy 2: If no series found, try getting markets directly and filter
    if (nbaSeries.length === 0) {
      // Try direct series ticker fetch first (most reliable for this workspace).
      try {
        const byTicker = await api.getMarkets({
          series_ticker: 'KXNBAGAME',
          status: 'open',
          limit: 200,
        });
        if (byTicker.length > 0) return cacheAndReturn(byTicker);
      } catch {
        // fall through
      }

      const allMarkets = await api.getMarkets({ status: 'open', limit: 200 });
      return cacheAndReturn(
        allMarkets.filter((m) => {
          const mt = m.market_ticker?.toUpperCase?.() ?? '';
          if (mt.startsWith('KXNBAGAME-')) return true;

          const title = m.title?.toUpperCase?.() ?? '';
          const subtitle = m.subtitle?.toUpperCase?.() ?? '';
          const eventTicker = m.event_ticker?.toUpperCase?.() ?? '';
          return NBA_KEYWORDS.some(
            (keyword) =>
              title.includes(keyword) || subtitle.includes(keyword) || eventTicker.includes(keyword)
          );
        })
      );
    }

    // Get markets for all NBA series in parallel
    const seriesResults = await Promise.all(
      nbaSeries.map((seriesItem) =>
        api.getMarkets({
          series_ticker: seriesItem.series_ticker,
          status: 'open',
          limit: 200,
        })
      )
    );
    const markets = seriesResults.flat();

    // Filter by date (upcoming games)
    const now = new Date();
    const cutoff = new Date(now.getTime() + daysAhead * 24 * 60 * 60 * 1000);

    return cacheAndReturn(
      markets.filter((m) => {
        if (!m.expiry_time && !m.close_time) return false;
        const marketDate = new Date(m.expiry_time || m.close_time);
        return marketDate >= now && marketDate <= cutoff;
      })
    );
  } catch (error) {
    console.error('Error discovering NBA markets:', error);
    return [];
  }
}

/**
 * Transform markets to NBA rows
 */
export async function transformToNBARows(
  markets: KalshiMarket[],
  api: KalshiApiClient
): Promise<NBAMarketRow[]> {
  // Pre-filter markets to only those with extractable teams/sides
  const prepared: Array<{
    market: KalshiMarket;
    teams: { away: string; home: string };
    side: 'away' | 'home';
  }> = [];

  for (const market of markets) {
    const teams = extractTeams(market.title, market.subtitle);
    if (!teams) continue;

    const isAwayMarket =
      market.title.toUpperCase().includes(teams.away) &&
      !market.title.toUpperCase().includes(teams.home);
    const isHomeMarket =
      market.title.toUpperCase().includes(teams.home) &&
      !market.title.toUpperCase().includes(teams.away);

    const side = isAwayMarket ? 'away' : isHomeMarket ? 'home' : null;
    if (!side) continue;

    prepared.push({ market, teams, side });
  }

  // Fetch all orderbooks in parallel (batched to avoid hammering)
  const BATCH = 20;
  const orderbooks: (KalshiOrderbook | null)[] = new Array(prepared.length).fill(null);
  for (let i = 0; i < prepared.length; i += BATCH) {
    const batch = prepared.slice(i, i + BATCH);
    const results = await Promise.all(
      batch.map((p) => api.getOrderbook(p.market.market_ticker).catch(() => null))
    );
    for (let j = 0; j < results.length; j++) {
      orderbooks[i + j] = results[j];
    }
  }

  // Build rows
  const rows: NBAMarketRow[] = [];
  for (let i = 0; i < prepared.length; i++) {
    const { market, teams, side } = prepared[i];
    const orderbook = orderbooks[i];

    const prob = computeProbability(market, 'yes');
    const liq = await computeLiquidity(market, orderbook, 'yes');

    const gameTime = market.expiry_time || market.close_time || market.open_time;
    const sortKey = `${gameTime}_${teams.away}`;

    rows.push({
      market_ticker: market.market_ticker,
      gameDate: gameTime,
      gameTime: gameTime,
      awayTeam: teams.away,
      homeTeam: teams.home,
      awayProb: side === 'away' ? prob : null,
      homeProb: side === 'home' ? prob : null,
      awayLiq: side === 'away' ? liq : 0,
      homeLiq: side === 'home' ? liq : 0,
      sortKey,
      awayYesBid: side === 'away' ? market.yes_bid : null,
      awayYesAsk: side === 'away' ? market.yes_ask : null,
      homeYesBid: side === 'home' ? market.yes_bid : null,
      homeYesAsk: side === 'home' ? market.yes_ask : null,
    });
  }

  // Group by event and merge away/home markets
  const grouped = new Map<string, NBAMarketRow>();

  for (const row of rows) {
    const key = `${row.gameTime}_${row.awayTeam}_${row.homeTeam}`;
    const existing = grouped.get(key);

    if (existing) {
      existing.awayProb = existing.awayProb ?? row.awayProb;
      existing.homeProb = existing.homeProb ?? row.homeProb;
      existing.awayLiq = Math.max(existing.awayLiq, row.awayLiq);
      existing.homeLiq = Math.max(existing.homeLiq, row.homeLiq);
      existing.awayYesBid = existing.awayYesBid ?? row.awayYesBid;
      existing.awayYesAsk = existing.awayYesAsk ?? row.awayYesAsk;
      existing.homeYesBid = existing.homeYesBid ?? row.homeYesBid;
      existing.homeYesAsk = existing.homeYesAsk ?? row.homeYesAsk;
    } else {
      grouped.set(key, { ...row });
    }
  }

  return Array.from(grouped.values());
}

/**
 * Generate mock NBA data for testing
 */
export function generateMockNBARows(): NBAMarketRow[] {
  const teams = [
    ['Lakers', 'Warriors'],
    ['Celtics', 'Heat'],
    ['Nets', 'Bucks'],
    ['Suns', 'Nuggets'],
    ['Mavericks', 'Clippers'],
  ];

  const now = new Date();
  const rows: NBAMarketRow[] = [];

  for (let i = 0; i < teams.length; i++) {
    const [away, home] = teams[i];
    const gameTime = new Date(now.getTime() + (i + 1) * 24 * 60 * 60 * 1000);
    gameTime.setHours(19, 0, 0, 0); // 7 PM

    rows.push({
      market_ticker: `MOCK-${i}`,
      gameDate: gameTime.toISOString(),
      gameTime: gameTime.toISOString(),
      awayTeam: away,
      homeTeam: home,
      awayProb: 0.45 + Math.random() * 0.1,
      homeProb: 0.45 + Math.random() * 0.1,
      awayLiq: Math.random() * 50000 + 10000,
      homeLiq: Math.random() * 50000 + 10000,
      sortKey: `${gameTime.toISOString()}_${away}`,
    });
  }

  return rows;
}
