import { MONTH_MAP } from '../kalshiTicker/types';

export interface KalshiNbaEventInfo {
  eventTicker: string; // KXNBAGAME-26JAN15BOSMIA
  dateYyyyMmDd: string; // 2026-01-15
  awayCode: string; // BOS
  homeCode: string; // MIA
}

export interface KalshiNbaMarketInfo {
  eventTicker: string;
  marketTicker: string; // KXNBAGAME-26JAN15BOSMIA-BOS
  teamCode: string; // BOS
}

export function parseKalshiNbaEventTicker(eventTickerRaw: string): KalshiNbaEventInfo | null {
  const eventTicker = String(eventTickerRaw || '')
    .trim()
    .toUpperCase();
  // Example: KXNBAGAME-26JAN15BOSMIA
  const parts = eventTicker.split('-');
  if (parts.length < 2) return null;
  const mid = parts[1] || '';
  if (mid.length < 7) return null; // YY + MON + DD + (codes...)

  const yy = Number(mid.slice(0, 2));
  const monStr = mid.slice(2, 5);
  const dd = Number(mid.slice(5, 7));
  const mm = MONTH_MAP[monStr];
  if (!Number.isFinite(yy) || !Number.isFinite(dd) || !mm) return null;

  const year = 2000 + yy;
  const month = String(mm).padStart(2, '0');
  const day = String(dd).padStart(2, '0');

  // Away/home codes are the final 6 chars of the ticker (per `nba/kalshi/feeder_ws_to_json.py`)
  const tail = eventTicker.slice(-6);
  if (tail.length !== 6) return null;
  const awayCode = tail.slice(0, 3).toUpperCase();
  const homeCode = tail.slice(3).toUpperCase();
  if (!/^[A-Z]{3}$/.test(awayCode) || !/^[A-Z]{3}$/.test(homeCode)) return null;

  return {
    eventTicker,
    dateYyyyMmDd: `${year}-${month}-${day}`,
    awayCode,
    homeCode,
  };
}

export function parseKalshiNbaMarketTicker(marketTickerRaw: string): KalshiNbaMarketInfo | null {
  const marketTicker = String(marketTickerRaw || '')
    .trim()
    .toUpperCase();
  // Examples observed:
  // - KXNBAGAME-26JAN15BOSMIA-BOS
  // - KXNBAGAME-26JAN23BOSBKN--BOS   (note the double-dash before team code)
  //
  // Prefer the `--TEAM` split when present, since simple `-` splitting
  // will include an empty segment and produce a trailing '-' event ticker.
  let eventTicker = '';
  let teamCode = '';
  if (marketTicker.includes('--')) {
    const parts2 = marketTicker.split('--').filter(Boolean);
    if (parts2.length !== 2) return null;
    eventTicker = parts2[0] ?? '';
    teamCode = parts2[1] ?? '';
  } else {
    const parts = marketTicker.split('-');
    if (parts.length < 3) return null;
    teamCode = parts[parts.length - 1] ?? '';
    eventTicker = parts.slice(0, -1).join('-');
  }
  if (!/^[A-Z]{3}$/.test(teamCode)) return null;
  if (!eventTicker.startsWith('KXNBAGAME-')) return null;
  return { eventTicker, marketTicker, teamCode };
}
