import type { BookLevel, TeamSide, Venue, VenueGameBooks } from './types';

export const KALSHI_TAKER_FEE_RATE = 0.07;
export const POLY_TAKER_FEE_RATE = 0.0;

export function bidToAsk(bidCents: number | null | undefined): number | null {
  if (bidCents === null || bidCents === undefined) return null;
  const v = Number(bidCents);
  if (!Number.isFinite(v)) return null;
  return 100 - v;
}

/**
 * Mirrors `nba/static/index.html` taker fee approximation.
 * Fee is computed on a notional `contracts` (default 1000) and then spread per contract.
 */
export function applyTakerFee(
  priceCents: number | null,
  feeRate: number,
  contracts = 1000
): number | null {
  if (priceCents === null) return null;
  const p = Number(priceCents);
  if (!Number.isFinite(p) || p <= 0 || p >= 100) return p;

  const P = p / 100.0;
  const feeTotalCents = Math.ceil(feeRate * contracts * P * (1 - P) * 100);
  const feePerContractCents = feeTotalCents / contracts;
  return p + feePerContractCents;
}

export function takerFeeForVenue(venue: Venue): number {
  return venue === 'kalshi' ? KALSHI_TAKER_FEE_RATE : POLY_TAKER_FEE_RATE;
}

export function getEffectivePriceFromBid(args: {
  bidCents: number | null;
  isTakerMode: boolean;
  takerFeeRate: number;
}): number | null {
  const { bidCents, isTakerMode, takerFeeRate } = args;
  if (bidCents === null) return null;
  if (!isTakerMode) return bidCents;
  const ask = bidToAsk(bidCents);
  return applyTakerFee(ask, takerFeeRate);
}

export interface TopResult {
  /** Display price (maker bid or taker ask+fee), cents */
  priceCents: number | null;
  /** Liquidity (contracts) at the selected raw bid */
  liq: number | null;
  /** Raw bid price used (cents), used for order placement */
  rawBidPriceCents: number | null;
}

function bestBid(levels: BookLevel[] | null | undefined): BookLevel | null {
  if (!Array.isArray(levels) || levels.length === 0) return null;
  let best = levels[0];
  for (const lvl of levels) {
    if (lvl && Number(lvl.priceCents) > Number(best.priceCents)) best = lvl;
  }
  return best ?? null;
}

/**
 * Mirrors `topNoFromLevels` from `nba/static/index.html`.
 *
 * Interpretation:
 * - Team win price is computed as **NO on opponent**.
 * - In maker mode, use the opponent NO bid ladder.
 * - In taker mode, use opponent YES bids to infer NO ask (100 - best YES bid), then apply taker fee.
 *
 * Important: in taker mode, if YES ladder is empty, return null (do not fall back).
 */
export function topNoFromLevels(args: {
  noLevels: BookLevel[] | null | undefined;
  yesLevels: BookLevel[] | null | undefined;
  isTakerMode: boolean;
  takerFeeRate: number;
}): TopResult {
  const { noLevels, yesLevels, isTakerMode, takerFeeRate } = args;

  if (!Array.isArray(noLevels) || noLevels.length === 0) {
    return { priceCents: null, liq: null, rawBidPriceCents: null };
  }

  if (isTakerMode) {
    // Mirror nba JS behavior: `if (isTakerMode && yesLevels)` is true even for []
    if (!Array.isArray(yesLevels) || yesLevels.length === 0) {
      return { priceCents: null, liq: null, rawBidPriceCents: null };
    }
    const bestYes = bestBid(yesLevels);
    if (!bestYes) return { priceCents: null, liq: null, rawBidPriceCents: null };
    const noAsk = bidToAsk(bestYes.priceCents);
    const effective = applyTakerFee(noAsk, takerFeeRate);
    return {
      priceCents: effective,
      liq: bestYes.size ?? null,
      rawBidPriceCents: bestYes.priceCents ?? null,
    };
  }

  const bestNo = bestBid(noLevels);
  if (!bestNo) return { priceCents: null, liq: null, rawBidPriceCents: null };
  const effective = getEffectivePriceFromBid({
    bidCents: bestNo.priceCents ?? null,
    isTakerMode,
    takerFeeRate,
  });
  return {
    priceCents: effective,
    liq: bestNo.size ?? null,
    rawBidPriceCents: bestNo.priceCents ?? null,
  };
}

export interface TeamWinData {
  priceCents: number | null;
  liq: number | null;
  rawBidPriceCents: number | null;
  marketTicker: string | null;
  tokenId?: string;
  noLevels: BookLevel[];
  yesLevels: BookLevel[];
}

/**
 * Port of `getTeamWinData()` from `nba/static/index.html`.
 *
 * This returns the price for "TEAM WINS" computed as **NO on the opponent market**.
 * For teamSide=away, opponent is the *home* market.
 */
export function getTeamWinData(args: {
  game: VenueGameBooks | null | undefined;
  venue: Venue;
  teamSide: TeamSide;
  isTakerMode: boolean;
}): TeamWinData {
  const { game, venue, teamSide, isTakerMode } = args;
  if (!game) {
    return {
      priceCents: null,
      liq: null,
      rawBidPriceCents: null,
      marketTicker: null,
      noLevels: [],
      yesLevels: [],
    };
  }

  const opp = teamSide === 'away' ? game.markets.home : game.markets.away;
  const noLevels = opp?.no ?? [];
  const yesLevels = opp?.yes ?? [];
  const top = topNoFromLevels({
    noLevels,
    yesLevels,
    isTakerMode,
    takerFeeRate: takerFeeForVenue(venue),
  });

  return {
    priceCents: top.priceCents,
    liq: top.liq,
    rawBidPriceCents: top.rawBidPriceCents,
    marketTicker: opp?.marketTicker ?? null,
    tokenId: opp?.tokenId,
    noLevels,
    yesLevels,
  };
}

export function next3NoLadders(
  noLevels: BookLevel[],
  bestBidPriceCents: number | null
): BookLevel[] {
  if (!Array.isArray(noLevels) || noLevels.length === 0) return [];
  if (bestBidPriceCents === null) return [];
  const below = noLevels.filter((lvl) => Number(lvl.priceCents) < bestBidPriceCents);
  below.sort((a, b) => Number(b.priceCents) - Number(a.priceCents));
  return below.slice(0, 3);
}

export function next3YesLadders(
  yesLevels: BookLevel[],
  bestBidPriceCents: number | null
): BookLevel[] {
  if (!Array.isArray(yesLevels) || yesLevels.length === 0) return [];
  if (bestBidPriceCents === null) return [];
  const below = yesLevels.filter((lvl) => Number(lvl.priceCents) < bestBidPriceCents);
  below.sort((a, b) => Number(b.priceCents) - Number(a.priceCents));
  return below.slice(0, 3);
}

export function liquidityDollars(
  liqContracts: number | null,
  priceCents: number | null
): number | null {
  if (liqContracts === null || priceCents === null) return null;
  const dollars = (Number(liqContracts) * Number(priceCents)) / 100;
  return Number.isFinite(dollars) ? dollars : null;
}

/**
 * Compute order price from raw bid price.
 * In maker mode, returns the raw bid price.
 * In taker mode, converts bid to ask (100 - bid).
 */
export function computeOrderPriceFromRawBid(
  rawBidPriceCents: number | null,
  isTakerMode: boolean
): number | null {
  if (rawBidPriceCents === null) return null;
  if (!isTakerMode) return rawBidPriceCents;
  return bidToAsk(rawBidPriceCents);
}
