/**
 * Start Time Settings
 *
 * Kalshi reports game start times that are typically ~10 minutes early.
 * This setting allows configuring an offset to adjust the displayed/used start time.
 */

const STORAGE_KEY = 'nba-start-time-offset-minutes';
const DEFAULT_OFFSET_MINUTES = 9;

/**
 * Get the configured start time offset in minutes.
 * Positive values shift the start time later (e.g., +9 means add 9 minutes).
 */
export function getStartTimeOffsetMinutes(): number {
  const stored = localStorage.getItem(STORAGE_KEY);
  if (!stored) return DEFAULT_OFFSET_MINUTES;

  const parsed = parseInt(stored, 10);
  if (!Number.isFinite(parsed)) return DEFAULT_OFFSET_MINUTES;

  return parsed;
}

/**
 * Set the start time offset in minutes.
 */
export function setStartTimeOffsetMinutes(minutes: number): void {
  localStorage.setItem(STORAGE_KEY, String(minutes));
}

/**
 * Apply the offset to an ISO timestamp string.
 * Returns a new ISO timestamp with the offset applied.
 */
export function applyStartTimeOffset(isoTimestamp: string): string {
  const offset = getStartTimeOffsetMinutes();
  if (offset === 0) return isoTimestamp;

  const date = new Date(isoTimestamp);
  if (!Number.isFinite(date.getTime())) return isoTimestamp;

  date.setMinutes(date.getMinutes() + offset);
  return date.toISOString();
}
