export type Venue = 'kalshi' | 'polymarket';

export type TeamSide = 'away' | 'home';

export interface BookLevel {
  /** Price in cents, 0..100 (may be float in upstream snapshots; we keep number) */
  priceCents: number;
  /** Size/quantity at that price level */
  size: number;
}

/**
 * Orderbook levels as used by the client `nba/` project:
 * - `yes` and `no` arrays are both **bid** ladders (descending by price)
 * - Asks are derived by inversion:
 *   - YES ask = 100 - best NO bid
 *   - NO ask  = 100 - best YES bid
 */
export interface VenueMarketBook {
  /** Kalshi market_ticker, or synthetic market_ticker for Polymarket rows */
  marketTicker: string;
  /** Polymarket token_id (asset_id). Present only for polymarket */
  tokenId?: string;
  /** Polymarket condition ID (market id) for user-channel subscription */
  conditionId?: string;
  /** Polymarket tick size (from Gamma) */
  tickSize?: string;
  /** Polymarket negRisk flag (from Gamma) */
  negRisk?: boolean;

  /** YES bids (descending by price) */
  yes: BookLevel[];
  /** NO bids (descending by price) */
  no: BookLevel[];
}

export interface VenueGameBooks {
  /** Venue source */
  venue: Venue;

  /** Venue-native event identifier (Kalshi: event_ticker; Poly: slug/condition id) */
  eventId: string;

  awayCode: string;
  homeCode: string;

  /** Game legs */
  markets: {
    away: VenueMarketBook;
    home: VenueMarketBook;
  };

  /** Snapshot timestamp (ms) if available */
  tsMs?: number;
}

export interface ConsolidatedGameKeyParts {
  date: string; // YYYY-MM-DD
  awayCode: string;
  homeCode: string;
  /** Display name for away team/player */
  awayName?: string;
  /** Display name for home team/player */
  homeName?: string;
}

export function makeGameKey(parts: ConsolidatedGameKeyParts): string {
  return `${parts.date}|${parts.awayCode.toUpperCase()}|${parts.homeCode.toUpperCase()}`;
}

export interface ConsolidatedGameBooks extends ConsolidatedGameKeyParts {
  /** Stable key for UI expansion/sorting. */
  key: string;

  /** Start time in PT (HH:MM) if known */
  startTimePt?: string | null;

  /** Away team rotation number from The Odds API */
  awayRotation?: number | null;
  /** Home team rotation number from The Odds API */
  homeRotation?: number | null;

  kalshi?: VenueGameBooks | null;
  polymarket?: VenueGameBooks | null;
}
