import { createRelayRequest, relayHttp } from '@/lib/relayHttp';
import type { Sport } from '@/lib/sportsDiscovery';

export interface GammaMarketRecord {
  slug?: string;
  conditionId?: string;
  orderPriceMinTickSize?: number | string | null;
  enableNegRisk?: boolean | null;
  events?: Array<{ negRisk?: boolean | null }>;
  clobTokenIds?: unknown;
  outcomes?: unknown;
}

export interface PolyMarketInfo {
  slug: string;
  awayTokenId: string;
  homeTokenId: string;
  awayPriceCents?: number;
  homePriceCents?: number;
  liquidityUsd?: number;
  conditionId?: string;
  tickSize?: string;
  negRisk?: boolean;
}

export interface PolySpreadMarketInfo {
  slug: string;
  favoriteSide: 'away' | 'home';
  spreadValue: number;
  favoriteTokenId: string;
  underdogTokenId: string;
  favoritePriceCents?: number;
  underdogPriceCents?: number;
  liquidityUsd?: number;
  conditionId?: string;
  tickSize?: string;
  negRisk?: boolean;
}

export interface PolyTotalMarketInfo {
  slug: string;
  totalLine: number;
  overTokenId: string;
  underTokenId: string;
  overPriceCents?: number;
  underPriceCents?: number;
  liquidityUsd?: number;
  conditionId?: string;
  tickSize?: string;
  negRisk?: boolean;
}

const GAMMA_HEADERS: Record<string, string> = {
  Accept: 'application/json',
  'User-Agent': 'Mozilla/5.0',
};

/** Fetch from Gamma API with optional relay fallback */
async function fetchGammaJson(args: {
  url: string;
  useRelay: boolean;
  requestId: string;
}): Promise<unknown> {
  const fetchDirect = async (): Promise<unknown> => {
    const resp = await fetch(args.url, { method: 'GET', headers: GAMMA_HEADERS });
    if (!resp.ok) return null;
    return resp.json();
  };

  if (args.useRelay) {
    try {
      const envelope = createRelayRequest(args.requestId, 'GET', args.url, GAMMA_HEADERS);
      const resp = await relayHttp(envelope);
      if (resp.status < 400) {
        const data = JSON.parse(resp.body || 'null');
        if (data !== null) return data;
      }
    } catch {
      // Fall through to direct fetch
    }
  }

  try {
    return await fetchDirect();
  } catch {
    // Direct fetch can fail with CORS TypeError when called from the browser.
    // If both relay and direct fail, "no data" is the correct result.
    return null;
  }
}

/** Parse tickSize from Gamma record */
function parseTickSize(raw: number | string | null | undefined): string | undefined {
  if (raw === null || raw === undefined) return undefined;
  return typeof raw === 'string' ? raw : String(raw);
}

/** Parse negRisk from Gamma record */
function parseNegRisk(rec: {
  enableNegRisk?: boolean | null;
  events?: Array<{ negRisk?: boolean | null }>;
}): boolean | undefined {
  if (typeof rec?.events?.[0]?.negRisk === 'boolean') return Boolean(rec.events[0].negRisk);
  if (typeof rec?.enableNegRisk === 'boolean') return Boolean(rec.enableNegRisk);
  return undefined;
}

function parseOutcomes(raw: unknown): string[] {
  const toStrings = (arr: unknown[]): string[] =>
    arr.map((x) => String(x ?? '')).filter((s) => s.length > 0);
  if (Array.isArray(raw)) return toStrings(raw);
  if (typeof raw === 'string') {
    try {
      const parsed = JSON.parse(raw);
      if (Array.isArray(parsed)) return toStrings(parsed);
    } catch {
      return [];
    }
  }
  return [];
}

function parseClobTokenIds(raw: unknown): string[] {
  if (Array.isArray(raw)) return raw.map((x) => String(x)).filter(Boolean);
  if (typeof raw === 'string') {
    try {
      const parsed = JSON.parse(raw);
      if (Array.isArray(parsed)) return parsed.map((x) => String(x)).filter(Boolean);
    } catch {
      return [];
    }
  }
  return [];
}

export function makeNbaPolySlug(args: {
  dateYyyyMmDd: string;
  awayCode: string;
  homeCode: string;
}): string {
  return `nba-${args.awayCode.toLowerCase()}-${args.homeCode.toLowerCase()}-${args.dateYyyyMmDd}`;
}

function polySportPrefix(sport: Sport): string | null {
  if (sport === 'nba') return 'nba';
  if (sport === 'nfl') return 'nfl';
  if (sport === 'nhl') return 'nhl';
  if (sport === 'cbb') return 'cbb';
  if (sport === 'mlb') return 'mlb';
  if (sport === 'tennis-atp') return 'atp';
  if (sport === 'tennis-wta') return 'wta';
  // wcbb, ufc — no Polymarket equivalent
  return null;
}

export function makePolyGameSlug(args: {
  sport: Sport;
  dateYyyyMmDd: string;
  awayCode: string;
  homeCode: string;
}): string | null {
  const prefix = polySportPrefix(args.sport);
  if (!prefix) return null;
  return `${prefix}-${args.awayCode.toLowerCase()}-${args.homeCode.toLowerCase()}-${args.dateYyyyMmDd}`;
}

/**
 * Detect whether Gamma outcomes need to be swapped so that index 0 = away.
 *
 * Strategy (in priority order):
 * 1. If explicit awayHint/homeHint are provided, match outcomes against those.
 * 2. Fall back to slug-based matching: slug format `{sport}-{awayCode}-{homeCode}-{date}`.
 *
 * Returns true if a swap is needed (outcomes[0] matches the home team, not away).
 */
function shouldSwapOutcomes(
  slug: string,
  outcomes: string[],
  awayHint?: string,
  homeHint?: string
): boolean {
  if (outcomes.length < 2) return false;

  const outcome0 = normalizeToken(outcomes[0]);
  const outcome1 = normalizeToken(outcomes[1]);

  // Strategy 1: explicit team name hints (from Kalshi event data)
  if (awayHint && homeHint) {
    const awayNorm = normalizeToken(awayHint);
    const homeNorm = normalizeToken(homeHint);
    // Use last significant token (team mascot/city) for matching
    const awayLast = pickLastNameToken(awayHint);
    const homeLast = pickLastNameToken(homeHint);

    const o0MatchesAway = outcome0.includes(awayNorm) || outcome0.includes(awayLast);
    const o0MatchesHome = outcome0.includes(homeNorm) || outcome0.includes(homeLast);
    const o1MatchesAway = outcome1.includes(awayNorm) || outcome1.includes(awayLast);
    const o1MatchesHome = outcome1.includes(homeNorm) || outcome1.includes(homeLast);

    if (o0MatchesHome && !o0MatchesAway && o1MatchesAway && !o1MatchesHome) return true;
    if (o0MatchesAway && !o0MatchesHome && o1MatchesHome && !o1MatchesAway) return false;
  }

  // Strategy 2: slug-based (e.g. "nba-lal-bos-2026-02-19")
  const parts = slug.split('-');
  if (parts.length >= 4) {
    const awayCode = parts[1].toLowerCase();
    const homeCode = parts[2].toLowerCase();

    const o0HasAway = outcome0.includes(awayCode);
    const o0HasHome = outcome0.includes(homeCode);
    const o1HasAway = outcome1.includes(awayCode);
    const o1HasHome = outcome1.includes(homeCode);

    if (o0HasHome && !o0HasAway && o1HasAway && !o1HasHome) return true;
    if (o0HasAway && !o0HasHome && o1HasHome && !o1HasAway) return false;
  }

  // Ambiguous — default to no swap (matches existing assumption)
  return false;
}

/**
 * Resolve a Polymarket slug -> (awayTokenId, homeTokenId) using Gamma.
 *
 * Verifies outcome ordering against the slug's away/home convention.
 * If Gamma returns outcomes in [home, away] order, swaps tokens and prices.
 */
export async function resolvePolyMarketInfoBySlug(args: {
  slug: string;
  useRelay: boolean;
  awayName?: string;
  homeName?: string;
}): Promise<PolyMarketInfo | null> {
  const url = `https://gamma-api.polymarket.com/markets?slug=${encodeURIComponent(args.slug)}`;
  const data = await fetchGammaJson({
    url,
    useRelay: args.useRelay,
    requestId: `${Date.now()}-gamma`,
  });

  if (!Array.isArray(data) || data.length === 0) return null;
  const rec = data[0] as GammaMarketRecord;
  const tokenIds = parseClobTokenIds(rec?.clobTokenIds);
  if (tokenIds.length < 2) return null;
  const recAny = rec as GammaMarketRecord & { outcomePrices?: unknown; liquidity?: unknown };
  const prices = parseOutcomePrices(recAny.outcomePrices);
  const liquidityRaw = Number(recAny.liquidity);
  const outcomes = parseOutcomes(rec?.outcomes);

  const swap = shouldSwapOutcomes(args.slug, outcomes, args.awayName, args.homeName);
  if (swap) {
    console.warn(
      `[Poly] Swapping outcome order for ${args.slug}: outcomes=${JSON.stringify(outcomes)}`
    );
  }

  const i0 = swap ? 1 : 0;
  const i1 = swap ? 0 : 1;

  return {
    slug: args.slug,
    awayTokenId: String(tokenIds[i0]),
    homeTokenId: String(tokenIds[i1]),
    awayPriceCents: prices[i0],
    homePriceCents: prices[i1],
    liquidityUsd: Number.isFinite(liquidityRaw) ? liquidityRaw : undefined,
    conditionId: rec?.conditionId ? String(rec.conditionId) : undefined,
    tickSize: parseTickSize(rec?.orderPriceMinTickSize),
    negRisk: parseNegRisk(rec),
  };
}

interface GammaEventRecord {
  slug?: string;
  title?: string;
  startTime?: string;
  markets?: Array<{
    slug?: string;
    sportsMarketType?: string;
    conditionId?: string;
    orderPriceMinTickSize?: number | string | null;
    enableNegRisk?: boolean | null;
    events?: Array<{ negRisk?: boolean | null }>;
    clobTokenIds?: unknown;
    outcomePrices?: unknown;
    liquidity?: number | string | null;
  }>;
}

function gammaSeriesIdForSport(sport: Sport): string | null {
  if (sport === 'cbb') return '10470';
  if (sport === 'tennis-atp') return '10365';
  if (sport === 'tennis-wta') return '10366';
  return null;
}

/**
 * In-flight / recently-resolved cache for Gamma events-by-series fetches.
 * Prevents redundant identical requests when resolving many games for the
 * same sport concurrently (e.g. 100 CBB events all need series_id=10470).
 * Key: the full URL. Value: the in-flight fetch promise.
 */
const gammaEventsBySeriesCache = new Map<string, Promise<GammaEventRecord[]>>();

/** Clear the cached Gamma events-by-series responses. Call between hydration passes. */
export function clearGammaEventsCache(): void {
  gammaEventsBySeriesCache.clear();
}

/**
 * Fetch Gamma events for a sport series, deduplicating concurrent identical requests.
 * The first caller triggers the actual fetch; all concurrent callers await the same promise.
 */
async function fetchGammaEventsBySeries(args: {
  seriesId: string;
  useRelay: boolean;
}): Promise<GammaEventRecord[]> {
  const url = `https://gamma-api.polymarket.com/events?series_id=${encodeURIComponent(args.seriesId)}&tag_id=100639&active=true&closed=false&limit=500`;

  const cached = gammaEventsBySeriesCache.get(url);
  if (cached) return cached;

  const promise = fetchGammaJson({
    url,
    useRelay: args.useRelay,
    requestId: `${Date.now()}-gamma-events-series-${args.seriesId}`,
  }).then((data) => {
    return Array.isArray(data) ? (data as GammaEventRecord[]) : [];
  });

  gammaEventsBySeriesCache.set(url, promise);
  return promise;
}

function normalizeToken(s: string): string {
  return s
    .normalize('NFKD')
    .replace(/[^\w\s-]/g, '')
    .toLowerCase()
    .trim();
}

function pickLastNameToken(fullName: string): string {
  const cleaned = normalizeToken(fullName);
  const parts = cleaned.split(/\s+/).filter(Boolean);
  return parts[parts.length - 1] ?? cleaned;
}

async function fetchGammaEventsBySlug(args: {
  slug: string;
  useRelay: boolean;
  requestIdSuffix: string;
}): Promise<GammaEventRecord[]> {
  const url = `https://gamma-api.polymarket.com/events?slug=${encodeURIComponent(args.slug)}`;
  const data = await fetchGammaJson({
    url,
    useRelay: args.useRelay,
    requestId: `${Date.now()}-${args.requestIdSuffix}`,
  });
  return Array.isArray(data) ? (data as GammaEventRecord[]) : [];
}

/**
 * Resolve a Polymarket game slug by searching events for participant names.
 * Works for tennis and CBB where slugs can't be deterministically constructed.
 */
export async function resolvePolyGameSlugByParticipants(args: {
  sport: Sport;
  dateYyyyMmDd: string;
  awayName?: string;
  homeName?: string;
  useRelay: boolean;
}): Promise<string | null> {
  const seriesId = gammaSeriesIdForSport(args.sport);
  if (!seriesId) return null;

  const awayLast = args.awayName ? pickLastNameToken(args.awayName) : '';
  const homeLast = args.homeName ? pickLastNameToken(args.homeName) : '';
  if (!awayLast || !homeLast) return null;

  const candidates = await fetchGammaEventsBySeries({
    seriesId,
    useRelay: args.useRelay,
  });
  if (candidates.length === 0) return null;
  const byDate = candidates.filter((e) => {
    const slug = String(e.slug ?? '').toLowerCase();
    if (slug.endsWith(`-${args.dateYyyyMmDd}`)) return true;
    const start = String(e.startTime ?? '');
    return start.startsWith(args.dateYyyyMmDd);
  });
  const pool = byDate.length > 0 ? byDate : candidates;

  let bestSlug: string | null = null;
  let bestScore = -1;
  for (const evt of pool) {
    const slug = String(evt.slug ?? '');
    if (!slug) continue;
    const hay = `${normalizeToken(slug)} ${normalizeToken(String(evt.title ?? ''))}`;
    let score = 0;
    if (hay.includes(awayLast)) score += 2;
    if (hay.includes(homeLast)) score += 2;
    if (slug.toLowerCase().endsWith(`-${args.dateYyyyMmDd}`)) score += 1;
    if (score > bestScore) {
      bestScore = score;
      bestSlug = slug;
    }
  }
  return bestScore >= 4 ? bestSlug : null;
}

/** @deprecated Use resolvePolyGameSlugByParticipants instead */
export const resolvePolyTennisGameSlug = resolvePolyGameSlugByParticipants;

function parseOutcomePrices(raw: unknown): number[] {
  const toPrices = (arr: unknown[]): number[] =>
    arr
      .map((v) => Number(v))
      .filter((n) => Number.isFinite(n) && n > 0 && n < 1)
      .map((n) => Math.round(n * 100));

  if (Array.isArray(raw)) return toPrices(raw);
  if (typeof raw === 'string') {
    try {
      const parsed = JSON.parse(raw);
      if (Array.isArray(parsed)) return toPrices(parsed);
    } catch {
      return [];
    }
  }
  return [];
}

function parseSpreadMetaFromSlug(
  slug: string
): { favoriteSide: 'away' | 'home'; spreadValue: number } | null {
  const m = slug.match(/-(?:spread|set-handicap)-(away|home)-(\d+)(?:pt(\d+))?$/i);
  if (!m || !m[1] || !m[2]) return null;
  const whole = Number(m[2]);
  const fracRaw = m[3];
  if (!Number.isFinite(whole)) return null;
  const frac = fracRaw ? Number(`0.${fracRaw}`) : 0;
  const spreadValue = whole + frac;
  if (!Number.isFinite(spreadValue)) return null;
  return {
    favoriteSide: m[1].toLowerCase() === 'away' ? 'away' : 'home',
    spreadValue,
  };
}

export async function resolvePolySpreadMarketsByGameSlug(args: {
  slug: string;
  useRelay: boolean;
}): Promise<PolySpreadMarketInfo[]> {
  const events = await fetchGammaEventsBySlug({
    slug: args.slug,
    useRelay: args.useRelay,
    requestIdSuffix: 'gamma-event-spread',
  });
  if (events.length === 0) return [];
  const evt = events[0] as GammaEventRecord;
  const markets = Array.isArray(evt.markets) ? evt.markets : [];
  const out: PolySpreadMarketInfo[] = [];

  for (const m of markets) {
    const marketType = String(m?.sportsMarketType ?? '').toLowerCase();
    if (
      marketType !== 'spreads' &&
      marketType !== 'spread' &&
      marketType !== 'tennis_set_handicap'
    ) {
      continue;
    }

    const slug = String(m?.slug ?? '');
    const meta = parseSpreadMetaFromSlug(slug);
    if (!slug || !meta) continue;

    const tokenIds = parseClobTokenIds(m?.clobTokenIds);
    if (tokenIds.length < 2) continue;
    const prices = parseOutcomePrices(m?.outcomePrices);
    const liquidityRaw = Number(m?.liquidity);

    out.push({
      slug,
      favoriteSide: meta.favoriteSide,
      spreadValue: meta.spreadValue,
      favoriteTokenId: String(tokenIds[0]),
      underdogTokenId: String(tokenIds[1]),
      favoritePriceCents: prices[0],
      underdogPriceCents: prices[1],
      liquidityUsd: Number.isFinite(liquidityRaw) ? liquidityRaw : undefined,
      conditionId: m?.conditionId ? String(m.conditionId) : undefined,
      tickSize: parseTickSize(m?.orderPriceMinTickSize),
      negRisk: parseNegRisk(m),
    });
  }

  return out;
}

function parseTotalMetaFromSlug(slug: string): { totalLine: number } | null {
  const m = slug.match(/-(?:total|match-total)-(\d+)(?:pt(\d+))?$/i);
  if (!m || !m[1]) return null;
  const whole = Number(m[1]);
  const fracRaw = m[2];
  if (!Number.isFinite(whole)) return null;
  const frac = fracRaw ? Number(`0.${fracRaw}`) : 0;
  const totalLine = whole + frac;
  if (!Number.isFinite(totalLine)) return null;
  return { totalLine };
}

export async function resolvePolyTotalMarketsByGameSlug(args: {
  slug: string;
  useRelay: boolean;
}): Promise<PolyTotalMarketInfo[]> {
  const events = await fetchGammaEventsBySlug({
    slug: args.slug,
    useRelay: args.useRelay,
    requestIdSuffix: 'gamma-event-total',
  });
  if (events.length === 0) return [];
  const evt = events[0] as GammaEventRecord;
  const markets = Array.isArray(evt.markets) ? evt.markets : [];
  const out: PolyTotalMarketInfo[] = [];

  for (const m of markets) {
    const marketType = String(m?.sportsMarketType ?? '').toLowerCase();
    if (marketType !== 'totals' && marketType !== 'total' && marketType !== 'tennis_match_totals') {
      continue;
    }

    const slug = String(m?.slug ?? '');
    const meta = parseTotalMetaFromSlug(slug);
    if (!slug || !meta) continue;

    const tokenIds = parseClobTokenIds(m?.clobTokenIds);
    if (tokenIds.length < 2) continue;
    const prices = parseOutcomePrices(m?.outcomePrices);
    const liquidityRaw = Number(m?.liquidity);

    out.push({
      slug,
      totalLine: meta.totalLine,
      overTokenId: String(tokenIds[0]),
      underTokenId: String(tokenIds[1]),
      overPriceCents: prices[0],
      underPriceCents: prices[1],
      liquidityUsd: Number.isFinite(liquidityRaw) ? liquidityRaw : undefined,
      conditionId: m?.conditionId ? String(m.conditionId) : undefined,
      tickSize: parseTickSize(m?.orderPriceMinTickSize),
      negRisk: parseNegRisk(m),
    });
  }

  return out;
}
