/**
 * Position Tracker
 *
 * Tracks positions from orders and fills, calculates P&L
 */

import type { Position, FillNotification } from '../types';

export class PositionTracker {
  private positions: Map<string, Position> = new Map();
  private fills: FillNotification[] = [];

  /**
   * Add a fill and update position
   */
  addFill(fill: FillNotification, currentPrice: number | null = null): void {
    this.fills.push(fill);

    const key = `${fill.ticker}-${fill.side}`;
    const existing = this.positions.get(key);

    if (existing) {
      // Update existing position
      const totalCost = existing.averagePrice * Math.abs(existing.quantity);
      const fillCost = fill.price * fill.quantity;

      let newQuantity: number;
      if (fill.action === 'buy') {
        newQuantity = existing.quantity + fill.quantity;
      } else {
        newQuantity = existing.quantity - fill.quantity;
      }

      // Calculate new average price
      let newAvgPrice: number;
      if (newQuantity === 0) {
        newAvgPrice = 0;
      } else if (
        (existing.quantity > 0 && fill.action === 'buy') ||
        (existing.quantity < 0 && fill.action === 'sell')
      ) {
        // Adding to position
        newAvgPrice = (totalCost + fillCost) / Math.abs(newQuantity);
      } else {
        // Reducing position - average price stays the same
        newAvgPrice = existing.averagePrice;
      }

      const position: Position = {
        ticker: fill.ticker,
        side: fill.side,
        quantity: newQuantity,
        averagePrice: newAvgPrice,
        currentPrice: currentPrice,
        realizedPnl: existing.realizedPnl, // TODO: Calculate realized P&L on close
        unrealizedPnl: 0,
        totalPnl: 0,
        pnlPercent: 0,
      };

      this.updatePnl(position);
      this.positions.set(key, position);

      // Remove if position is closed
      if (newQuantity === 0) {
        this.positions.delete(key);
      }
    } else {
      // New position
      const quantity = fill.action === 'buy' ? fill.quantity : -fill.quantity;
      const position: Position = {
        ticker: fill.ticker,
        side: fill.side,
        quantity,
        averagePrice: fill.price,
        currentPrice: currentPrice,
        realizedPnl: 0,
        unrealizedPnl: 0,
        totalPnl: 0,
        pnlPercent: 0,
      };

      this.updatePnl(position);
      this.positions.set(key, position);
    }
  }

  /**
   * Update P&L for a position
   */
  updatePnl(position: Position, currentPrice?: number | null): void {
    if (currentPrice !== undefined) {
      position.currentPrice = currentPrice;
    }

    if (position.currentPrice === null || position.quantity === 0) {
      position.unrealizedPnl = 0;
      position.totalPnl = position.realizedPnl;
      position.pnlPercent = 0;
      return;
    }

    // Calculate unrealized P&L
    // IMPORTANT: We treat prices as the contract price for the position side:
    // - YES position uses YES contract price
    // - NO position uses NO contract price (1 - YES price)
    // With that convention, P&L is the same formula for both sides.
    const priceDiff = position.currentPrice - position.averagePrice;
    const direction = position.quantity > 0 ? 1 : -1;

    position.unrealizedPnl = priceDiff * direction * Math.abs(position.quantity);
    position.totalPnl = position.realizedPnl + position.unrealizedPnl;

    if (position.averagePrice > 0) {
      position.pnlPercent =
        (position.unrealizedPnl / (position.averagePrice * Math.abs(position.quantity))) * 100;
    } else {
      position.pnlPercent = 0;
    }
  }

  /**
   * Update all positions with current prices
   */
  updatePrices(yesPriceMap: Map<string, number>): void {
    for (const [_key, position] of this.positions.entries()) {
      const yesPrice = yesPriceMap.get(position.ticker);
      if (yesPrice !== undefined) {
        const contractPrice = position.side === 'yes' ? yesPrice : 1 - yesPrice;
        this.updatePnl(position, contractPrice);
      }
    }
  }

  /**
   * Get all positions
   */
  getPositions(): Position[] {
    return Array.from(this.positions.values());
  }

  /**
   * Get position for a ticker and side
   */
  getPosition(ticker: string, side: 'yes' | 'no'): Position | null {
    const key = `${ticker}-${side}`;
    return this.positions.get(key) || null;
  }

  /**
   * Get total P&L across all positions
   */
  getTotalPnl(): number {
    let total = 0;
    for (const position of this.positions.values()) {
      total += position.totalPnl;
    }
    return total;
  }

  /**
   * Clear all positions
   */
  clear(): void {
    this.positions.clear();
    this.fills = [];
  }

  /**
   * Set position directly (for loading from API)
   */
  setPosition(position: Position): void {
    const key = `${position.ticker}-${position.side}`;
    this.positions.set(key, position);
  }

  /**
   * Get all fills
   */
  getFills(): FillNotification[] {
    return [...this.fills];
  }
}
