/**
 * HTTP Relay Client
 *
 * Sends requests through the relay server instead of directly to Kalshi.
 * This solves CORS and provides a consistent transport layer.
 */

import type { HttpRelayRequest, HttpRelayResponse } from '@galactus/shared';

const envRelayUrl = (import.meta.env.VITE_RELAY_HTTP_URL as string | undefined)?.trim();
const RELAY_URL =
  envRelayUrl && envRelayUrl.length > 0
    ? envRelayUrl
    : import.meta.env.DEV
      ? 'http://localhost:8787/relay/http'
      : '/relay/http';

/**
 * Forward HTTP request through relay
 */
export async function relayHttp(envelope: HttpRelayRequest): Promise<HttpRelayResponse> {
  // Add timeout to prevent hanging if relay server is down
  const controller = new AbortController();
  const timeoutId = setTimeout(() => controller.abort(), 10000); // 10 second timeout

  try {
    const response = await fetch(RELAY_URL, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
      },
      body: JSON.stringify(envelope),
      signal: controller.signal,
    });

    clearTimeout(timeoutId);

    if (!response.ok) {
      const error = await response.text().catch(() => 'Unknown error');
      throw new Error(`Relay error: ${response.status} ${error}`);
    }

    const result = (await response.json()) as HttpRelayResponse;
    return result;
  } catch (error) {
    clearTimeout(timeoutId);
    if (error instanceof Error && error.name === 'AbortError') {
      throw new Error('Relay error: Connection timeout. Is the relay server running on port 8787?');
    }
    if (error instanceof TypeError && error.message.includes('Failed to fetch')) {
      throw new Error(
        "Relay error: Cannot connect to relay server. Make sure it's running on localhost:8787"
      );
    }
    throw error;
  }
}

/**
 * Create a relay request envelope
 */
export function createRelayRequest(
  id: string,
  method: 'GET' | 'POST' | 'PUT' | 'DELETE' | 'PATCH',
  url: string,
  headers: Record<string, string>,
  body?: string
): HttpRelayRequest {
  return {
    id,
    method,
    url,
    headers,
    body,
  };
}
