/**
 * WebSocket Relay Client
 *
 * Connects to relay WebSocket, then forwards messages to upstream Kalshi WS.
 */

import type { WsRelayMessage, WsRelayFrame } from '@galactus/shared';

const envRelayWsUrl = (import.meta.env.VITE_RELAY_WS_URL as string | undefined)?.trim();
const RELAY_WS_URL =
  envRelayWsUrl && envRelayWsUrl.length > 0
    ? envRelayWsUrl
    : import.meta.env.DEV
      ? 'ws://localhost:8787/relay/ws'
      : `${window.location.protocol === 'https:' ? 'wss:' : 'ws:'}//${window.location.host}/relay/ws`;
export interface RelayWsClient {
  connect(streamId: string, upstreamUrl: string, headers: Record<string, string>): Promise<void>;
  send(streamId: string, payload: unknown): void;
  subscribe(streamId: string, payload?: unknown): void;
  close(streamId: string): void;
  onMessage(callback: (frame: WsRelayFrame) => void): void;
  onError(callback: (error: Error) => void): void;
  disconnect(): void;
}

/**
 * Create WebSocket relay client
 */
export function createRelayWs(): RelayWsClient {
  let ws: WebSocket | null = null;
  const messageCallbacks: Set<(frame: WsRelayFrame) => void> = new Set();
  const errorCallbacks: Set<(error: Error) => void> = new Set();

  const connect = (): Promise<void> => {
    return new Promise((resolve, reject) => {
      if (ws && ws.readyState === WebSocket.OPEN) {
        resolve();
        return;
      }

      ws = new WebSocket(RELAY_WS_URL);

      const timeout = setTimeout(() => {
        reject(new Error('WebSocket connection timeout'));
      }, 10000);

      ws.onopen = () => {
        clearTimeout(timeout);
        resolve();
      };

      ws.onerror = (error) => {
        clearTimeout(timeout);
        errorCallbacks.forEach((cb) => cb(new Error('WebSocket connection failed')));
        reject(error);
      };

      ws.onmessage = (event) => {
        try {
          const frame = JSON.parse(event.data) as WsRelayFrame;
          messageCallbacks.forEach((cb) => cb(frame));
        } catch (error) {
          console.error('Failed to parse relay frame:', error);
        }
      };

      ws.onclose = () => {
        ws = null;
      };
    });
  };

  const sendMessage = (message: WsRelayMessage) => {
    if (!ws || ws.readyState !== WebSocket.OPEN) {
      throw new Error('WebSocket not connected');
    }
    ws.send(JSON.stringify(message));
  };

  return {
    async connect(streamId: string, upstreamUrl: string, headers: Record<string, string>) {
      await connect();
      sendMessage({
        op: 'connect',
        id: streamId,
        url: upstreamUrl,
        headers,
      });
    },

    send(streamId: string, payload: unknown) {
      sendMessage({
        op: 'send',
        id: streamId,
        payload,
      });
    },

    subscribe(streamId: string, payload?: unknown) {
      sendMessage({
        op: 'subscribe',
        id: streamId,
        payload,
      });
    },

    close(streamId: string) {
      // Guard: no-op if WebSocket not connected (avoids crash in cleanup functions)
      if (!ws || ws.readyState !== WebSocket.OPEN) {
        return;
      }
      sendMessage({
        op: 'close',
        id: streamId,
      });
    },

    onMessage(callback: (frame: WsRelayFrame) => void) {
      messageCallbacks.add(callback);
    },

    onError(callback: (error: Error) => void) {
      errorCallbacks.add(callback);
    },

    disconnect() {
      if (ws) {
        ws.close();
        ws = null;
      }
      messageCallbacks.clear();
      errorCallbacks.clear();
    },
  };
}
