/**
 * SmartRelayStream — WebSocket client for the relay's /stream/markets endpoint.
 *
 * Speaks the StreamSubscribe/StreamSnapshot/StreamUpdate protocol.
 * Auto-reconnects with exponential backoff.
 */

import type {
  StreamClientMessage,
  StreamServerMessage,
  StreamSnapshot,
  StreamUpdate,
  StreamStatus,
  CachedMarket,
  CachedOrderbook,
} from '@galactus/shared';

export interface SmartRelayStreamCallbacks {
  onMarketSnapshot?: (markets: CachedMarket[]) => void;
  onOrderbookSnapshot?: (orderbooks: CachedOrderbook[]) => void;
  onMarketUpdate?: (market: CachedMarket) => void;
  onOrderbookUpdate?: (orderbook: CachedOrderbook) => void;
  onStatus?: (status: StreamStatus) => void;
  onError?: (error: Error) => void;
  onConnectionChange?: (connected: boolean) => void;
}

export interface SmartRelayStream {
  connect(): void;
  disconnect(): void;
  subscribe(channel: 'markets' | 'orderbooks', tickers?: string[]): void;
  unsubscribe(channel: 'markets' | 'orderbooks', tickers?: string[]): void;
  isConnected(): boolean;
}

export function createSmartRelayStream(callbacks: SmartRelayStreamCallbacks): SmartRelayStream {
  let ws: WebSocket | null = null;
  let connected = false;
  let reconnectAttempts = 0;
  let reconnectTimer: ReturnType<typeof setTimeout> | null = null;
  let stopped = false;

  // Determine WS URL from environment or default
  const getWsUrl = (): string => {
    let base: string;
    const override = import.meta.env.VITE_RELAY_WS_URL;
    if (override) {
      // Replace the path portion with /stream/markets
      try {
        const url = new URL(override);
        url.pathname = '/stream/markets';
        base = url.toString();
      } catch {
        base = override;
      }
    } else {
      // Default: same host, ws protocol
      const proto = location.protocol === 'https:' ? 'wss:' : 'ws:';
      base = `${proto}//${location.host}/stream/markets`;
    }

    // Append stream token if configured
    const token = import.meta.env.VITE_STREAM_TOKEN;
    if (token) {
      const sep = base.includes('?') ? '&' : '?';
      base = `${base}${sep}token=${encodeURIComponent(token)}`;
    }

    return base;
  };

  const send = (msg: StreamClientMessage) => {
    if (ws && ws.readyState === WebSocket.OPEN) {
      ws.send(JSON.stringify(msg));
    }
  };

  const handleMessage = (data: string) => {
    try {
      const msg = JSON.parse(data) as StreamServerMessage;

      switch (msg.type) {
        case 'snapshot': {
          const snap = msg as StreamSnapshot;
          if (snap.channel === 'markets' && snap.markets) {
            callbacks.onMarketSnapshot?.(snap.markets);
          }
          if (snap.channel === 'orderbooks' && snap.orderbooks) {
            callbacks.onOrderbookSnapshot?.(snap.orderbooks);
          }
          break;
        }
        case 'update': {
          const upd = msg as StreamUpdate;
          if (upd.channel === 'markets' && upd.market) {
            callbacks.onMarketUpdate?.(upd.market);
          }
          if (upd.channel === 'orderbooks' && upd.orderbook) {
            callbacks.onOrderbookUpdate?.(upd.orderbook);
          }
          break;
        }
        case 'status':
          callbacks.onStatus?.(msg as StreamStatus);
          break;
        case 'error':
          callbacks.onError?.(new Error(msg.message));
          break;
      }
    } catch (_err) {
      callbacks.onError?.(new Error('Failed to parse smart relay message'));
    }
  };

  const doConnect = () => {
    if (stopped) return;

    try {
      const url = getWsUrl();
      ws = new WebSocket(url);

      ws.onopen = () => {
        connected = true;
        reconnectAttempts = 0;
        callbacks.onConnectionChange?.(true);
      };

      ws.onmessage = (event) => {
        handleMessage(typeof event.data === 'string' ? event.data : String(event.data));
      };

      ws.onclose = () => {
        connected = false;
        callbacks.onConnectionChange?.(false);
        scheduleReconnect();
      };

      ws.onerror = () => {
        callbacks.onError?.(new Error('Smart relay WebSocket error'));
      };
    } catch (_err) {
      callbacks.onError?.(new Error('Smart relay connect failed'));
      scheduleReconnect();
    }
  };

  const scheduleReconnect = () => {
    if (stopped) return;
    if (reconnectAttempts >= 20) return;

    const delay = Math.min(1000 * Math.pow(2, reconnectAttempts), 30000);
    reconnectAttempts++;
    reconnectTimer = setTimeout(doConnect, delay);
  };

  return {
    connect() {
      stopped = false;
      doConnect();
    },

    disconnect() {
      stopped = true;
      if (reconnectTimer) {
        clearTimeout(reconnectTimer);
        reconnectTimer = null;
      }
      if (ws) {
        ws.close();
        ws = null;
      }
      connected = false;
    },

    subscribe(channel, tickers) {
      send({ op: 'subscribe', channel, tickers });
    },

    unsubscribe(channel, tickers) {
      send({ op: 'unsubscribe', channel, tickers });
    },

    isConnected() {
      return connected;
    },
  };
}
