import { describe, expect, it } from 'vitest';
import { parseEventTicker, parseMarketTicker, buildEventTicker } from './tickerParser';

describe('parseEventTicker', () => {
  it('parses NBA event ticker', () => {
    const result = parseEventTicker('KXNBAGAME-26FEB04BOSMIA');
    expect(result).toMatchObject({
      eventTicker: 'KXNBAGAME-26FEB04BOSMIA',
      seriesTicker: 'KXNBAGAME',
      sport: 'nba',
      marketType: 'moneyline',
      dateYyyyMmDd: '2026-02-04',
      awayCode: 'BOS',
      homeCode: 'MIA',
    });
  });

  it('parses NHL event ticker', () => {
    const result = parseEventTicker('KXNHLGAME-26FEB04EDMCGY');
    expect(result).toMatchObject({
      sport: 'nhl',
      marketType: 'moneyline',
      dateYyyyMmDd: '2026-02-04',
      awayCode: 'EDM',
      homeCode: 'CGY',
    });
  });

  it('parses CBB event ticker with 3+3 codes', () => {
    const result = parseEventTicker('KXNCAAMBGAME-26FEB04DUKNCA');
    expect(result).toMatchObject({
      sport: 'cbb',
      marketType: 'moneyline',
      awayCode: 'DUK',
      homeCode: 'NCA',
    });
  });

  it('parses CBB event ticker with 4-char away code', () => {
    const result = parseEventTicker('KXNCAAMBGAME-26FEB04UTAHBYU');
    expect(result).toMatchObject({
      sport: 'cbb',
      awayCode: 'UTAH',
      homeCode: 'BYU',
    });
  });

  it('parses Tennis ATP event ticker', () => {
    const result = parseEventTicker('KXATPMATCH-26FEB03VAVBLA');
    expect(result).toMatchObject({
      sport: 'tennis-atp',
      marketType: 'moneyline',
      dateYyyyMmDd: '2026-02-03',
      awayCode: 'VAV',
      homeCode: 'BLA',
      player1Code: 'VAV',
      player2Code: 'BLA',
    });
  });

  it('parses NBA spread event ticker', () => {
    const result = parseEventTicker('KXNBASPREAD-26FEB03NYKWAS');
    expect(result).toMatchObject({
      sport: 'nba',
      marketType: 'spread',
      awayCode: 'NYK',
      homeCode: 'WAS',
    });
  });

  it('parses NBA total event ticker', () => {
    const result = parseEventTicker('KXNBATOTAL-26FEB03PHXPOR');
    expect(result).toMatchObject({
      sport: 'nba',
      marketType: 'total',
      awayCode: 'PHX',
      homeCode: 'POR',
    });
  });

  it('returns null for invalid/empty input', () => {
    expect(parseEventTicker('')).toBeNull();
    expect(parseEventTicker('GARBAGE')).toBeNull();
    expect(parseEventTicker('KXNBAGAME-')).toBeNull();
  });

  it('returns null for unknown series ticker', () => {
    expect(parseEventTicker('KXFOO-26FEB04ABCDEF')).toBeNull();
  });
});

describe('parseMarketTicker', () => {
  it('parses NBA moneyline market ticker', () => {
    const result = parseMarketTicker('KXNBAGAME-26FEB04BOSMIA-BOS');
    expect(result).toMatchObject({
      eventTicker: 'KXNBAGAME-26FEB04BOSMIA',
      marketTicker: 'KXNBAGAME-26FEB04BOSMIA-BOS',
      sport: 'nba',
      marketType: 'moneyline',
      outcomeCode: 'BOS',
    });
  });

  it('parses NBA spread market ticker with bucket', () => {
    const result = parseMarketTicker('KXNBASPREAD-26FEB03NYKWAS-NYK7');
    expect(result).toMatchObject({
      sport: 'nba',
      marketType: 'spread',
      outcomeCode: 'NYK',
      spreadBucket: 7,
    });
  });

  it('parses NBA total market ticker with strike', () => {
    const result = parseMarketTicker('KXNBATOTAL-26FEB03PHXPOR-237');
    expect(result).toMatchObject({
      sport: 'nba',
      marketType: 'total',
      outcomeCode: 'O237',
      totalStrike: 237,
    });
  });

  it('parses double-dash format', () => {
    const result = parseMarketTicker('KXNBAGAME-26JAN23BOSBKN--BOS');
    expect(result).toMatchObject({
      eventTicker: 'KXNBAGAME-26JAN23BOSBKN',
      outcomeCode: 'BOS',
    });
  });

  it('returns null for invalid input', () => {
    expect(parseMarketTicker('')).toBeNull();
    expect(parseMarketTicker('NOTENOUGH')).toBeNull();
  });
});

describe('buildEventTicker', () => {
  it('builds NBA event ticker', () => {
    expect(buildEventTicker('KXNBAGAME', '2026-02-04', 'BOS', 'MIA')).toBe(
      'KXNBAGAME-26FEB04BOSMIA'
    );
  });

  it('builds NHL event ticker', () => {
    expect(buildEventTicker('KXNHLGAME', '2026-01-15', 'EDM', 'CGY')).toBe(
      'KXNHLGAME-26JAN15EDMCGY'
    );
  });

  it('round-trips through parseEventTicker', () => {
    const ticker = buildEventTicker('KXNBAGAME', '2026-03-10', 'LAL', 'GSW');
    const parsed = parseEventTicker(ticker);
    expect(parsed).not.toBeNull();
    expect(parsed!.dateYyyyMmDd).toBe('2026-03-10');
    expect(parsed!.awayCode).toBe('LAL');
    expect(parsed!.homeCode).toBe('GSW');
  });
});
