/**
 * Sports Ticker Parser
 *
 * Parses Kalshi event and market tickers for all supported sports.
 * Handles moneylines, spreads, and totals.
 */

import { MONTH_MAP } from '../kalshiTicker/types';
import {
  type Sport,
  type ParsedEventInfo,
  type ParsedMarketInfo,
  getConfigFromSeriesTicker,
} from './types';

/**
 * Parse date from YYMONDD format
 */
function parseDateToken(dateToken: string): string | null {
  if (dateToken.length < 7) return null;

  const yy = Number(dateToken.slice(0, 2));
  const monStr = dateToken.slice(2, 5).toUpperCase();
  const dd = Number(dateToken.slice(5, 7));
  const mm = MONTH_MAP[monStr];

  if (!Number.isFinite(yy) || !Number.isFinite(dd) || !mm) return null;

  const year = 2000 + yy;
  const month = String(mm).padStart(2, '0');
  const day = String(dd).padStart(2, '0');

  return `${year}-${month}-${day}`;
}

/**
 * Extract series ticker from event/market ticker
 */
function extractSeriesTicker(ticker: string): string | null {
  const upper = ticker.toUpperCase().trim();
  const dashIdx = upper.indexOf('-');
  if (dashIdx <= 0) return null;
  return upper.slice(0, dashIdx);
}

/**
 * NFL team codes that are 2 characters (all others are 3)
 */
const NFL_2CHAR_CODES = new Set(['GB', 'KC', 'LA', 'LV', 'NE', 'NO', 'SF', 'TB']);

/**
 * MLB team codes that are 2 characters (all others are 3)
 */
const MLB_2CHAR_CODES = new Set(['KC', 'SD', 'SF', 'TB']);

/**
 * CBB team codes that are 4 characters.
 * Used to disambiguate 7-char matchup strings (4+3 vs 3+4).
 */
const CBB_4CHAR_CODES = new Set([
  'UTAH',
  'UCLA',
  'SDSU',
  'UNLV',
  'UCSB',
  'ETSU',
  'UMBC',
  'UMKC',
  'UTEP',
  'UTSA',
  'MVSU',
  'SIUE',
  'FGCU',
  'NJIT',
  'SEMO',
  'SELU',
  'UAPB',
  'UALR',
  'MSST',
  'SHSU',
  'NMSU',
  'FAMU',
  'PVAM',
  'TXSO',
  'CCSU',
  'NCCU',
  'SCST',
  'AAMU',
  'ALST',
  'GRAM',
  'UNCG',
  'UNCW',
  'UNCA',
  'TAMC',
  'IUPU',
]);

/**
 * Parse team codes from the matchup portion of an event ticker.
 * Handles different code lengths by sport:
 *   - NBA/NHL: strictly 3+3 = 6 chars
 *   - NFL: variable 2-3 chars per team (4-6 total)
 *   - CBB: variable 3-4 chars per team (6-8 total)
 *   - Tennis: 3+3 = 6 chars
 *
 * For fixed codes: "BOSMIA" -> { away: "BOS", home: "MIA" }
 * For NFL variable: "SEANE" -> { away: "SEA", home: "NE" }
 * For CBB variable: "WSUORST" -> { away: "WSU", home: "ORST" }
 */
function parseTeamCodes(
  matchupStr: string,
  sport: Sport
): { awayCode: string; homeCode: string } | null {
  const upper = matchupStr.toUpperCase();
  const len = upper.length;

  // NBA, NHL, Tennis, UFC: strictly 3+3
  if (
    sport === 'nba' ||
    sport === 'nhl' ||
    sport === 'tennis-atp' ||
    sport === 'tennis-wta' ||
    sport === 'ufc'
  ) {
    if (len < 6) return null;
    return {
      awayCode: upper.slice(0, 3),
      homeCode: upper.slice(3, 6),
    };
  }

  // MLB: variable 2-3 chars per team (same logic as NFL)
  if (sport === 'mlb') {
    if (len === 6) {
      return { awayCode: upper.slice(0, 3), homeCode: upper.slice(3) };
    }
    if (len === 5) {
      const away3 = upper.slice(0, 3);
      const home2 = upper.slice(3);
      if (MLB_2CHAR_CODES.has(home2)) {
        return { awayCode: away3, homeCode: home2 };
      }
      const away2 = upper.slice(0, 2);
      const home3 = upper.slice(2);
      if (MLB_2CHAR_CODES.has(away2)) {
        return { awayCode: away2, homeCode: home3 };
      }
    }
    if (len === 4) {
      const away2 = upper.slice(0, 2);
      const home2 = upper.slice(2);
      if (MLB_2CHAR_CODES.has(away2) && MLB_2CHAR_CODES.has(home2)) {
        return { awayCode: away2, homeCode: home2 };
      }
    }
    if (len >= 6) {
      return { awayCode: upper.slice(0, 3), homeCode: upper.slice(3) };
    }
    return null;
  }

  // NFL: variable 2-3 chars per team
  // Try to match known 2-char codes
  if (sport === 'nfl') {
    // Try 3+3 first
    if (len === 6) {
      return { awayCode: upper.slice(0, 3), homeCode: upper.slice(3) };
    }
    // Try 3+2 (away=3, home=2)
    if (len === 5) {
      const away3 = upper.slice(0, 3);
      const home2 = upper.slice(3);
      if (NFL_2CHAR_CODES.has(home2)) {
        return { awayCode: away3, homeCode: home2 };
      }
    }
    // Try 2+3 (away=2, home=3)
    if (len === 5) {
      const away2 = upper.slice(0, 2);
      const home3 = upper.slice(2);
      if (NFL_2CHAR_CODES.has(away2)) {
        return { awayCode: away2, homeCode: home3 };
      }
    }
    // Try 2+2
    if (len === 4) {
      const away2 = upper.slice(0, 2);
      const home2 = upper.slice(2);
      if (NFL_2CHAR_CODES.has(away2) && NFL_2CHAR_CODES.has(home2)) {
        return { awayCode: away2, homeCode: home2 };
      }
    }
    // Fallback for 6+ chars: assume 3+rest
    if (len >= 6) {
      return { awayCode: upper.slice(0, 3), homeCode: upper.slice(3) };
    }
    return null;
  }

  // CBB: variable length team codes (2-5 chars each)
  // Try common patterns: 3+3, 3+4, 4+3, 4+4
  if (len === 6) {
    // 3+3
    return { awayCode: upper.slice(0, 3), homeCode: upper.slice(3) };
  }
  if (len === 7) {
    // Could be 3+4 or 4+3 — check known 4-char codes to disambiguate
    const first4 = upper.slice(0, 4);
    if (CBB_4CHAR_CODES.has(first4)) {
      return { awayCode: first4, homeCode: upper.slice(4) }; // 4+3
    }
    return { awayCode: upper.slice(0, 3), homeCode: upper.slice(3) }; // 3+4
  }
  if (len === 8) {
    // Could be 4+4, 3+5, or 5+3
    const first4 = upper.slice(0, 4);
    const last4 = upper.slice(4);
    if (CBB_4CHAR_CODES.has(first4) || CBB_4CHAR_CODES.has(last4)) {
      return { awayCode: first4, homeCode: last4 }; // 4+4
    }
    return { awayCode: first4, homeCode: last4 }; // default 4+4
  }

  // Fallback: split in half
  const mid = Math.floor(len / 2);
  return { awayCode: upper.slice(0, mid), homeCode: upper.slice(mid) };
}

/**
 * Parse a sports event ticker
 *
 * Format: {SERIES}-{YYMONDD}{AWAY}{HOME}
 * Examples:
 *   KXNBAGAME-26FEB04BOSMIA
 *   KXNHLGAME-26FEB04EDMCGY
 *   KXNCAAMBGAME-26FEB04WSUORST
 *   KXATPMATCH-26FEB03VAVBLA
 */
export function parseEventTicker(eventTickerRaw: string): ParsedEventInfo | null {
  const eventTicker = eventTickerRaw.trim().toUpperCase();
  const seriesTicker = extractSeriesTicker(eventTicker);
  if (!seriesTicker) return null;

  const config = getConfigFromSeriesTicker(seriesTicker);
  if (!config) return null;

  const { sport, marketType } = config;

  // Extract the portion after the series ticker
  const rest = eventTicker.slice(seriesTicker.length + 1); // +1 for the dash
  if (rest.length < 7) return null;

  // Parse date (first 7 chars: YYMONDD)
  const dateYyyyMmDd = parseDateToken(rest.slice(0, 7));
  if (!dateYyyyMmDd) return null;

  // Parse team/player codes (remaining chars)
  const matchupStr = rest.slice(7);

  // Tennis/UFC have player/fighter codes instead of team codes
  const isIndividualSport = sport === 'tennis-atp' || sport === 'tennis-wta' || sport === 'ufc';
  if (isIndividualSport) {
    // Individual sports: codes are 3 chars each
    if (matchupStr.length < 6) return null;
    return {
      eventTicker,
      seriesTicker,
      sport,
      marketType,
      dateYyyyMmDd,
      awayCode: matchupStr.slice(0, 3),
      homeCode: matchupStr.slice(3, 6),
      player1Code: matchupStr.slice(0, 3),
      player2Code: matchupStr.slice(3, 6),
    };
  }

  // Team sports
  const teams = parseTeamCodes(matchupStr, sport);
  if (!teams) return null;

  return {
    eventTicker,
    seriesTicker,
    sport,
    marketType,
    dateYyyyMmDd,
    awayCode: teams.awayCode,
    homeCode: teams.homeCode,
  };
}

/**
 * Parse a sports market ticker
 *
 * Formats vary by market type:
 *   Moneyline: {SERIES}-{YYMONDD}{AWAY}{HOME}-{TEAM}
 *   Spread:    {SERIES}-{YYMONDD}{AWAY}{HOME}-{TEAM}{BUCKET}
 *   Total:     {SERIES}-{YYMONDD}{AWAY}{HOME}-{STRIKE}
 *
 * Examples:
 *   KXNBAGAME-26FEB04BOSMIA-BOS
 *   KXNBASPREAD-26FEB03NYKWAS-NYK7
 *   KXNBATOTAL-26FEB03PHXPOR-237
 *   KXATPMATCH-26FEB03VAVBLA-VAV
 */
export function parseMarketTicker(marketTickerRaw: string): ParsedMarketInfo | null {
  const marketTicker = marketTickerRaw.trim().toUpperCase();

  // Handle double-dash format (e.g., KXNBAGAME-26JAN23BOSBKN--BOS)
  let eventTicker: string;
  let outcomePart: string;

  if (marketTicker.includes('--')) {
    const parts = marketTicker.split('--').filter(Boolean);
    if (parts.length !== 2) return null;
    eventTicker = parts[0] ?? '';
    outcomePart = parts[1] ?? '';
  } else {
    const parts = marketTicker.split('-');
    if (parts.length < 3) return null;
    outcomePart = parts[parts.length - 1] ?? '';
    eventTicker = parts.slice(0, -1).join('-');
  }

  const seriesTicker = extractSeriesTicker(eventTicker);
  if (!seriesTicker) return null;

  const config = getConfigFromSeriesTicker(seriesTicker);
  if (!config) return null;

  const { sport, marketType } = config;

  // Parse outcome based on market type
  if (marketType === 'moneyline') {
    // Outcome is team/player code
    return {
      eventTicker,
      marketTicker,
      sport,
      marketType,
      outcomeCode: outcomePart,
    };
  }

  if (marketType === 'spread') {
    // Outcome is {TEAM}{BUCKET} like "NYK7" or "VGK1"
    // Extract team code and bucket number
    const match = outcomePart.match(/^([A-Z]+)(\d+)$/);
    if (!match) {
      // Fallback: treat entire thing as outcome code
      return {
        eventTicker,
        marketTicker,
        sport,
        marketType,
        outcomeCode: outcomePart,
      };
    }
    return {
      eventTicker,
      marketTicker,
      sport,
      marketType,
      outcomeCode: match[1] ?? outcomePart,
      spreadBucket: Number(match[2]),
    };
  }

  if (marketType === 'total') {
    // Outcome is strike value like "237" or "6"
    const strike = Number(outcomePart);
    if (!Number.isFinite(strike)) {
      return {
        eventTicker,
        marketTicker,
        sport,
        marketType,
        outcomeCode: outcomePart,
      };
    }
    return {
      eventTicker,
      marketTicker,
      sport,
      marketType,
      outcomeCode: `O${outcomePart}`, // Synthetic outcome code for over
      totalStrike: strike,
    };
  }

  return null;
}

/**
 * Build event ticker from components
 */
export function buildEventTicker(
  seriesTicker: string,
  dateYyyyMmDd: string,
  awayCode: string,
  homeCode: string
): string {
  // Convert date to YYMONDD format
  const [year, month, day] = dateYyyyMmDd.split('-');
  const yy = year?.slice(2) ?? '00';
  const monthNames = [
    '',
    'JAN',
    'FEB',
    'MAR',
    'APR',
    'MAY',
    'JUN',
    'JUL',
    'AUG',
    'SEP',
    'OCT',
    'NOV',
    'DEC',
  ];
  const mon = monthNames[Number(month)] ?? 'JAN';
  const dd = day ?? '01';

  return `${seriesTicker}-${yy}${mon}${dd}${awayCode}${homeCode}`.toUpperCase();
}
