/**
 * Sports Discovery Types
 *
 * Defines sport types, market types, and series configurations
 * for multi-sport market discovery.
 */

/** Supported sports */
export type Sport =
  | 'nba'
  | 'nfl'
  | 'nhl'
  | 'cbb'
  | 'wcbb'
  | 'mlb'
  | 'tennis-atp'
  | 'tennis-wta'
  | 'ufc';

/** Market types available for sports */
export type MarketType = 'moneyline' | 'spread' | 'total';

/** Sport display configuration */
export interface SportConfig {
  sport: Sport;
  label: string;
  shortLabel: string;
  /** Available market types for this sport */
  marketTypes: MarketType[];
  /** Team code length (3 for most, 3-4 for CBB) */
  teamCodeLength: number | 'variable';
}

/** Series ticker mapping */
export interface SeriesConfig {
  sport: Sport;
  marketType: MarketType;
  seriesTicker: string;
}

/** Parsed event info (common across sports) */
export interface ParsedEventInfo {
  eventTicker: string;
  seriesTicker: string;
  sport: Sport;
  marketType: MarketType;
  dateYyyyMmDd: string;
  /** For team sports: away team code */
  awayCode: string;
  /** For team sports: home team code */
  homeCode: string;
  /** For tennis: player 1 code */
  player1Code?: string;
  /** For tennis: player 2 code */
  player2Code?: string;
}

/** Parsed market info */
export interface ParsedMarketInfo {
  eventTicker: string;
  marketTicker: string;
  sport: Sport;
  marketType: MarketType;
  /** For moneylines: team code or player code */
  outcomeCode: string;
  /** For spreads: the bucket number from ticker (not actual spread) */
  spreadBucket?: number;
  /** For totals: the strike value from ticker */
  totalStrike?: number;
}

/** Sport configurations */
export const SPORT_CONFIGS: SportConfig[] = [
  {
    sport: 'nba',
    label: 'NBA',
    shortLabel: 'NBA',
    marketTypes: ['moneyline', 'spread', 'total'],
    teamCodeLength: 3,
  },
  // NFL season is over — hidden from tabs (config preserved for future reactivation)
  // {
  //   sport: 'nfl',
  //   label: 'NFL',
  //   shortLabel: 'NFL',
  //   marketTypes: ['moneyline', 'spread', 'total'],
  //   teamCodeLength: 3,
  // },
  {
    sport: 'nhl',
    label: 'NHL',
    shortLabel: 'NHL',
    marketTypes: ['moneyline', 'spread', 'total'],
    teamCodeLength: 3,
  },
  {
    sport: 'cbb',
    label: "College Basketball (Men's)",
    shortLabel: 'CBB',
    marketTypes: ['moneyline', 'spread', 'total'],
    teamCodeLength: 'variable', // 3-4 chars
  },
  {
    sport: 'tennis-atp',
    label: 'Tennis (ATP)',
    shortLabel: 'ATP',
    marketTypes: ['moneyline'],
    teamCodeLength: 3,
  },
  {
    sport: 'tennis-wta',
    label: 'Tennis (WTA)',
    shortLabel: 'WTA',
    marketTypes: ['moneyline'],
    teamCodeLength: 3,
  },
  // Dev-only sports — accessible via DEV tab, not promoted to production tabs yet.
  // Move to the active section above when ready for first-class support.
  // {
  //   sport: 'wcbb',
  //   label: "College Basketball (Women's)",
  //   shortLabel: 'WCBB',
  //   marketTypes: ['moneyline'],
  //   teamCodeLength: 'variable',
  // },
  // {
  //   sport: 'mlb',
  //   label: 'MLB',
  //   shortLabel: 'MLB',
  //   marketTypes: ['moneyline'],
  //   teamCodeLength: 'variable',
  // },
  // {
  //   sport: 'ufc',
  //   label: 'UFC',
  //   shortLabel: 'UFC',
  //   marketTypes: ['moneyline'],
  //   teamCodeLength: 3,
  // },
];

/** Series ticker mapping for each sport/market-type combination */
export const SERIES_CONFIGS: SeriesConfig[] = [
  // NBA
  { sport: 'nba', marketType: 'moneyline', seriesTicker: 'KXNBAGAME' },
  { sport: 'nba', marketType: 'spread', seriesTicker: 'KXNBASPREAD' },
  { sport: 'nba', marketType: 'total', seriesTicker: 'KXNBATOTAL' },
  // NFL
  { sport: 'nfl', marketType: 'moneyline', seriesTicker: 'KXNFLGAME' },
  { sport: 'nfl', marketType: 'spread', seriesTicker: 'KXNFLSPREAD' },
  { sport: 'nfl', marketType: 'total', seriesTicker: 'KXNFLTOTAL' },
  // NHL
  { sport: 'nhl', marketType: 'moneyline', seriesTicker: 'KXNHLGAME' },
  { sport: 'nhl', marketType: 'spread', seriesTicker: 'KXNHLSPREAD' },
  { sport: 'nhl', marketType: 'total', seriesTicker: 'KXNHLTOTAL' },
  // CBB (Men's)
  { sport: 'cbb', marketType: 'moneyline', seriesTicker: 'KXNCAAMBGAME' },
  { sport: 'cbb', marketType: 'spread', seriesTicker: 'KXNCAAMBSPREAD' },
  { sport: 'cbb', marketType: 'total', seriesTicker: 'KXNCAAMBTOTAL' },
  // WCBB (Women's)
  { sport: 'wcbb', marketType: 'moneyline', seriesTicker: 'KXNCAAWBGAME' },
  // MLB
  { sport: 'mlb', marketType: 'moneyline', seriesTicker: 'KXMLBGAME' },
  // Tennis
  { sport: 'tennis-atp', marketType: 'moneyline', seriesTicker: 'KXATPMATCH' },
  { sport: 'tennis-wta', marketType: 'moneyline', seriesTicker: 'KXWTAMATCH' },
  // UFC
  { sport: 'ufc', marketType: 'moneyline', seriesTicker: 'KXUFCFIGHT' },
];

/**
 * Get series ticker for a sport and market type
 */
export function getSeriesTicker(sport: Sport, marketType: MarketType): string | null {
  const config = SERIES_CONFIGS.find((c) => c.sport === sport && c.marketType === marketType);
  return config?.seriesTicker ?? null;
}

/**
 * Get sport config by sport id
 */
export function getSportConfig(sport: Sport): SportConfig | null {
  return SPORT_CONFIGS.find((c) => c.sport === sport) ?? null;
}

/**
 * Get all series tickers for a sport (all market types)
 */
export function getAllSeriesForSport(sport: Sport): string[] {
  return SERIES_CONFIGS.filter((c) => c.sport === sport).map((c) => c.seriesTicker);
}

/**
 * Determine sport and market type from series ticker
 */
export function getConfigFromSeriesTicker(
  seriesTicker: string
): { sport: Sport; marketType: MarketType } | null {
  const upper = seriesTicker.toUpperCase();
  const config = SERIES_CONFIGS.find((c) => c.seriesTicker === upper);
  if (!config) return null;
  return { sport: config.sport, marketType: config.marketType };
}
