/**
 * In-memory state management and diffing
 */

import type { NBAMarketRow } from '../types';

export class StateManager {
  private rows: Map<string, NBAMarketRow> = new Map();

  /**
   * Update state with new rows and return diff
   */
  update(newRows: NBAMarketRow[]): {
    snapshot: NBAMarketRow[];
    changedTickers: string[];
  } {
    const changedTickers: string[] = [];

    // Track existing tickers
    const existingTickers = new Set(this.rows.keys());

    for (const row of newRows) {
      const existing = this.rows.get(row.market_ticker);

      if (!existing) {
        // New row
        changedTickers.push(row.market_ticker);
        this.rows.set(row.market_ticker, row);
      } else {
        // Check if changed
        if (
          existing.awayProb !== row.awayProb ||
          existing.homeProb !== row.homeProb ||
          existing.awayLiq !== row.awayLiq ||
          existing.homeLiq !== row.homeLiq ||
          existing.gameTime !== row.gameTime
        ) {
          changedTickers.push(row.market_ticker);
          this.rows.set(row.market_ticker, row);
        }
      }
    }

    // Check for removed rows (markets that closed)
    for (const ticker of existingTickers) {
      if (!newRows.find((r) => r.market_ticker === ticker)) {
        changedTickers.push(ticker);
        this.rows.delete(ticker);
      }
    }

    // Return sorted snapshot
    const snapshot = Array.from(this.rows.values()).sort((a, b) => {
      // Sort by game time, then away team
      const timeCompare = a.sortKey.localeCompare(b.sortKey);
      if (timeCompare !== 0) return timeCompare;
      return a.awayTeam.localeCompare(b.awayTeam);
    });

    return { snapshot, changedTickers };
  }

  /**
   * Get current snapshot
   */
  getSnapshot(): NBAMarketRow[] {
    return Array.from(this.rows.values()).sort((a, b) => {
      const timeCompare = a.sortKey.localeCompare(b.sortKey);
      if (timeCompare !== 0) return timeCompare;
      return a.awayTeam.localeCompare(b.awayTeam);
    });
  }

  /**
   * Clear all state
   */
  clear(): void {
    this.rows.clear();
  }

  /**
   * Get row by ticker
   */
  getRow(ticker: string): NBAMarketRow | undefined {
    return this.rows.get(ticker);
  }
}
