/**
 * Market Stream (WebSocket-based)
 *
 * Uses WebSocket market data stream instead of polling.
 * Integrates with market state and time-series store.
 */

import type { KalshiApiClient, StreamUpdate, StreamCallback, NBAMarketRow } from '../types';
import { discoverNBAMarkets, transformToNBARows, generateMockNBARows } from './nba';
import { StateManager } from './state';
import { createMarketStream, type MarketStream } from './marketStream';
import { MarketState } from './marketState';
import type { PriceUpdate } from '../types';
import type { Environment } from '@galactus/shared';
import type { ConsolidatedGameBooks } from './nbaConsolidated/types';
import { makeGameKey, type VenueGameBooks } from './nbaConsolidated/types';
import {
  parseKalshiNbaEventTicker,
  parseKalshiNbaMarketTicker,
} from './nbaConsolidated/kalshiNbaTicker';
import { createPolyMarketStream } from './polymarket/marketStream';
import { makeNbaPolySlug, resolvePolyMarketInfoBySlug } from './polymarket/gamma';
import { polyAsksAsNoLevels, polyBidsAsYesLevels } from './polymarket/normalize';
import { applyStartTimeOffset } from './nbaConsolidated/startTimeSettings';
import { createSmartRelayConsolidatedStream } from './smartRelayAdapter';
import type { DataMode } from './dataMode';

export function orderbookToLevels(
  raw: Array<[number, number]> | null | undefined
): Array<{ priceCents: number; size: number }> {
  if (!Array.isArray(raw)) return [];
  const out: Array<{ priceCents: number; size: number }> = [];
  for (const rec of raw) {
    if (!Array.isArray(rec) || rec.length < 2) continue;
    const [p, q] = rec;
    const priceCents = Number(p);
    const size = Number(q);
    if (!Number.isFinite(priceCents) || !Number.isFinite(size)) continue;
    if (size <= 0) continue;
    out.push({ priceCents, size });
  }
  // Ensure descending by price like the `nba/` snapshots assume
  out.sort((a, b) => b.priceCents - a.priceCents);
  return out;
}

export interface NbaStream {
  start: () => void;
  stop: () => void;
  onUpdate: (callback: StreamCallback) => void;
  offUpdate: (callback: StreamCallback) => void;
}

export interface ConsolidatedNbaUpdate {
  games: ConsolidatedGameBooks[];
  lastUpdateMs: number;
}

export interface ConsolidatedNbaStream {
  start: () => void;
  stop: () => void;
  onUpdate: (callback: (update: ConsolidatedNbaUpdate) => void) => void;
  offUpdate: (callback: (update: ConsolidatedNbaUpdate) => void) => void;
}

interface StreamOptions {
  api: KalshiApiClient | null;
  accessKeyId: string;
  privateKey: CryptoKey;
  environment: Environment;
  useRelay: boolean;
  useMock: boolean;
  dataMode?: DataMode;
}

/**
 * Create NBA market stream (WebSocket-based)
 */
export function createNbaStream(options: StreamOptions): NbaStream {
  const { api, accessKeyId, privateKey, environment, useRelay, useMock } = options;
  const state = new StateManager();
  const marketState = new MarketState();
  setMarketState(marketState); // Make accessible globally
  const callbacks: Set<StreamCallback> = new Set();
  let marketStream: MarketStream | null = null;
  let _isFirstUpdate = true;
  let subscribedTickers: string[] = [];

  const emit = (update: StreamUpdate) => {
    callbacks.forEach((cb) => {
      try {
        cb(update);
      } catch (error) {
        console.error('Error in stream callback:', error);
      }
    });
  };

  // Handle price updates from WebSocket
  const handlePriceUpdate = (update: PriceUpdate) => {
    // Update market state
    marketState.updatePrice(update);

    // Update NBA rows with new price
    const row = state.getRow(update.ticker);
    if (row) {
      const previousPrice = row.currentPrice;
      row.currentPrice = update.price;
      row.priceChange = previousPrice !== undefined ? update.price - previousPrice : undefined;

      // Emit incremental update
      emit({
        type: 'price_update',
        priceUpdate: update,
        changedTickers: [update.ticker],
      });
    }
  };

  // Initial snapshot: fetch markets via REST, then subscribe to WebSocket
  const initialize = async () => {
    try {
      let rows: NBAMarketRow[];

      if (useMock || !api) {
        rows = generateMockNBARows();
      } else {
        // Fetch initial market data via REST
        const markets = await discoverNBAMarkets(api);
        rows = await transformToNBARows(markets, api);
      }

      // Update state with initial snapshot
      const { snapshot, changedTickers: _changedTickers } = state.update(rows);

      // Extract tickers for WebSocket subscription
      subscribedTickers = rows.map((r) => r.market_ticker);

      // Emit initial snapshot
      emit({ type: 'snapshot', rows: snapshot });
      _isFirstUpdate = false;

      // Connect to WebSocket market stream
      if (!useMock && api) {
        marketStream = createMarketStream(useRelay);

        // Set up price update handler
        marketStream.onPriceUpdate(handlePriceUpdate);

        // Set up error handler
        marketStream.onError((error) => {
          console.error('Market stream error:', error);
          // Could emit error to UI here
        });

        // Connect and subscribe
        await marketStream.connect(accessKeyId, privateKey, environment);
        marketStream.subscribe(subscribedTickers);
      }
    } catch (error) {
      console.error('Stream initialization error:', error);
      emit({
        type: 'snapshot',
        rows: [],
      });
    }
  };

  return {
    start: async () => {
      if (marketStream && marketStream.isConnected()) {
        return; // Already started
      }

      await initialize();
    },

    stop: () => {
      if (marketStream) {
        marketStream.disconnect();
        marketStream = null;
      }
      state.clear();
      marketState.clear();
      _isFirstUpdate = true;
      subscribedTickers = [];
    },

    onUpdate: (callback: StreamCallback) => {
      callbacks.add(callback);
    },

    offUpdate: (callback: StreamCallback) => {
      callbacks.delete(callback);
    },
  };
}

/**
 * Create consolidated NBA stream for the NBA Value Dashboard.
 *
 * Unlike `createNbaStream()`, this keeps **both** Kalshi market tickers per game
 * and subscribes to `orderbook_delta` for nba-style pricing parity.
 */
export function createConsolidatedNbaStream(options: StreamOptions): ConsolidatedNbaStream {
  // Smart relay mode: skip all REST discovery, use relay stream
  if (options.dataMode === 'smart-relay') {
    return createSmartRelayConsolidatedStream();
  }

  const { api, accessKeyId, privateKey, environment, useRelay, useMock } = options;
  const callbacks: Set<(u: ConsolidatedNbaUpdate) => void> = new Set();
  let marketStream: MarketStream | null = null;
  let polyStream: ReturnType<typeof createPolyMarketStream> | null = null;
  const gamesByEvent: Map<string, ConsolidatedGameBooks> = new Map();
  const eventToVenueBooks: Map<string, VenueGameBooks> = new Map();
  let tickers: string[] = [];
  let lastUpdateMs = 0;
  const polyTokenToEvent: Map<string, { eventTicker: string; side: 'away' | 'home' }> = new Map();

  const emit = () => {
    const games = Array.from(gamesByEvent.values());
    // stable ordering similar to nba/
    games.sort(
      (a, b) =>
        (a.date || '').localeCompare(b.date || '') ||
        a.awayCode.localeCompare(b.awayCode) ||
        a.homeCode.localeCompare(b.homeCode)
    );
    const payload: ConsolidatedNbaUpdate = { games, lastUpdateMs };
    callbacks.forEach((cb) => {
      try {
        cb(payload);
      } catch (err) {
        console.error('Error in consolidated stream callback:', err);
      }
    });
  };

  const initialize = async () => {
    gamesByEvent.clear();
    eventToVenueBooks.clear();
    tickers = [];
    lastUpdateMs = Date.now();

    if (useMock || !api) {
      emit();
      return;
    }

    // Discover NBA markets, then group by event_ticker keeping both market tickers.
    const markets = await discoverNBAMarkets(api);
    const grouped: Map<
      string,
      {
        eventInfo: ReturnType<typeof parseKalshiNbaEventTicker>;
        away?: string;
        home?: string;
        awayName?: string;
        homeName?: string;
      }
    > = new Map();

    for (const m of markets) {
      const mi = parseKalshiNbaMarketTicker(m.market_ticker);
      if (!mi) continue;
      const ei = parseKalshiNbaEventTicker(mi.eventTicker);
      if (!ei) continue;

      const entry = grouped.get(mi.eventTicker) ?? { eventInfo: ei };
      // Extract team names from yes_sub_title (e.g., "Boston Celtics")
      if (mi.teamCode === ei.awayCode) {
        entry.away = mi.marketTicker;
        if (m.yes_sub_title) entry.awayName = m.yes_sub_title;
      }
      if (mi.teamCode === ei.homeCode) {
        entry.home = mi.marketTicker;
        if (m.yes_sub_title) entry.homeName = m.yes_sub_title;
      }
      grouped.set(mi.eventTicker, entry);
    }

    // Build initial orderbooks via REST so UI has data before WS deltas arrive.
    for (const [eventTicker, entry] of grouped.entries()) {
      if (!entry.eventInfo || !entry.away || !entry.home) continue;
      const { dateYyyyMmDd, awayCode, homeCode } = entry.eventInfo;

      const [awayOb, homeOb, event] = await Promise.all([
        api.getOrderbook(entry.away),
        api.getOrderbook(entry.home),
        api.getEvent(eventTicker),
      ]);

      const rawStartTime = event?.start_time ?? null;
      let startTimePt: string | null = null;
      if (rawStartTime) {
        try {
          // Apply configurable offset (Kalshi reports times ~10 min early)
          const adjustedStartTime = applyStartTimeOffset(rawStartTime);
          const parts = new Intl.DateTimeFormat('en-US', {
            hour: '2-digit',
            minute: '2-digit',
            hour12: false,
          }).formatToParts(new Date(adjustedStartTime));
          const hh = parts.find((p) => p.type === 'hour')?.value;
          const mm = parts.find((p) => p.type === 'minute')?.value;
          if (hh && mm) startTimePt = `${hh}:${mm}`;
        } catch {
          startTimePt = null;
        }
      }

      const venueBooks: VenueGameBooks = {
        venue: 'kalshi',
        eventId: eventTicker,
        awayCode,
        homeCode,
        tsMs: Date.now(),
        markets: {
          away: {
            marketTicker: entry.away,
            yes: orderbookToLevels(awayOb?.yes),
            no: orderbookToLevels(awayOb?.no),
          },
          home: {
            marketTicker: entry.home,
            yes: orderbookToLevels(homeOb?.yes),
            no: orderbookToLevels(homeOb?.no),
          },
        },
      };

      const game: ConsolidatedGameBooks = {
        date: dateYyyyMmDd,
        awayCode,
        homeCode,
        awayName: entry.awayName,
        homeName: entry.homeName,
        key: makeGameKey({ date: dateYyyyMmDd, awayCode, homeCode }),
        startTimePt,
        kalshi: venueBooks,
        polymarket: null,
      };

      gamesByEvent.set(eventTicker, game);
      eventToVenueBooks.set(eventTicker, venueBooks);
      tickers.push(entry.away, entry.home);
    }

    lastUpdateMs = Date.now();
    emit();

    // Polymarket: resolve token IDs via Gamma and subscribe to market WS.
    try {
      polyTokenToEvent.clear();
      const slugsByEvent: Array<{
        eventTicker: string;
        slug: string;
        awayName?: string;
        homeName?: string;
      }> = [];
      for (const [eventTicker, g] of gamesByEvent.entries()) {
        slugsByEvent.push({
          eventTicker,
          slug: makeNbaPolySlug({
            dateYyyyMmDd: g.date,
            awayCode: g.awayCode,
            homeCode: g.homeCode,
          }),
          awayName: g.awayName,
          homeName: g.homeName,
        });
      }

      const infos = await Promise.all(
        slugsByEvent.map(async ({ eventTicker, slug, awayName, homeName }) => {
          const info = await resolvePolyMarketInfoBySlug({
            slug,
            useRelay,
            awayName,
            homeName,
          });
          return { eventTicker, info };
        })
      );

      const tokenIds: string[] = [];
      for (const { eventTicker, info } of infos) {
        if (!info) continue;
        const g = gamesByEvent.get(eventTicker);
        if (!g) continue;

        const venueBooks: VenueGameBooks = {
          venue: 'polymarket',
          eventId: info.slug,
          awayCode: g.awayCode,
          homeCode: g.homeCode,
          tsMs: Date.now(),
          markets: {
            away: {
              marketTicker: `asset:${info.awayTokenId}`,
              tokenId: info.awayTokenId,
              conditionId: info.conditionId,
              tickSize: info.tickSize,
              negRisk: info.negRisk,
              yes: [],
              no: [],
            },
            home: {
              marketTicker: `asset:${info.homeTokenId}`,
              tokenId: info.homeTokenId,
              conditionId: info.conditionId,
              tickSize: info.tickSize,
              negRisk: info.negRisk,
              yes: [],
              no: [],
            },
          },
        };

        g.polymarket = venueBooks;
        polyTokenToEvent.set(info.awayTokenId, { eventTicker, side: 'away' });
        polyTokenToEvent.set(info.homeTokenId, { eventTicker, side: 'home' });
        tokenIds.push(info.awayTokenId, info.homeTokenId);
      }

      if (tokenIds.length > 0) {
        polyStream = createPolyMarketStream();
        polyStream.onError((e) => console.warn('Polymarket market stream error:', e));
        polyStream.onBook((snap) => {
          const m = polyTokenToEvent.get(snap.assetId);
          if (!m) return;
          const g = gamesByEvent.get(m.eventTicker);
          if (!g || !g.polymarket) return;

          const toYes = polyBidsAsYesLevels(snap.bids);
          const toNo = polyAsksAsNoLevels(snap.asks);

          const market = m.side === 'away' ? g.polymarket.markets.away : g.polymarket.markets.home;
          market.yes = toYes;
          market.no = toNo;
          g.polymarket.tsMs = snap.tsMs;
          lastUpdateMs = Date.now();
          emit();
        });

        await polyStream.connect();
        polyStream.subscribe(Array.from(new Set(tokenIds)));
        lastUpdateMs = Date.now();
        emit();
      }
    } catch (err) {
      console.warn('Polymarket integration unavailable:', err);
    }

    // Connect WS and subscribe to orderbooks for all tickers.
    marketStream = createMarketStream(useRelay);
    marketStream.onError((error) => {
      console.error('Consolidated market stream error:', error);
    });

    marketStream.onOrderbookUpdate((u) => {
      const ticker = u.ticker;
      lastUpdateMs = Date.now();

      // Find which event this ticker belongs to by scanning our grouped state.
      // (Ticker list is small; keep it simple and robust.)
      for (const [_eventTicker, game] of gamesByEvent.entries()) {
        const kb = game.kalshi;
        if (!kb) continue;
        if (kb.markets.away.marketTicker === ticker) {
          kb.markets.away.yes = u.yes.map((lvl) => ({ priceCents: lvl.price, size: lvl.quantity }));
          kb.markets.away.no = u.no.map((lvl) => ({ priceCents: lvl.price, size: lvl.quantity }));
          kb.tsMs = lastUpdateMs;
          break;
        }
        if (kb.markets.home.marketTicker === ticker) {
          kb.markets.home.yes = u.yes.map((lvl) => ({ priceCents: lvl.price, size: lvl.quantity }));
          kb.markets.home.no = u.no.map((lvl) => ({ priceCents: lvl.price, size: lvl.quantity }));
          kb.tsMs = lastUpdateMs;
          break;
        }
      }

      emit();
    });

    await marketStream.connect(accessKeyId, privateKey, environment);
    marketStream.subscribeOrderbook(tickers);
  };

  return {
    start: async () => {
      if (marketStream && marketStream.isConnected()) return;
      await initialize();
    },
    stop: () => {
      if (marketStream) {
        marketStream.disconnect();
        marketStream = null;
      }
      if (polyStream) {
        polyStream.disconnect();
        polyStream = null;
      }
      gamesByEvent.clear();
      eventToVenueBooks.clear();
      tickers = [];
      lastUpdateMs = 0;
      polyTokenToEvent.clear();
    },
    onUpdate: (cb) => callbacks.add(cb),
    offUpdate: (cb) => callbacks.delete(cb),
  };
}

/**
 * Get market state (for accessing candlesticks, etc.)
 * This is a helper to access the market state from outside the stream
 */
let globalMarketState: MarketState | null = null;

export function getMarketState(): MarketState | null {
  return globalMarketState;
}

export function setMarketState(state: MarketState): void {
  globalMarketState = state;
}
