/**
 * Theme Manager
 *
 * Handles light/dark theme switching with localStorage persistence.
 * Follows the same pattern as pricingPOV.ts.
 */

export type Theme = 'light' | 'dark';

const STORAGE_KEY = 'app_theme';
const LIGHT_CLASS = 'light';

let currentTheme: Theme = 'dark';

/**
 * Apply theme class to document
 */
function applyTheme(theme: Theme): void {
  if (theme === 'light') {
    document.documentElement.classList.add(LIGHT_CLASS);
  } else {
    document.documentElement.classList.remove(LIGHT_CLASS);
  }
}

/**
 * Initialize theme from localStorage or system preference
 */
export function initTheme(): Theme {
  const saved = localStorage.getItem(STORAGE_KEY);

  if (saved === 'light' || saved === 'dark') {
    currentTheme = saved;
  } else {
    // Check system preference if no saved preference
    const prefersDark = window.matchMedia('(prefers-color-scheme: dark)').matches;
    currentTheme = prefersDark ? 'dark' : 'light';
  }

  applyTheme(currentTheme);
  return currentTheme;
}

/**
 * Get current theme
 */
export function getTheme(): Theme {
  return currentTheme;
}

/**
 * Set theme and save to localStorage
 */
export function setTheme(theme: Theme): void {
  currentTheme = theme;
  localStorage.setItem(STORAGE_KEY, theme);
  applyTheme(theme);
}

/**
 * Toggle between light and dark
 */
export function toggleTheme(): Theme {
  const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
  setTheme(newTheme);
  return newTheme;
}
