/**
 * Time-Series Store
 *
 * Stores price ticks and aggregates them into candlesticks.
 * Uses rolling window to manage memory.
 */

import type { PriceTick, Candlestick, CandlestickPeriod } from '../types';

const MAX_TICKS_PER_MARKET = 10000; // Keep last ~24h at 1 tick/second
const _DEFAULT_CANDLE_PERIODS: CandlestickPeriod[] = ['1m', '5m', '15m', '1h', '1d'];

export class TimeSeriesStore {
  // Raw ticks per market (rolling window)
  private ticks: Map<string, PriceTick[]> = new Map();

  // Pre-aggregated candles per market and period
  private candles: Map<string, Map<CandlestickPeriod, Candlestick[]>> = new Map();

  // Last update timestamp per market
  private lastUpdate: Map<string, number> = new Map();

  /**
   * Add a price tick
   */
  addTick(ticker: string, tick: PriceTick): void {
    if (!this.ticks.has(ticker)) {
      this.ticks.set(ticker, []);
    }

    const tickList = this.ticks.get(ticker)!;
    tickList.push(tick);

    // Trim to max size (rolling window)
    if (tickList.length > MAX_TICKS_PER_MARKET) {
      tickList.shift(); // Remove oldest
    }

    // Update last update time
    this.lastUpdate.set(ticker, tick.timestamp);

    // Invalidate candles (will be recalculated on-demand)
    // For performance, could update incrementally, but on-demand is simpler
    this.invalidateCandles(ticker);
  }

  /**
   * Get ticks for a time range
   */
  getTicks(ticker: string, start: number, end: number): PriceTick[] {
    const tickList = this.ticks.get(ticker) || [];
    return tickList.filter((t) => t.timestamp >= start && t.timestamp <= end);
  }

  /**
   * Get latest N ticks
   */
  getLatestTicks(ticker: string, count: number): PriceTick[] {
    const tickList = this.ticks.get(ticker) || [];
    return tickList.slice(-count);
  }

  /**
   * Get all ticks for a market
   */
  getAllTicks(ticker: string): PriceTick[] {
    return this.ticks.get(ticker) || [];
  }

  /**
   * Get candlesticks for a market and period
   */
  getCandles(ticker: string, period: CandlestickPeriod, count?: number): Candlestick[] {
    // Check cache
    const marketCandles = this.candles.get(ticker);
    if (marketCandles) {
      const cached = marketCandles.get(period);
      if (cached && cached.length > 0) {
        if (count) {
          return cached.slice(-count);
        }
        return cached;
      }
    }

    // Aggregate from ticks
    const ticks = this.getAllTicks(ticker);
    if (ticks.length === 0) {
      return [];
    }

    const candles = this.aggregateTicks(ticks, period, ticker);

    // Cache
    if (!marketCandles) {
      this.candles.set(ticker, new Map());
    }
    this.candles.get(ticker)!.set(period, candles);

    if (count) {
      return candles.slice(-count);
    }
    return candles;
  }

  /**
   * Get latest price for a market
   */
  getLatestPrice(ticker: string): number | null {
    const tickList = this.ticks.get(ticker);
    if (!tickList || tickList.length === 0) {
      return null;
    }
    return tickList[tickList.length - 1].price;
  }

  /**
   * Get last update timestamp
   */
  getLastUpdate(ticker: string): number | null {
    return this.lastUpdate.get(ticker) || null;
  }

  /**
   * Aggregate ticks into candlesticks
   */
  private aggregateTicks(
    ticks: PriceTick[],
    period: CandlestickPeriod,
    ticker: string
  ): Candlestick[] {
    if (ticks.length === 0) return [];

    const periodMs = this.getPeriodMs(period);
    const _candles: Candlestick[] = [];
    const candleMap = new Map<number, Candlestick>(); // Keyed by period start timestamp

    for (const tick of ticks) {
      const periodStart = Math.floor(tick.timestamp / periodMs) * periodMs;

      let candle = candleMap.get(periodStart);
      if (!candle) {
        candle = {
          open: tick.price,
          high: tick.price,
          low: tick.price,
          close: tick.price,
          volume: tick.volume || 0,
          timestamp: periodStart,
          period,
          ticker,
        };
        candleMap.set(periodStart, candle);
      } else {
        // Update existing candle
        candle.high = Math.max(candle.high, tick.price);
        candle.low = Math.min(candle.low, tick.price);
        candle.close = tick.price;
        candle.volume += tick.volume || 0;
      }
    }

    // Convert map to sorted array
    return Array.from(candleMap.values()).sort((a, b) => a.timestamp - b.timestamp);
  }

  /**
   * Get period duration in milliseconds
   */
  private getPeriodMs(period: CandlestickPeriod): number {
    switch (period) {
      case '1m':
        return 60 * 1000;
      case '5m':
        return 5 * 60 * 1000;
      case '15m':
        return 15 * 60 * 1000;
      case '1h':
        return 60 * 60 * 1000;
      case '1d':
        return 24 * 60 * 60 * 1000;
      default:
        return 60 * 1000;
    }
  }

  /**
   * Invalidate candles cache for a market
   */
  private invalidateCandles(ticker: string): void {
    const marketCandles = this.candles.get(ticker);
    if (marketCandles) {
      marketCandles.clear();
    }
  }

  /**
   * Clear all data for a market
   */
  clearMarket(ticker: string): void {
    this.ticks.delete(ticker);
    this.candles.delete(ticker);
    this.lastUpdate.delete(ticker);
  }

  /**
   * Clear all data
   */
  clear(): void {
    this.ticks.clear();
    this.candles.clear();
    this.lastUpdate.clear();
  }

  /**
   * Get all markets with data
   */
  getMarkets(): string[] {
    return Array.from(this.ticks.keys());
  }
}
