/**
 * Type coercion utilities for safely parsing unknown data
 *
 * Use these when handling JSON from external APIs or WebSocket messages.
 */

export type UnknownRecord = Record<string, unknown>;

/** Safely cast unknown to a Record, returns null if not an object */
export function asRecord(v: unknown): UnknownRecord | null {
  if (!v || typeof v !== 'object') return null;
  return v as UnknownRecord;
}

/** Parse unknown to number, returns null if not a valid finite number */
export function toNumber(v: unknown): number | null {
  if (typeof v === 'number' && Number.isFinite(v)) return v;
  if (typeof v === 'string') {
    const n = Number(v);
    if (Number.isFinite(n)) return n;
  }
  return null;
}

/** Parse unknown to non-empty string, returns null otherwise */
export function toString(v: unknown): string | null {
  return typeof v === 'string' && v.trim() ? v : null;
}

/** Parse unknown to 'yes' | 'no', returns null if invalid */
export function toSide(v: unknown): 'yes' | 'no' | null {
  if (typeof v !== 'string') return null;
  const s = v.toLowerCase();
  return s === 'yes' || s === 'no' ? s : null;
}

/** Parse unknown to 'buy' | 'sell', returns null if invalid */
export function toAction(v: unknown): 'buy' | 'sell' | null {
  if (typeof v !== 'string') return null;
  const s = v.toLowerCase();
  return s === 'buy' || s === 'sell' ? s : null;
}

/** Return the first non-null/undefined value */
export function pickFirst(...values: unknown[]): unknown {
  for (const v of values) {
    if (v !== undefined && v !== null) return v;
  }
  return undefined;
}

/**
 * Normalize a timestamp to milliseconds
 *
 * Handles: seconds, milliseconds, Date objects, ISO strings
 * Returns null if unable to parse (use normalizeTimestampMsWithFallback for Date.now() fallback)
 */
export function normalizeTimestampMs(raw: unknown): number | null {
  if (raw === null || raw === undefined) return null;

  if (raw instanceof Date) {
    const t = raw.getTime();
    return Number.isFinite(t) ? t : null;
  }

  if (typeof raw === 'number' && Number.isFinite(raw)) {
    // Heuristic: seconds vs ms
    // - seconds since epoch ~ 1.7e9 (2026)
    // - ms since epoch ~ 1.7e12
    if (raw > 1e12) return Math.floor(raw);
    if (raw > 1e9) return Math.floor(raw * 1000);
    return null;
  }

  if (typeof raw === 'string') {
    const trimmed = raw.trim();
    if (!trimmed) return null;

    // numeric string?
    if (/^\d+(\.\d+)?$/.test(trimmed)) {
      const n = Number(trimmed);
      return normalizeTimestampMs(n);
    }

    // ISO date?
    const parsed = Date.parse(trimmed);
    if (!Number.isNaN(parsed)) return parsed;
  }

  return null;
}

/**
 * Normalize a timestamp to milliseconds, falling back to Date.now() if unparseable
 */
export function normalizeTimestampMsWithFallback(raw: unknown): number {
  return normalizeTimestampMs(raw) ?? Date.now();
}

/** Safe JSON parse that returns unknown (null on error) */
export function safeJsonParse(s: string): unknown {
  try {
    return JSON.parse(s);
  } catch {
    return null;
  }
}
