/**
 * Mock candlestick data for chart demonstrations
 */

export interface MockCandle {
  timestamp: number;
  open: number;
  high: number;
  low: number;
  close: number;
  volume: number;
}

/**
 * Generate realistic-looking candlestick data
 */
export function generateMockCandles(
  basePrice: number = 50,
  count: number = 100,
  intervalMs: number = 5 * 60 * 1000, // 5 minutes default
  volatility: number = 0.02
): MockCandle[] {
  const candles: MockCandle[] = [];
  let currentPrice = basePrice;
  const now = Date.now();
  const startTime = now - count * intervalMs;

  for (let i = 0; i < count; i++) {
    const timestamp = startTime + i * intervalMs;

    // Random walk with mean reversion
    const trend = (basePrice - currentPrice) * 0.01; // Mean reversion
    const noise = (Math.random() - 0.5) * 2 * volatility * currentPrice;
    const change = trend + noise;

    const open = currentPrice;
    const close = Math.max(1, Math.min(99, currentPrice + change));

    // High and low based on volatility
    const range = Math.abs(close - open) + Math.random() * volatility * currentPrice;
    const high = Math.min(99, Math.max(open, close) + Math.random() * range * 0.5);
    const low = Math.max(1, Math.min(open, close) - Math.random() * range * 0.5);

    // Volume with some randomness
    const volume = Math.floor(1000 + Math.random() * 5000 * (1 + Math.abs(change) / currentPrice));

    candles.push({
      timestamp,
      open: Math.round(open * 100) / 100,
      high: Math.round(high * 100) / 100,
      low: Math.round(low * 100) / 100,
      close: Math.round(close * 100) / 100,
      volume,
    });

    currentPrice = close;
  }

  return candles;
}

/**
 * Generate candles for a specific market scenario
 */
export function generateScenarioCandles(
  scenario: 'bullish' | 'bearish' | 'sideways' | 'volatile',
  basePrice: number = 50,
  count: number = 100
): MockCandle[] {
  const scenarios = {
    bullish: { drift: 0.001, volatility: 0.015 },
    bearish: { drift: -0.001, volatility: 0.015 },
    sideways: { drift: 0, volatility: 0.008 },
    volatile: { drift: 0, volatility: 0.035 },
  };

  const config = scenarios[scenario];
  const candles: MockCandle[] = [];
  let currentPrice = basePrice;
  const now = Date.now();
  const intervalMs = 5 * 60 * 1000;
  const startTime = now - count * intervalMs;

  for (let i = 0; i < count; i++) {
    const timestamp = startTime + i * intervalMs;

    const drift = config.drift * currentPrice;
    const noise = (Math.random() - 0.5) * 2 * config.volatility * currentPrice;
    const change = drift + noise;

    const open = currentPrice;
    const close = Math.max(1, Math.min(99, currentPrice + change));

    const range = Math.abs(close - open) + Math.random() * config.volatility * currentPrice;
    const high = Math.min(99, Math.max(open, close) + Math.random() * range * 0.5);
    const low = Math.max(1, Math.min(open, close) - Math.random() * range * 0.5);

    const volume = Math.floor(1000 + Math.random() * 5000);

    candles.push({
      timestamp,
      open: Math.round(open * 100) / 100,
      high: Math.round(high * 100) / 100,
      low: Math.round(low * 100) / 100,
      close: Math.round(close * 100) / 100,
      volume,
    });

    currentPrice = close;
  }

  return candles;
}

// Pre-generated data for consistent demos
export const mockLakersCandles = generateScenarioCandles('bullish', 42, 150);
export const mockChiefsCandles = generateScenarioCandles('sideways', 51, 150);
export const mockMahomesCandles = generateScenarioCandles('volatile', 38, 150);
export const mockLebronCandles = generateScenarioCandles('bearish', 72, 150);
