// Kalshi API Types
import type { Environment } from '@galactus/shared';

export type { Environment };

// Re-export core app types that are implemented in modules but widely used across UI.
export type { KalshiApiClient, KalshiOrder, KalshiFill } from './lib/kalshiApi';
export type { PricingPOV } from './lib/pricingPOV';
export type { OddsFormat } from './lib/oddsConversion';

export interface KalshiCredentials {
  accessKeyId: string;
  privateKeyPem: string;
  environment: Environment;
}

export interface KalshiSeries {
  series_ticker: string;
  ticker?: string; // API sometimes returns 'ticker' instead of 'series_ticker'
  title: string;
  category: string;
  subcategory?: string;
  frequency?: string; // e.g., "daily", "weekly", "one-time"
  tags?: string[];
  volume?: number;
  volume_fp?: string; // Fixed-point volume string
  fee_type?: string;
  fee_multiplier?: number;
  settlement_sources?: Array<{ name: string; url: string }>;
  contract_url?: string;
  min_tick_size?: string;
  expiry_time?: string;
}

export interface KalshiMarket {
  market_ticker: string;
  title: string;
  series_ticker: string;
  event_ticker: string;
  subtitle: string;
  open_time: string;
  close_time: string;
  expiry_time: string;
  yes_bid: number | null;
  yes_ask: number | null;
  no_bid: number | null;
  no_ask: number | null;
  last_price: number | null;
  previous_price: number | null;
  volume: number;
  open_interest: number;
  liquidity: number;
  yes_price?: number;
  no_price?: number;
  // Strike fields - used to differentiate markets within the same event
  floor_strike?: number | null;
  cap_strike?: number | null;
  strike_type?: string; // e.g., "greater", "less", "between"
  functional_strike?: string; // Human-readable strike description
  yes_sub_title?: string; // Description of what Yes means
  no_sub_title?: string; // Description of what No means
}

// Raw orderbook from API - arrays of [price, quantity] tuples
// The API returns bids only for each side (yes/no)
// YES asks are derived from NO bids (100 - NO_bid_price)
export interface KalshiOrderbookRaw {
  orderbook: {
    yes: Array<[number, number]>; // [[price, qty], ...] - YES bids
    no: Array<[number, number]>; // [[price, qty], ...] - NO bids
  };
}

// Legacy interface (kept for backward compatibility with existing components)
export interface KalshiOrderbook {
  market_ticker?: string;
  yes: Array<[number, number]>; // [[price, qty], ...] - YES bids
  no: Array<[number, number]>; // [[price, qty], ...] - NO bids
}

export interface KalshiEvent {
  event_ticker: string;
  title: string;
  subtitle?: string;
  sub_title?: string; // API sometimes uses this
  category: string;
  series_ticker: string;
  start_time?: string;
  expiry_time?: string;
  strike_date?: string; // When the event resolves
  strike_period?: string; // e.g., "Q1 2026"
  mutually_exclusive?: boolean;
  status?: string;
  markets?: KalshiMarket[]; // When with_nested_markets=true
}

// Price update types (WebSocket)
export interface PriceTick {
  ticker: string;
  timestamp: number; // ms since epoch
  price: number; // last_price or mid_price
  yes_bid?: number;
  yes_ask?: number;
  no_bid?: number;
  no_ask?: number;
  volume?: number;
  open_interest?: number;
}

export interface PriceUpdate {
  ticker: string;
  price: number;
  timestamp: number;
  // Only changed fields
  yes_bid?: number;
  yes_ask?: number;
  no_bid?: number;
  no_ask?: number;
  volume?: number;
}

// Candlestick types
export type CandlestickPeriod = '1m' | '5m' | '15m' | '1h' | '1d';

export interface Candlestick {
  open: number;
  high: number;
  low: number;
  close: number;
  volume: number;
  timestamp: number; // start of period (ms)
  period: CandlestickPeriod;
  ticker: string;
}

// Portfolio types
export interface Portfolio {
  id: string;
  name: string;
  markets: Array<{
    ticker: string;
    position?: number; // for weighted aggregation
    weight?: number; // for weighted aggregation
  }>;
}

export interface PortfolioCandlestick {
  portfolioId: string;
  period: CandlestickPeriod;
  candles: Candlestick[];
}

// NBA Market Row (legacy, for compatibility)
export interface NBAMarketRow {
  market_ticker: string;
  gameDate: string;
  gameTime: string;
  awayTeam: string;
  homeTeam: string;
  awayProb: number | null;
  homeProb: number | null;
  awayLiq: number;
  homeLiq: number;
  sortKey: string;
  // Add current price for real-time updates
  currentPrice?: number;
  priceChange?: number;
  // Bid/ask data
  awayYesBid?: number | null;
  awayYesAsk?: number | null;
  homeYesBid?: number | null;
  homeYesAsk?: number | null;
}

// Stream Events
export interface StreamUpdate {
  type: 'snapshot' | 'update' | 'price_update';
  rows?: NBAMarketRow[];
  priceUpdate?: PriceUpdate;
  changedTickers?: string[];
}

export type StreamCallback = (update: StreamUpdate) => void;

// State
export interface AppState {
  connected: boolean;
  lastUpdated: Date | null;
  error: string | null;
  rows: Map<string, NBAMarketRow>;
}

// Position tracking
export interface Position {
  ticker: string;
  side: 'yes' | 'no';
  quantity: number; // Positive = long, negative = short
  averagePrice: number;
  currentPrice: number | null;
  realizedPnl: number;
  unrealizedPnl: number;
  totalPnl: number;
  pnlPercent: number;
}

// Fill notification
export interface FillNotification {
  ticker: string;
  side: 'yes' | 'no';
  action: 'buy' | 'sell';
  quantity: number;
  price: number;
  timestamp: number;
  orderId: string;
}
