import { ReactNode } from 'react';
import {
  Target,
  Fence,
  Lightbulb,
  AlertTriangle,
  Sparkles,
  ArrowRight,
  Play,
  BookOpen,
} from 'lucide-react';

/**
 * Lesson Block Components
 *
 * Visual language for different content types in lessons.
 * Each block has distinct styling to aid quick recognition.
 */

// =============================================================================
// OBJECTIVE BLOCK
// What you'll learn - appears at the top of lessons
// =============================================================================

interface ObjectiveBlockProps {
  children: ReactNode;
}

export function ObjectiveBlock({ children }: ObjectiveBlockProps) {
  return (
    <div className="bg-primary/5 border-primary/20 flex items-start gap-3 rounded-lg border p-4">
      <Target className="text-primary mt-0.5 h-5 w-5 flex-shrink-0" />
      <div className="text-sm">
        <span className="text-primary font-medium">Objective: </span>
        <span className="text-foreground">{children}</span>
      </div>
    </div>
  );
}

// =============================================================================
// CONSTRAINT BLOCK
// Scope boundaries - what you're NOT doing
// =============================================================================

interface ConstraintBlockProps {
  items: string[];
  title?: string;
}

export function ConstraintBlock({ items, title = 'Constraints' }: ConstraintBlockProps) {
  return (
    <div className="relative rounded-r-lg border-l-4 border-amber-500/70 bg-amber-500/5 py-4 pl-4 pr-4">
      <div className="mb-2 flex items-center gap-2">
        <Fence className="h-4 w-4 text-amber-600 dark:text-amber-400" />
        <h3 className="text-sm font-semibold text-amber-700 dark:text-amber-300">{title}</h3>
      </div>
      <p className="text-muted-foreground mb-3 text-xs">
        These limits help you focus. Don&apos;t try to do more than this.
      </p>
      <ul className="space-y-1.5">
        {items.map((item, i) => (
          <li
            key={i}
            className="flex items-start gap-2 text-sm"
          >
            <span className="mt-1 text-xs text-amber-500">&#x25A0;</span>
            <span>{item}</span>
          </li>
        ))}
      </ul>
    </div>
  );
}

// =============================================================================
// STEP LIST
// Numbered instructions - the core "what to do"
// =============================================================================

interface StepListProps {
  steps: string[];
  title?: string;
}

export function StepList({ steps, title = 'Instructions' }: StepListProps) {
  return (
    <div className="space-y-4">
      <h2 className="text-lg font-semibold">{title}</h2>
      <ol className="space-y-4">
        {steps.map((step, i) => (
          <li
            key={i}
            className="flex gap-4"
          >
            <span className="bg-primary/10 text-primary flex h-7 w-7 flex-shrink-0 items-center justify-center rounded-full text-sm font-semibold">
              {i + 1}
            </span>
            <p className="pt-1 text-sm leading-relaxed">{step}</p>
          </li>
        ))}
      </ol>
    </div>
  );
}

// =============================================================================
// TIP BLOCK
// Helpful hints and advice
// =============================================================================

interface TipBlockProps {
  children: ReactNode;
  title?: string;
}

export function TipBlock({ children, title = 'Tip' }: TipBlockProps) {
  return (
    <div className="relative rounded-r-lg border-l-4 border-sky-500/70 bg-sky-500/5 py-3 pl-4 pr-4">
      <div className="mb-1 flex items-center gap-2">
        <Lightbulb className="h-4 w-4 text-sky-600 dark:text-sky-400" />
        <h3 className="text-sm font-medium text-sky-700 dark:text-sky-300">{title}</h3>
      </div>
      <div className="text-muted-foreground text-sm">{children}</div>
    </div>
  );
}

// =============================================================================
// WARNING BLOCK
// Common mistakes and cautions
// =============================================================================

interface WarningBlockProps {
  children: ReactNode;
  title?: string;
}

export function WarningBlock({ children, title = 'Watch out' }: WarningBlockProps) {
  return (
    <div className="relative rounded-r-lg border-l-4 border-rose-500/70 bg-rose-500/5 py-3 pl-4 pr-4">
      <div className="mb-1 flex items-center gap-2">
        <AlertTriangle className="h-4 w-4 text-rose-600 dark:text-rose-400" />
        <h3 className="text-sm font-medium text-rose-700 dark:text-rose-300">{title}</h3>
      </div>
      <div className="text-muted-foreground text-sm">{children}</div>
    </div>
  );
}

// =============================================================================
// INSIGHT BLOCK
// Key takeaways and important concepts
// =============================================================================

interface InsightBlockProps {
  children: ReactNode;
  title?: string;
}

export function InsightBlock({ children, title = 'Key Insight' }: InsightBlockProps) {
  return (
    <div className="relative rounded-r-lg border-l-4 border-violet-500/70 bg-violet-500/5 py-3 pl-4 pr-4">
      <div className="mb-1 flex items-center gap-2">
        <Sparkles className="h-4 w-4 text-violet-600 dark:text-violet-400" />
        <h3 className="text-sm font-medium text-violet-700 dark:text-violet-300">{title}</h3>
      </div>
      <div className="text-muted-foreground text-sm">{children}</div>
    </div>
  );
}

// =============================================================================
// TRY THIS BLOCK
// Interactive prompts and exercises
// =============================================================================

interface TryThisBlockProps {
  children: ReactNode;
  title?: string;
}

export function TryThisBlock({ children, title = 'Try This' }: TryThisBlockProps) {
  return (
    <div className="rounded-lg border-2 border-dashed border-emerald-500/50 bg-emerald-500/5 p-4">
      <div className="mb-2 flex items-center gap-2">
        <Play className="h-4 w-4 text-emerald-600 dark:text-emerald-400" />
        <h3 className="text-sm font-medium text-emerald-700 dark:text-emerald-300">{title}</h3>
      </div>
      <div className="text-sm">{children}</div>
    </div>
  );
}

// =============================================================================
// CONCEPT BLOCK
// Detailed theory and explanations - works in sidebar or main content
// =============================================================================

interface ConceptBlockProps {
  title: string;
  children: ReactNode;
}

export function ConceptBlock({ title, children }: ConceptBlockProps) {
  return (
    <div className="bg-card rounded-lg border p-4">
      <div className="mb-2 flex items-center gap-2">
        <BookOpen className="text-muted-foreground h-4 w-4" />
        <h3 className="text-sm font-medium">{title}</h3>
      </div>
      <div className="text-muted-foreground text-sm">{children}</div>
    </div>
  );
}

// =============================================================================
// CONCEPT CARD
// Brief concept overview - for grids showing multiple related concepts
// =============================================================================

interface ConceptCardProps {
  title: string;
  description: string;
  icon?: React.ReactNode;
}

export function ConceptCard({ title, description, icon }: ConceptCardProps) {
  return (
    <div className="bg-card/50 flex gap-3 rounded-lg border p-4">
      {icon && (
        <div className="bg-primary/10 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-md">
          <span className="text-primary">{icon}</span>
        </div>
      )}
      <div>
        <p className="text-sm font-medium">{title}</p>
        <p className="text-muted-foreground mt-0.5 text-xs">{description}</p>
      </div>
    </div>
  );
}

// =============================================================================
// NEXT STEP BLOCK
// Session completion and what to do after
// =============================================================================

interface NextStepBlockProps {
  href: string;
  title?: string;
  description?: string;
  buttonText?: string;
}

export function NextStepBlock({
  href,
  title = "When you're done",
  description = 'Continue to the next lesson.',
  buttonText = 'Next Lesson',
}: NextStepBlockProps) {
  return (
    <div className="from-primary/10 via-primary/5 border-primary/20 rounded-lg border bg-gradient-to-br to-transparent p-5">
      <div className="flex items-start gap-3">
        <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
          <ArrowRight className="text-primary h-4 w-4" />
        </div>
        <div className="space-y-3">
          <div>
            <h3 className="font-semibold">{title}</h3>
            <p className="text-muted-foreground mt-1 text-sm">{description}</p>
          </div>
          <a
            href={href}
            className="bg-primary text-primary-foreground hover:bg-primary/90 inline-flex items-center justify-center rounded-md px-4 py-2 text-sm font-medium transition-colors"
          >
            {buttonText}
          </a>
        </div>
      </div>
    </div>
  );
}

// =============================================================================
// SIDEBAR SECTION
// Wrapper for sidebar content groups
// =============================================================================

interface SidebarSectionProps {
  title?: string;
  children: ReactNode;
}

export function SidebarSection({ title, children }: SidebarSectionProps) {
  return (
    <div className="space-y-3">
      {title && (
        <h3 className="text-muted-foreground text-xs font-semibold uppercase tracking-wider">
          {title}
        </h3>
      )}
      <div className="space-y-3">{children}</div>
    </div>
  );
}

// =============================================================================
// TERM DEFINITION
// For glossary-style inline definitions
// =============================================================================

interface TermProps {
  term: string;
  children: ReactNode;
}

export function Term({ term, children }: TermProps) {
  return (
    <div className="border-muted border-l-2 py-1 pl-3">
      <dt className="text-foreground text-sm font-medium">{term}</dt>
      <dd className="text-muted-foreground mt-0.5 text-sm">{children}</dd>
    </div>
  );
}

// =============================================================================
// EXAMPLE BOX
// For showing concrete examples
// =============================================================================

interface ExampleBoxProps {
  title?: string;
  children: ReactNode;
}

export function ExampleBox({ title = 'Example', children }: ExampleBoxProps) {
  return (
    <div className="bg-muted/30 rounded-lg border p-4">
      <div className="text-muted-foreground mb-2 text-xs font-semibold uppercase tracking-wider">
        {title}
      </div>
      <div className="text-sm">{children}</div>
    </div>
  );
}
