import { useState } from 'react';
import { LessonRow, SectionHeader } from '../LessonLayout';
import { TipBlock, InsightBlock, WarningBlock, ConstraintBlock } from '../blocks';

/**
 * Building for Market Makers
 *
 * This is a PRODUCT guide, not a trading guide.
 * It's designed to help someone building trading software understand
 * their market maker users deeply enough to build the right UX.
 */

// Interactive visualization of the dual market structure
function DualMarketVisualization() {
  const [hoveredBook, setHoveredBook] = useState<string | null>(null);
  const [selectedScenario, setSelectedScenario] = useState<'retail' | 'mm'>('retail');

  const scenarios = {
    retail: {
      title: "Retail Bettor's Perspective",
      description: '"I think the Lakers will win, so I\'ll buy Lakers YES"',
      flow: [
        { step: 1, action: 'Opens Lakers market', book: 'lakers-yes' },
        { step: 2, action: 'Sees Lakers YES price: 55¢', book: 'lakers-yes' },
        { step: 3, action: 'Clicks BUY on Lakers YES', book: 'lakers-yes' },
        { step: 4, action: 'Gets matched with someone selling', book: 'lakers-no' },
      ],
      insight:
        "Retail thinks in team loyalty: 'My team YES'. They never look at the opponent's NO book.",
    },
    mm: {
      title: "Market Maker's Perspective",
      description: '"I\'ll provide liquidity on the NO side and capture spread"',
      flow: [
        { step: 1, action: 'Analyzes fair price (~54%)', book: null },
        { step: 2, action: 'Posts bid on Lakers NO at 46¢', book: 'lakers-no' },
        { step: 3, action: 'Joins queue, waits for retail', book: 'lakers-no' },
        { step: 4, action: 'Retail buys Lakers YES, MM gets filled', book: 'lakers-no' },
      ],
      insight: "Market maker sits in NO books because that's where they intercept retail YES flow.",
    },
  };

  const currentScenario = scenarios[selectedScenario];

  return (
    <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-6">
      <h4 className="mb-4 text-lg font-semibold text-white">
        The Dual Market Structure: Two Perspectives
      </h4>

      <div className="mb-6 flex gap-2">
        <button
          onClick={() => setSelectedScenario('retail')}
          className={`rounded-lg px-4 py-2 text-sm font-medium ${
            selectedScenario === 'retail'
              ? 'bg-blue-600 text-white'
              : 'bg-slate-700 text-slate-300 hover:bg-slate-600'
          }`}
        >
          Retail Bettor
        </button>
        <button
          onClick={() => setSelectedScenario('mm')}
          className={`rounded-lg px-4 py-2 text-sm font-medium ${
            selectedScenario === 'mm'
              ? 'bg-purple-600 text-white'
              : 'bg-slate-700 text-slate-300 hover:bg-slate-600'
          }`}
        >
          Market Maker (Your Client)
        </button>
      </div>

      <div className="mb-6 rounded-lg bg-slate-900/50 p-4">
        <div className="font-medium text-white">{currentScenario.title}</div>
        <div className="text-sm italic text-slate-400">{currentScenario.description}</div>
      </div>

      {/* Visual representation of the 4 order books */}
      <div className="mb-6 grid grid-cols-2 gap-4">
        <div className="space-y-2">
          <div className="text-center text-sm font-medium text-slate-400">LAKERS</div>
          <div
            onMouseEnter={() => setHoveredBook('lakers-yes')}
            onMouseLeave={() => setHoveredBook(null)}
            className={`rounded-lg border-2 p-4 transition-all ${
              currentScenario.flow.some((f) => f.book === 'lakers-yes')
                ? 'border-blue-500 bg-blue-900/30'
                : hoveredBook === 'lakers-yes'
                  ? 'border-slate-500 bg-slate-700'
                  : 'border-slate-600 bg-slate-800'
            }`}
          >
            <div className="font-semibold text-green-400">Lakers YES</div>
            <div className="text-xs text-slate-400">Pays $1 if Lakers win</div>
            <div
              className={`mt-1 text-xs ${
                currentScenario.flow.some((f) => f.book === 'lakers-yes')
                  ? 'text-blue-400'
                  : 'text-slate-500'
              }`}
            >
              ↑ Retail buys here
            </div>
          </div>
          <div
            onMouseEnter={() => setHoveredBook('lakers-no')}
            onMouseLeave={() => setHoveredBook(null)}
            className={`rounded-lg border-2 p-4 transition-all ${
              currentScenario.flow.some((f) => f.book === 'lakers-no')
                ? selectedScenario === 'mm'
                  ? 'border-purple-500 bg-purple-900/30'
                  : 'border-blue-500 bg-blue-900/30'
                : hoveredBook === 'lakers-no'
                  ? 'border-slate-500 bg-slate-700'
                  : 'border-slate-600 bg-slate-800'
            }`}
          >
            <div className="font-semibold text-red-400">Lakers NO</div>
            <div className="text-xs text-slate-400">Pays $1 if Lakers lose</div>
            <div
              className={`mt-1 text-xs ${
                selectedScenario === 'mm' ? 'text-purple-400' : 'text-slate-500'
              }`}
            >
              {selectedScenario === 'mm' ? '★ MM posts here' : '= Celtics YES'}
            </div>
          </div>
        </div>

        <div className="space-y-2">
          <div className="text-center text-sm font-medium text-slate-400">CELTICS</div>
          <div
            onMouseEnter={() => setHoveredBook('celtics-yes')}
            onMouseLeave={() => setHoveredBook(null)}
            className={`rounded-lg border-2 p-4 transition-all ${
              hoveredBook === 'celtics-yes'
                ? 'border-slate-500 bg-slate-700'
                : 'border-slate-600 bg-slate-800'
            }`}
          >
            <div className="font-semibold text-green-400">Celtics YES</div>
            <div className="text-xs text-slate-400">Pays $1 if Celtics win</div>
            <div className="mt-1 text-xs text-slate-500">= Lakers NO</div>
          </div>
          <div
            onMouseEnter={() => setHoveredBook('celtics-no')}
            onMouseLeave={() => setHoveredBook(null)}
            className={`rounded-lg border-2 p-4 transition-all ${
              hoveredBook === 'celtics-no'
                ? 'border-slate-500 bg-slate-700'
                : 'border-slate-600 bg-slate-800'
            }`}
          >
            <div className="font-semibold text-red-400">Celtics NO</div>
            <div className="text-xs text-slate-400">Pays $1 if Celtics lose</div>
            <div className="mt-1 text-xs text-slate-500">= Lakers YES</div>
          </div>
        </div>
      </div>

      {/* Flow steps */}
      <div className="mb-4 space-y-2">
        {currentScenario.flow.map((step) => (
          <div
            key={step.step}
            className="flex items-center gap-3 rounded bg-slate-900/50 p-2"
          >
            <div
              className={`flex h-6 w-6 items-center justify-center rounded-full text-xs font-bold ${
                selectedScenario === 'mm' ? 'bg-purple-600' : 'bg-blue-600'
              } text-white`}
            >
              {step.step}
            </div>
            <div className="text-sm text-slate-300">{step.action}</div>
          </div>
        ))}
      </div>

      <div className="rounded-lg border border-amber-700/50 bg-amber-900/20 p-3">
        <div className="text-sm font-medium text-amber-400">Key Insight</div>
        <div className="text-sm text-slate-300">{currentScenario.insight}</div>
      </div>
    </div>
  );
}

// Queue position visualization
function QueueVisualization() {
  const [queuePosition, setQueuePosition] = useState(47);
  const [totalInQueue, setTotalInQueue] = useState(234);
  const [fillRate, setFillRate] = useState(12); // contracts per minute

  const timeToFill = Math.ceil(queuePosition / fillRate);
  const percentThrough = ((totalInQueue - queuePosition) / totalInQueue) * 100;

  return (
    <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-6">
      <h4 className="mb-4 text-lg font-semibold text-white">Understanding Queue Position</h4>
      <p className="mb-6 text-sm text-slate-400">
        When multiple orders are at the same price, they&apos;re filled first-come-first-served.
        Your position in queue determines when you get filled.
      </p>

      <div className="mb-6 grid grid-cols-3 gap-4">
        <div>
          <label className="mb-1 block text-sm text-slate-400">Your Position</label>
          <input
            type="range"
            min="1"
            max={totalInQueue}
            value={queuePosition}
            onChange={(e) => setQueuePosition(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center font-mono text-white">#{queuePosition}</div>
        </div>
        <div>
          <label className="mb-1 block text-sm text-slate-400">Total in Queue</label>
          <input
            type="range"
            min="50"
            max="500"
            value={totalInQueue}
            onChange={(e) => setTotalInQueue(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center font-mono text-white">{totalInQueue} contracts</div>
        </div>
        <div>
          <label className="mb-1 block text-sm text-slate-400">Fill Rate</label>
          <input
            type="range"
            min="1"
            max="50"
            value={fillRate}
            onChange={(e) => setFillRate(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center font-mono text-white">{fillRate}/min</div>
        </div>
      </div>

      {/* Visual queue */}
      <div className="mb-6">
        <div className="mb-1 flex justify-between text-xs text-slate-400">
          <span>Front of queue</span>
          <span>Back of queue</span>
        </div>
        <div className="relative h-8 overflow-hidden rounded-full bg-slate-700">
          {/* Filled portion */}
          <div
            className="absolute left-0 top-0 h-full bg-green-600 transition-all"
            style={{ width: `${percentThrough}%` }}
          />
          {/* Your position marker */}
          <div
            className="absolute top-0 h-full w-1 bg-yellow-400 transition-all"
            style={{ left: `${percentThrough}%` }}
          />
        </div>
        <div className="mt-1 flex justify-between text-xs">
          <span className="text-green-400">{totalInQueue - queuePosition} filled</span>
          <span className="text-yellow-400">You: #{queuePosition}</span>
          <span className="text-slate-400">{queuePosition} ahead</span>
        </div>
      </div>

      <div className="grid grid-cols-2 gap-4">
        <div className="rounded-lg bg-slate-900/50 p-4 text-center">
          <div className="text-xs text-slate-400">Contracts Ahead of You</div>
          <div className="text-2xl font-bold text-white">{queuePosition - 1}</div>
        </div>
        <div className="rounded-lg bg-slate-900/50 p-4 text-center">
          <div className="text-xs text-slate-400">Est. Time to Fill</div>
          <div className="text-2xl font-bold text-amber-400">~{timeToFill} min</div>
        </div>
      </div>

      <div className="mt-4 rounded-lg border border-blue-700/50 bg-blue-900/20 p-3">
        <div className="text-sm font-medium text-blue-400">UX Implication</div>
        <div className="text-sm text-slate-300">
          Queue position is one of the most important pieces of information for your client. They
          need to see it <strong>prominently</strong> and <strong>update in real-time</strong>.
          Consider: progress bar, position number, estimated time to fill.
        </div>
      </div>
    </div>
  );
}

// Order flow visualization
function OrderFlowVisualization() {
  const [orders, setOrders] = useState([
    { id: 1, type: 'retail', side: 'buy', price: 55, size: 10, time: '10:32:01' },
    { id: 2, type: 'retail', side: 'buy', price: 55, size: 25, time: '10:32:03' },
    { id: 3, type: 'mm', side: 'sell', price: 55, size: 25, time: '10:32:03', filled: true },
    { id: 4, type: 'retail', side: 'buy', price: 56, size: 50, time: '10:32:07' },
    { id: 5, type: 'retail', side: 'buy', price: 55, size: 15, time: '10:32:12' },
  ]);

  const addRetailOrder = () => {
    const newOrder = {
      id: orders.length + 1,
      type: 'retail' as const,
      side: 'buy' as const,
      price: 54 + Math.floor(Math.random() * 4),
      size: 5 + Math.floor(Math.random() * 45),
      time: new Date().toLocaleTimeString(),
    };
    setOrders([newOrder, ...orders.slice(0, 9)]);
  };

  return (
    <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-6">
      <h4 className="mb-4 text-lg font-semibold text-white">Reading the Tape (Order Flow)</h4>
      <p className="mb-4 text-sm text-slate-400">
        The &quot;tape&quot; shows recent trades. Market makers watch this to see retail flow coming
        in.
      </p>

      <button
        onClick={addRetailOrder}
        className="mb-4 rounded-lg bg-green-600 px-4 py-2 text-sm text-white hover:bg-green-700"
      >
        Simulate Retail Buy
      </button>

      <div className="overflow-hidden rounded-lg bg-slate-900">
        <div className="grid grid-cols-5 gap-2 bg-slate-800 p-2 text-xs font-medium text-slate-400">
          <div>Time</div>
          <div>Type</div>
          <div>Side</div>
          <div>Price</div>
          <div>Size</div>
        </div>
        <div className="divide-y divide-slate-800">
          {orders.map((order) => (
            <div
              key={order.id}
              className={`grid grid-cols-5 gap-2 p-2 text-sm ${
                order.filled ? 'bg-purple-900/20' : ''
              }`}
            >
              <div className="font-mono text-xs text-slate-500">{order.time}</div>
              <div>
                <span
                  className={`rounded px-1.5 py-0.5 text-xs ${
                    order.type === 'retail'
                      ? 'bg-blue-900/50 text-blue-400'
                      : 'bg-purple-900/50 text-purple-400'
                  }`}
                >
                  {order.type === 'retail' ? 'RETAIL' : 'MM'}
                </span>
              </div>
              <div className={order.side === 'buy' ? 'text-green-400' : 'text-red-400'}>
                {order.side.toUpperCase()}
              </div>
              <div className="font-mono text-white">{order.price}¢</div>
              <div className="text-slate-300">{order.size}</div>
            </div>
          ))}
        </div>
      </div>

      <div className="mt-4 rounded-lg border border-blue-700/50 bg-blue-900/20 p-3">
        <div className="text-sm font-medium text-blue-400">UX Implication</div>
        <div className="text-sm text-slate-300">
          Your client needs to see recent fills scrolling in real-time. Key info: size, price, and
          whether it was their order that got filled. Highlight their fills differently.
        </div>
      </div>
    </div>
  );
}

// Market maker workflow diagram
function WorkflowDiagram() {
  const [activePhase, setActivePhase] = useState<number>(0);

  const phases = [
    {
      name: '1. Analyze',
      duration: '~30 sec',
      description: 'Assess current market state and determine fair price',
      needs: [
        'Current bid/ask prices',
        'Recent price movement',
        'Queue depth at each level',
        'Time until game starts',
      ],
      decisions: ['What is my fair value estimate?', 'How wide should my spread be?'],
      uxGoal: 'Fast visual scan of market state - no clicking required',
    },
    {
      name: '2. Quote',
      duration: '~5 sec',
      description: 'Place orders in the NO book at desired prices',
      needs: [
        'Quick order entry (keyboard)',
        'Price input',
        'Size input',
        'Confirmation of order placement',
      ],
      decisions: ['What price?', 'What size?', 'Which market?'],
      uxGoal: 'Minimal keystrokes to post an order - hotkeys essential',
    },
    {
      name: '3. Monitor',
      duration: 'Ongoing',
      description: 'Watch queue position and incoming flow',
      needs: [
        'Queue position (real-time)',
        'Incoming order flow (tape)',
        'Price movement alerts',
        'Fill notifications',
      ],
      decisions: ['Should I stay in queue?', 'Should I adjust price?', 'Is flow drying up?'],
      uxGoal: 'Passive monitoring - info comes to them, not the other way around',
    },
    {
      name: '4. Adjust',
      duration: '~2 sec',
      description: 'Cancel and replace orders as market moves',
      needs: ['One-click cancel', 'Quick reprice', 'See new queue position immediately'],
      decisions: ['Cancel or hold?', 'New price?', 'Change size?'],
      uxGoal: 'Cancel-replace in under 2 seconds, ideally keyboard-only',
    },
    {
      name: '5. Fill',
      duration: 'Instant',
      description: 'Order gets matched with incoming retail',
      needs: ['Clear fill notification', 'Updated position', 'Updated P&L', 'New exposure info'],
      decisions: ['Re-quote immediately?', 'Adjust risk?'],
      uxGoal: 'Unmissable notification + automatic position update',
    },
  ];

  return (
    <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-6">
      <h4 className="mb-4 text-lg font-semibold text-white">The Market Maker Workflow Loop</h4>
      <p className="mb-6 text-sm text-slate-400">
        This is the minute-by-minute workflow your client repeats continuously. Click each phase to
        see what they need.
      </p>

      {/* Phase selector */}
      <div className="mb-6 flex gap-1 overflow-x-auto pb-2">
        {phases.map((phase, i) => (
          <button
            key={i}
            onClick={() => setActivePhase(i)}
            className={`whitespace-nowrap rounded-lg px-3 py-2 text-sm ${
              activePhase === i
                ? 'bg-purple-600 text-white'
                : 'bg-slate-700 text-slate-300 hover:bg-slate-600'
            }`}
          >
            {phase.name}
          </button>
        ))}
      </div>

      {/* Active phase details */}
      <div className="space-y-4">
        <div className="rounded-lg bg-slate-900/50 p-4">
          <div className="mb-2 flex items-start justify-between">
            <div className="text-lg font-semibold text-white">{phases[activePhase].name}</div>
            <div className="rounded bg-slate-800 px-2 py-1 text-xs text-slate-400">
              {phases[activePhase].duration}
            </div>
          </div>
          <p className="text-slate-300">{phases[activePhase].description}</p>
        </div>

        <div className="grid gap-4 md:grid-cols-2">
          <div className="rounded-lg bg-slate-900/50 p-4">
            <div className="mb-2 text-sm font-medium text-slate-400">Information Needed</div>
            <ul className="space-y-1">
              {phases[activePhase].needs.map((need, i) => (
                <li
                  key={i}
                  className="flex gap-2 text-sm text-slate-300"
                >
                  <span className="text-green-400">•</span>
                  {need}
                </li>
              ))}
            </ul>
          </div>

          <div className="rounded-lg bg-slate-900/50 p-4">
            <div className="mb-2 text-sm font-medium text-slate-400">Decisions Made</div>
            <ul className="space-y-1">
              {phases[activePhase].decisions.map((decision, i) => (
                <li
                  key={i}
                  className="flex gap-2 text-sm text-slate-300"
                >
                  <span className="text-amber-400">?</span>
                  {decision}
                </li>
              ))}
            </ul>
          </div>
        </div>

        <div className="rounded-lg border border-purple-700/50 bg-purple-900/20 p-4">
          <div className="mb-1 text-sm font-medium text-purple-400">UX Goal for This Phase</div>
          <div className="text-white">{phases[activePhase].uxGoal}</div>
        </div>
      </div>

      {/* Cycle indicator */}
      <div className="mt-6 flex items-center justify-center gap-2 text-sm text-slate-400">
        <span>Continuous loop:</span>
        {phases.map((_, i) => (
          <span
            key={i}
            className="flex items-center"
          >
            <span
              className={`h-2 w-2 rounded-full ${
                i === activePhase ? 'bg-purple-400' : 'bg-slate-600'
              }`}
            />
            {i < phases.length - 1 && <span className="mx-1">→</span>}
          </span>
        ))}
        <span>↺</span>
      </div>
    </div>
  );
}

// Information hierarchy for dashboard design
function InformationHierarchy() {
  const categories = [
    {
      level: 'Critical (Always Visible)',
      color: 'red',
      items: [
        { name: 'Current Position', why: 'Need to know exposure at all times' },
        { name: 'Queue Position', why: 'Core to their strategy - when will I get filled?' },
        { name: 'Active Orders', why: 'What am I offering to the market right now?' },
        { name: 'Current P&L', why: 'Am I making or losing money?' },
      ],
    },
    {
      level: 'High Priority (Primary View)',
      color: 'amber',
      items: [
        { name: 'NO Book Depth', why: 'Where is the competition? Where should I price?' },
        { name: 'Recent Fills (Tape)', why: "What's the flow like? Is retail active?" },
        { name: 'Bid/Ask Spread', why: 'My profit margin - is it worth quoting?' },
        { name: 'YES Book (for context)', why: "What are retail seeing? Where's demand?" },
      ],
    },
    {
      level: 'Medium Priority (Secondary View)',
      color: 'blue',
      items: [
        { name: 'Historical P&L', why: 'How am I doing overall?' },
        { name: 'Fill History', why: 'What trades have I done today?' },
        { name: 'Market-wide Activity', why: 'Is this market active or dead?' },
        { name: 'Time to Expiry', why: 'How long until this settles?' },
      ],
    },
    {
      level: 'Low Priority (Available on Demand)',
      color: 'slate',
      items: [
        { name: 'News/Analysis', why: "They don't bet on outcomes - less relevant" },
        { name: 'Account Balance', why: 'Check occasionally, not constantly' },
        { name: 'Settings/Config', why: 'Set once, rarely change' },
        { name: 'Historical Charts', why: 'Price history matters less for MM' },
      ],
    },
  ];

  const colorMap: Record<string, string> = {
    red: 'border-red-500 bg-red-900/20',
    amber: 'border-amber-500 bg-amber-900/20',
    blue: 'border-blue-500 bg-blue-900/20',
    slate: 'border-slate-500 bg-slate-800/50',
  };

  const textColorMap: Record<string, string> = {
    red: 'text-red-400',
    amber: 'text-amber-400',
    blue: 'text-blue-400',
    slate: 'text-slate-400',
  };

  return (
    <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-6">
      <h4 className="mb-4 text-lg font-semibold text-white">
        Information Hierarchy for Dashboard Design
      </h4>
      <p className="mb-6 text-sm text-slate-400">
        Not all information is equally important. This hierarchy should guide your layout decisions.
      </p>

      <div className="space-y-4">
        {categories.map((category) => (
          <div
            key={category.level}
            className={`rounded-lg border-2 p-4 ${colorMap[category.color]}`}
          >
            <div className={`mb-3 font-semibold ${textColorMap[category.color]}`}>
              {category.level}
            </div>
            <div className="grid gap-2 md:grid-cols-2">
              {category.items.map((item) => (
                <div
                  key={item.name}
                  className="flex gap-2 text-sm"
                >
                  <span className="font-medium text-white">{item.name}</span>
                  <span className="text-slate-500">—</span>
                  <span className="text-slate-400">{item.why}</span>
                </div>
              ))}
            </div>
          </div>
        ))}
      </div>
    </div>
  );
}

// UX principles summary
function UXPrinciplesSummary() {
  const principles = [
    {
      principle: 'Speed Over Polish',
      description:
        'Every millisecond counts. A fast, ugly interface beats a slow, beautiful one. Keyboard shortcuts, minimal clicks, instant feedback.',
      doThis: [
        'Hotkeys for all common actions',
        'Instant order placement',
        'No confirmation dialogs for routine actions',
      ],
      notThis: ['Fancy animations', 'Multi-step wizards', 'Mouse-dependent workflows'],
    },
    {
      principle: 'Information Density',
      description:
        "Market makers need lots of data visible simultaneously. Don't spread information across multiple screens or tabs.",
      doThis: [
        'Single-screen dashboards',
        'Compact data tables',
        'Multiple data points visible at once',
      ],
      notThis: [
        'One metric per card',
        'Hidden behind clicks',
        'Sparse layouts with lots of whitespace',
      ],
    },
    {
      principle: 'Real-Time Everything',
      description:
        'Stale data is dangerous. Prices, positions, queue status, and P&L must update instantly via WebSocket.',
      doThis: [
        'WebSocket for all market data',
        'Sub-second updates',
        'Visual indicators when data updates',
      ],
      notThis: [
        'Polling every 5 seconds',
        'Manual refresh buttons',
        'Cached data without timestamps',
      ],
    },
    {
      principle: 'Error Prevention Over Error Handling',
      description:
        'Fat-finger mistakes can be costly. Prevent errors rather than asking for confirmation.',
      doThis: ['Position limits', 'Price sanity checks', 'Clear visual feedback before submit'],
      notThis: [
        'Confirmation dialogs (too slow)',
        'Error messages after the fact',
        'Letting any order through',
      ],
    },
    {
      principle: 'Glanceable Status',
      description:
        'Traders glance at their screen hundreds of times. Key status should be readable in under 1 second.',
      doThis: [
        'Color-coded P&L (green/red)',
        'Large position numbers',
        'Clear queue position indicator',
      ],
      notThis: ['Small text', 'Ambiguous states', 'Numbers without context'],
    },
  ];

  return (
    <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-6">
      <h4 className="mb-4 text-lg font-semibold text-white">
        Core UX Principles for Market Maker Software
      </h4>

      <div className="space-y-6">
        {principles.map((p) => (
          <div
            key={p.principle}
            className="rounded-lg bg-slate-900/50 p-4"
          >
            <div className="mb-2 text-lg font-semibold text-white">{p.principle}</div>
            <p className="mb-4 text-sm text-slate-300">{p.description}</p>

            <div className="grid gap-4 md:grid-cols-2">
              <div>
                <div className="mb-2 text-sm font-medium text-green-400">✓ Do This</div>
                <ul className="space-y-1">
                  {p.doThis.map((item, i) => (
                    <li
                      key={i}
                      className="text-sm text-slate-300"
                    >
                      • {item}
                    </li>
                  ))}
                </ul>
              </div>
              <div>
                <div className="mb-2 text-sm font-medium text-red-400">✗ Not This</div>
                <ul className="space-y-1">
                  {p.notThis.map((item, i) => (
                    <li
                      key={i}
                      className="text-sm text-slate-300"
                    >
                      • {item}
                    </li>
                  ))}
                </ul>
              </div>
            </div>
          </div>
        ))}
      </div>
    </div>
  );
}

export function BuildingForMarketMakersLesson() {
  return (
    <div className="space-y-8">
      {/* Introduction */}
      <section>
        <SectionHeader
          title="Building for Market Makers"
          subtitle="A product guide for understanding your users"
        />
        <LessonRow
          aside={
            <ConstraintBlock
              title="What You'll Learn"
              items={[
                "What market making is (and isn't)",
                'Why they use the NO side',
                'Their minute-by-minute workflow',
                'Information hierarchy for dashboards',
                'Core UX principles for trading software',
              ]}
            />
          }
        >
          <div className="mb-4 rounded-lg border border-purple-700/50 bg-purple-900/20 p-4">
            <p className="text-sm text-purple-300">
              <strong>Note:</strong> This guide is different from the other lessons. It&apos;s not
              about becoming a trader—it&apos;s about understanding traders deeply enough to build
              the right software for them.
            </p>
          </div>
          <p className="leading-relaxed text-slate-300">
            Your client is a <strong className="text-white">market maker</strong>. This is a
            specific type of trader with specific needs that are very different from a casual bettor
            or even a professional sports handicapper. This guide will help you understand:
          </p>
          <ul className="mt-4 space-y-2 text-slate-300">
            <li>• What market making actually is (and isn&apos;t)</li>
            <li>• Why they care about the NO side specifically</li>
            <li>• Their minute-by-minute workflow</li>
            <li>• What information they need and when</li>
            <li>• UX principles that will make your software valuable to them</li>
          </ul>
        </LessonRow>
      </section>

      {/* What is Market Making */}
      <section>
        <SectionHeader
          title="What Market Making Actually Is"
          subtitle="The fundamental difference from betting"
        />
        <LessonRow
          aside={
            <InsightBlock title="Key Mental Model">
              Market makers are like car dealers—they don't care which team is "better." They profit
              from the spread between buy and sell, not from predicting outcomes.
            </InsightBlock>
          }
        >
          <p className="leading-relaxed text-slate-300">
            A market maker is <strong className="text-white">not betting on outcomes</strong>. They
            don&apos;t care if the Lakers win or lose. Instead, they&apos;re providing a service:{' '}
            <strong className="text-white">liquidity</strong>. They make money by being the
            counterparty to other people&apos;s bets and capturing the spread.
          </p>

          <div className="my-6 rounded-lg border border-slate-700 bg-slate-800/50 p-6">
            <h4 className="mb-4 text-lg font-semibold text-white">Two Types of Traders</h4>
            <div className="grid gap-6 md:grid-cols-2">
              <div className="rounded-lg border border-blue-700/50 bg-blue-900/20 p-4">
                <div className="mb-2 font-semibold text-blue-400">Speculator (Bettor)</div>
                <ul className="space-y-1 text-sm text-slate-300">
                  <li>• Has an opinion on outcomes</li>
                  <li>• &quot;Lakers will win, so I bet Lakers&quot;</li>
                  <li>• Profits when their prediction is right</li>
                  <li>• Takes one side of a bet</li>
                </ul>
              </div>
              <div className="rounded-lg border border-purple-700/50 bg-purple-900/20 p-4">
                <div className="mb-2 font-semibold text-purple-400">Market Maker (Your Client)</div>
                <ul className="space-y-1 text-sm text-slate-300">
                  <li>• No opinion on outcomes</li>
                  <li>• &quot;I&apos;ll sell to whoever wants to buy&quot;</li>
                  <li>• Profits from spread between buy/sell</li>
                  <li>• Takes the other side of retail bets</li>
                </ul>
              </div>
            </div>
          </div>

          <p className="leading-relaxed text-slate-300">
            <strong className="text-white">Analogy:</strong> Think of a market maker like a car
            dealer. The dealer doesn&apos;t care if Toyotas are &quot;better&quot; than Hondas. They
            buy cars at one price and sell at a slightly higher price. The spread is their profit.
            Your client does the same thing with contracts.
          </p>
        </LessonRow>
      </section>

      {/* The Dual Market Structure */}
      <section>
        <SectionHeader
          title="The Dual Market Structure"
          subtitle="Why your client cares about the NO side"
        />
        <LessonRow
          aside={
            <TipBlock title="UX Implication">
              Your UI should default to showing NO books, not YES books. That's where your client
              lives.
            </TipBlock>
          }
        >
          <p className="mb-6 leading-relaxed text-slate-300">
            This is the key concept your client was explaining. On Kalshi, every game has multiple
            order books that are economically equivalent but behaviorally different. Understanding
            this is crucial for building the right UX.
          </p>
          <DualMarketVisualization />
        </LessonRow>
      </section>

      {/* Queue Position */}
      <section>
        <SectionHeader
          title="Queue Position: The Core Mechanic"
          subtitle="Why this is the most important number on your client's screen"
        />
        <LessonRow
          aside={
            <WarningBlock title="Don't Hide This">
              Queue position should NEVER be buried in a submenu or require a click to see. Make it
              as prominent as their P&L.
            </WarningBlock>
          }
        >
          <p className="mb-6 leading-relaxed text-slate-300">
            When multiple orders are at the same price, they&apos;re filled first-come-first-served.
            Your client&apos;s position in this queue determines when (and if) they get filled. This
            is not just &quot;nice to know&quot;—it&apos;s central to their entire strategy.
          </p>
          <QueueVisualization />
          <div className="mt-6 rounded-lg border border-amber-700/50 bg-amber-900/20 p-4">
            <h4 className="mb-2 font-semibold text-amber-400">
              Why Queue Position Matters So Much
            </h4>
            <ul className="space-y-2 text-sm text-slate-300">
              <li>
                <strong>It determines profitability:</strong> First in queue at a good price beats
                last in queue at a great price.
              </li>
              <li>
                <strong>It drives decisions:</strong> If they&apos;re far back in queue, they might
                cancel and reprice. If they&apos;re near front, they stay.
              </li>
              <li>
                <strong>It&apos;s constantly changing:</strong> As orders get filled or cancelled,
                their position shifts. Real-time updates are essential.
              </li>
            </ul>
          </div>
        </LessonRow>
      </section>

      {/* Order Flow */}
      <section>
        <SectionHeader
          title="Reading Order Flow"
          subtitle="The tape tells them if their strategy is working"
        />
        <LessonRow
          aside={
            <TipBlock title="The Tape">
              "Tape" is old-school trading jargon from when trades printed on ticker tape. It means
              the real-time feed of executed trades.
            </TipBlock>
          }
        >
          <p className="mb-6 leading-relaxed text-slate-300">
            Market makers watch the &quot;tape&quot; (stream of recent trades) to understand
            what&apos;s happening in the market. Is retail buying? Is flow picking up or dying down?
            This informs their quoting decisions.
          </p>
          <OrderFlowVisualization />
        </LessonRow>
      </section>

      {/* The Workflow */}
      <section>
        <SectionHeader
          title="The Minute-by-Minute Workflow"
          subtitle="Understanding what your client actually does"
        />
        <LessonRow
          aside={
            <InsightBlock title="Design for the Loop">
              Market making is repetitive. Design for the 1000th iteration, not the first. Reduce
              friction in the core loop.
            </InsightBlock>
          }
        >
          <p className="mb-6 leading-relaxed text-slate-300">
            Market making is a continuous loop of analyze → quote → monitor → adjust → fill.
            Understanding each phase tells you what information and controls they need at each
            moment.
          </p>
          <WorkflowDiagram />
        </LessonRow>
      </section>

      {/* Information Hierarchy */}
      <section>
        <SectionHeader
          title="Information Hierarchy"
          subtitle="What goes where on the screen"
        />
        <LessonRow
          aside={
            <WarningBlock title="Common Mistake">
              Don't give equal visual weight to everything. News and analysis are LOW priority for
              MMs—they don't bet on outcomes. Position and queue are HIGH priority.
            </WarningBlock>
          }
        >
          <p className="mb-6 leading-relaxed text-slate-300">
            Not all information is equally important. This hierarchy should directly inform your
            layout decisions—what gets prime screen real estate vs. what gets tucked away.
          </p>
          <InformationHierarchy />
        </LessonRow>
      </section>

      {/* UX Principles */}
      <section>
        <SectionHeader
          title="UX Principles for Market Maker Software"
          subtitle="What makes trading software good (and bad)"
        />
        <LessonRow
          aside={
            <InsightBlock title="Inverted Priorities">
              Consumer app UX: discoverability, beauty, onboarding. Trading UX: speed, density,
              keyboard shortcuts. They're almost opposites.
            </InsightBlock>
          }
        >
          <p className="mb-6 leading-relaxed text-slate-300">
            Trading software UX is different from consumer app UX. The priorities are inverted—
            speed and density matter more than aesthetics and discoverability.
          </p>
          <UXPrinciplesSummary />
        </LessonRow>
      </section>

      {/* Concrete Feature Recommendations */}
      <section>
        <SectionHeader
          title="Concrete Feature Recommendations"
          subtitle="What to build"
        />
        <LessonRow
          aside={
            <TipBlock title="MVP First">
              Ship the "Must Have" features first and get feedback. MMs will tell you exactly what's
              missing—they're vocal users.
            </TipBlock>
          }
        >
          <div className="space-y-4">
            <div className="rounded-lg border border-green-700/50 bg-green-900/20 p-4">
              <h4 className="mb-2 font-semibold text-green-400">Must Have (MVP)</h4>
              <ul className="space-y-1 text-sm text-slate-300">
                <li>
                  • <strong>NO Book View:</strong> Show NO order book for selected market, with
                  depth
                </li>
                <li>
                  • <strong>Queue Position:</strong> Prominent display of position in queue for
                  active orders
                </li>
                <li>
                  • <strong>Quick Order Entry:</strong> Keyboard-driven order placement
                </li>
                <li>
                  • <strong>Position Display:</strong> Current position and P&L always visible
                </li>
                <li>
                  • <strong>Order Management:</strong> List of active orders with one-click cancel
                </li>
                <li>
                  • <strong>Fill Notifications:</strong> Unmissable notification when orders fill
                </li>
              </ul>
            </div>

            <div className="rounded-lg border border-blue-700/50 bg-blue-900/20 p-4">
              <h4 className="mb-2 font-semibold text-blue-400">Should Have (v1.1)</h4>
              <ul className="space-y-1 text-sm text-slate-300">
                <li>
                  • <strong>Tape/Order Flow:</strong> Streaming recent fills
                </li>
                <li>
                  • <strong>Hotkey System:</strong> Configurable keyboard shortcuts
                </li>
                <li>
                  • <strong>Cancel-Replace:</strong> Modify order without losing queue (if possible)
                </li>
                <li>
                  • <strong>Multi-Market View:</strong> Monitor multiple games simultaneously
                </li>
                <li>
                  • <strong>Sound Alerts:</strong> Audio notifications for fills
                </li>
              </ul>
            </div>

            <div className="rounded-lg border border-slate-600 bg-slate-700/50 p-4">
              <h4 className="mb-2 font-semibold text-slate-300">Nice to Have (Later)</h4>
              <ul className="space-y-1 text-sm text-slate-400">
                <li>
                  • <strong>Analytics Dashboard:</strong> Historical performance, fill rates
                </li>
                <li>
                  • <strong>Automation:</strong> Auto-requote when price moves
                </li>
                <li>
                  • <strong>Inventory Management:</strong> Track exposure across markets
                </li>
                <li>
                  • <strong>API Access:</strong> For algorithmic trading
                </li>
              </ul>
            </div>
          </div>
        </LessonRow>
      </section>

      {/* Summary */}
      <section>
        <SectionHeader
          title="Summary: What Your Client Needs"
          subtitle="The key takeaways for building their software"
        />
        <LessonRow
          aside={
            <InsightBlock title="The Core Insight">
              Your client is running a business, not gambling. Design for efficiency and
              reliability, not excitement.
            </InsightBlock>
          }
        >
          <div className="rounded-lg border border-purple-800/50 bg-gradient-to-r from-purple-900/30 to-blue-900/30 p-6">
            <ul className="space-y-3 text-slate-300">
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>
                  <strong className="text-white">They&apos;re market makers, not bettors.</strong>{' '}
                  They don&apos;t care who wins—they profit from spread, not predictions.
                </span>
              </li>
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>
                  <strong className="text-white">They work exclusively on the NO side.</strong> This
                  is where they intercept retail flow. Your UI should default to NO books.
                </span>
              </li>
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>
                  <strong className="text-white">Queue position is critical.</strong> This single
                  number drives most of their decisions. Make it prominent and real-time.
                </span>
              </li>
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>
                  <strong className="text-white">Speed beats beauty.</strong> They need keyboard
                  shortcuts, minimal clicks, and instant feedback. Skip the animations.
                </span>
              </li>
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>
                  <strong className="text-white">Information density matters.</strong> They need
                  lots of data visible simultaneously. Single-screen dashboards, not multi-page
                  flows.
                </span>
              </li>
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>
                  <strong className="text-white">Real-time is non-negotiable.</strong> Stale data is
                  dangerous. WebSocket everything.
                </span>
              </li>
            </ul>
          </div>
        </LessonRow>
      </section>
    </div>
  );
}
