/**
 * Lesson: Understanding Buy/Sell and YES/NO
 *
 * Clarifies the two independent dimensions of Kalshi orders:
 * - Side (YES or NO) - which contract type
 * - Action (Buy or Sell) - what you're doing with it
 *
 * This is THE foundational confusion for new traders.
 */

import { ArrowLeftRight, ShoppingCart, TrendingUp, TrendingDown, Layers, Code } from 'lucide-react';
import { getGuide, getNextGuide } from '@/data/guides';
import { LessonLayout, LessonRow, LessonHeader, SectionHeader } from '../LessonLayout';
import {
  ObjectiveBlock,
  ConstraintBlock,
  TipBlock,
  InsightBlock,
  WarningBlock,
  ConceptBlock,
  ExampleBox,
  NextStepBlock,
  Term,
} from '../blocks';

export function BuySellYesNoLesson() {
  const guide = getGuide('buy-sell-yes-no')!;
  const nextGuide = getNextGuide('buy-sell-yes-no');

  return (
    <LessonLayout>
      {/* Header */}
      <LessonRow>
        <LessonHeader
          title={guide.title}
          description={guide.description}
          duration="15 min"
          category="Fundamentals"
        />
      </LessonRow>

      {/* Objective */}
      <LessonRow
        aside={
          <ConstraintBlock
            title="What We'll Cover"
            items={[
              'The two dimensions: Side and Action',
              'All four order combinations explained',
              'Can you sell without owning? (Yes!)',
              'How this maps to the Kalshi API',
            ]}
          />
        }
      >
        <ObjectiveBlock>
          By the end of this lesson, you'll understand that Kalshi orders have two independent
          dimensions (Side and Action), what each of the four combinations means, and how "Sell"
          works even when you don't own anything.
        </ObjectiveBlock>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 1: THE CONFUSION */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="You're Not Alone">
            This confusion is universal. Even experienced traders from other markets stumble here
            because Kalshi's model is different from stocks or options.
          </TipBlock>
        }
      >
        <SectionHeader
          title="The Source of Confusion"
          subtitle="Two dimensions that seem like one"
        />

        <p className="text-muted-foreground leading-relaxed">
          When you look at Kalshi's order interface, you see two choices that seem related but are
          actually <em>completely independent</em>:
        </p>

        <div className="mt-6 grid gap-4 sm:grid-cols-2">
          <div className="bg-card rounded-lg border p-4">
            <h4 className="mb-2 flex items-center gap-2 font-semibold">
              <div className="flex h-8 w-8 items-center justify-center rounded bg-emerald-500/20">
                <ShoppingCart className="h-4 w-4 text-emerald-400" />
              </div>
              Action: Buy or Sell
            </h4>
            <p className="text-muted-foreground text-sm">
              Are you <strong>acquiring</strong> contracts or <strong>disposing</strong> of them?
            </p>
          </div>
          <div className="bg-card rounded-lg border p-4">
            <h4 className="mb-2 flex items-center gap-2 font-semibold">
              <div className="flex h-8 w-8 items-center justify-center rounded bg-violet-500/20">
                <Layers className="h-4 w-4 text-violet-400" />
              </div>
              Side: YES or NO
            </h4>
            <p className="text-muted-foreground text-sm">
              <strong>Which contract</strong> are you trading? The YES contract or the NO contract?
            </p>
          </div>
        </div>

        <div className="from-primary/10 via-primary/5 border-primary/30 mt-6 rounded-lg border bg-gradient-to-br to-transparent p-4">
          <p className="text-foreground font-medium">
            The key insight: These are two separate choices.
          </p>
          <p className="text-muted-foreground mt-2">
            You pick an Action (Buy or Sell) AND you pick a Side (YES or NO). This creates four
            possible combinations, each with different meaning.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 2: WHAT ARE YES AND NO CONTRACTS? */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <ConceptBlock title="Binary Contracts">
            Every Kalshi market has exactly two contract types: YES and NO. They're like two
            different items you can buy or sell. One pays out if the event happens, the other pays
            out if it doesn't.
          </ConceptBlock>
        }
      >
        <SectionHeader
          title="YES and NO Are Two Different Things"
          subtitle="Think of them as two separate products"
        />

        <p className="text-muted-foreground leading-relaxed">
          Imagine a store that sells two products related to the same event:
        </p>

        <div className="mt-6 space-y-4">
          <Term term="YES Contract">
            A ticket that pays $1.00 if the event <strong>happens</strong>. If "Will it rain
            tomorrow?" resolves YES (it rained), your YES contract pays out. If it doesn't rain,
            your YES contract is worth $0.
          </Term>

          <Term term="NO Contract">
            A ticket that pays $1.00 if the event <strong>doesn't happen</strong>. If "Will it rain
            tomorrow?" resolves NO (no rain), your NO contract pays out. If it does rain, your NO
            contract is worth $0.
          </Term>
        </div>

        <ExampleBox title="The Rain Market">
          <div className="space-y-4">
            <p className="text-sm">
              <strong>Market:</strong> "Will it rain in NYC tomorrow?"
            </p>
            <div className="grid gap-3 sm:grid-cols-2">
              <div className="rounded border border-emerald-500/20 bg-emerald-500/10 p-3">
                <div className="font-medium text-emerald-400">YES Contract @ 70¢</div>
                <div className="text-muted-foreground mt-1 text-xs">
                  Pay 70¢ now → Get $1.00 if it rains
                </div>
              </div>
              <div className="rounded border border-rose-500/20 bg-rose-500/10 p-3">
                <div className="font-medium text-rose-400">NO Contract @ 30¢</div>
                <div className="text-muted-foreground mt-1 text-xs">
                  Pay 30¢ now → Get $1.00 if it doesn't rain
                </div>
              </div>
            </div>
            <p className="text-muted-foreground text-xs">
              Note: YES price + NO price always equals ~$1.00 (minus the spread)
            </p>
          </div>
        </ExampleBox>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 3: BUY VS SELL - THE ACTION */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="It's Like Any Market">
            Think of a farmer's market. You can BUY apples (acquire them) or SELL apples (give them
            away for money). The apple is the "side" - the action is what you do with it.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="Buy vs Sell: The Action"
          subtitle="What are you doing with the contract?"
        />

        <p className="text-muted-foreground leading-relaxed">
          Once you know which contract you're interested in (YES or NO), you decide what to do with
          it:
        </p>

        <div className="mt-6 space-y-4">
          <Term term="Buy">
            You are <strong>acquiring</strong> contracts. You pay money now to own contracts that
            might pay out later. You're adding to your position.
          </Term>

          <Term term="Sell">
            You are <strong>disposing</strong> of contracts. You receive money now by giving up
            contracts. This could mean closing a position you have, OR opening a new short position.
          </Term>
        </div>

        <div className="mt-6 rounded-lg border border-amber-500/30 bg-amber-500/10 p-4">
          <h4 className="mb-2 font-semibold text-amber-400">
            Wait - can I sell something I don't own?
          </h4>
          <p className="text-muted-foreground text-sm">
            <strong>Yes!</strong> This is the part that confuses people. On Kalshi, "Sell" doesn't
            require you to already own the contracts. If you sell contracts you don't have, you're
            opening a <strong>short position</strong>.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 4: THE FOUR COMBINATIONS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="Don't Conflate Them">
            "Buy YES" is not the opposite of "Buy NO". They're both buying - just different
            contracts. The opposite of "Buy YES" is "Sell YES".
          </WarningBlock>
        }
      >
        <SectionHeader
          title="The Four Combinations"
          subtitle="Each one has a distinct meaning"
        />

        <p className="text-muted-foreground mb-6 leading-relaxed">
          Two actions × two sides = four possible orders. Let's use a concrete example:
          <br />
          <strong>Market: "Will the Lakers win tonight?"</strong>
        </p>

        <div className="space-y-4">
          {/* BUY YES */}
          <div className="bg-card overflow-hidden rounded-lg border">
            <div className="border-b border-emerald-500/20 bg-emerald-500/10 p-4">
              <div className="flex items-center gap-3">
                <div className="flex h-10 w-10 items-center justify-center rounded-full bg-emerald-500/20">
                  <TrendingUp className="h-5 w-5 text-emerald-400" />
                </div>
                <div>
                  <h4 className="font-semibold text-emerald-400">Buy YES</h4>
                  <p className="text-muted-foreground text-xs">You're bullish on the event</p>
                </div>
              </div>
            </div>
            <div className="p-4">
              <p className="text-muted-foreground text-sm">
                <strong>What it means:</strong> You pay money to acquire YES contracts. You want the
                Lakers to win. If they do, each contract pays $1. If they lose, you lose your
                investment.
              </p>
              <div className="bg-muted/50 mt-3 rounded p-2 font-mono text-xs">
                Cost: 65¢ × 10 contracts = $6.50
                <br />
                If Lakers win: $10.00 payout → $3.50 profit
                <br />
                If Lakers lose: $0 payout → $6.50 loss
              </div>
            </div>
          </div>

          {/* SELL YES */}
          <div className="bg-card overflow-hidden rounded-lg border">
            <div className="border-b border-rose-500/20 bg-rose-500/10 p-4">
              <div className="flex items-center gap-3">
                <div className="flex h-10 w-10 items-center justify-center rounded-full bg-rose-500/20">
                  <TrendingDown className="h-5 w-5 text-rose-400" />
                </div>
                <div>
                  <h4 className="font-semibold text-rose-400">Sell YES</h4>
                  <p className="text-muted-foreground text-xs">You're bearish on the event</p>
                </div>
              </div>
            </div>
            <div className="p-4">
              <p className="text-muted-foreground text-sm">
                <strong>What it means:</strong> You receive money by selling YES contracts. You
                don't think the Lakers will win. If you already own YES contracts, this closes your
                position. If you don't own any, this opens a <em>short</em> position.
              </p>
              <div className="bg-muted/50 mt-3 rounded p-2 font-mono text-xs">
                Receive: 65¢ × 10 contracts = $6.50
                <br />
                If Lakers win: You owe $10.00 → $3.50 loss
                <br />
                If Lakers lose: You owe $0 → $6.50 profit
              </div>
            </div>
          </div>

          {/* BUY NO */}
          <div className="bg-card overflow-hidden rounded-lg border">
            <div className="border-b border-rose-500/20 bg-rose-500/10 p-4">
              <div className="flex items-center gap-3">
                <div className="flex h-10 w-10 items-center justify-center rounded-full bg-rose-500/20">
                  <TrendingDown className="h-5 w-5 text-rose-400" />
                </div>
                <div>
                  <h4 className="font-semibold text-rose-400">Buy NO</h4>
                  <p className="text-muted-foreground text-xs">You're bearish on the event</p>
                </div>
              </div>
            </div>
            <div className="p-4">
              <p className="text-muted-foreground text-sm">
                <strong>What it means:</strong> You pay money to acquire NO contracts. You don't
                think the Lakers will win. If they lose, each contract pays $1. If they win, you
                lose your investment.
              </p>
              <div className="bg-muted/50 mt-3 rounded p-2 font-mono text-xs">
                Cost: 35¢ × 10 contracts = $3.50
                <br />
                If Lakers lose: $10.00 payout → $6.50 profit
                <br />
                If Lakers win: $0 payout → $3.50 loss
              </div>
            </div>
          </div>

          {/* SELL NO */}
          <div className="bg-card overflow-hidden rounded-lg border">
            <div className="border-b border-emerald-500/20 bg-emerald-500/10 p-4">
              <div className="flex items-center gap-3">
                <div className="flex h-10 w-10 items-center justify-center rounded-full bg-emerald-500/20">
                  <TrendingUp className="h-5 w-5 text-emerald-400" />
                </div>
                <div>
                  <h4 className="font-semibold text-emerald-400">Sell NO</h4>
                  <p className="text-muted-foreground text-xs">You're bullish on the event</p>
                </div>
              </div>
            </div>
            <div className="p-4">
              <p className="text-muted-foreground text-sm">
                <strong>What it means:</strong> You receive money by selling NO contracts. You think
                the Lakers WILL win. If you already own NO contracts, this closes your position. If
                you don't own any, this opens a short position on NO.
              </p>
              <div className="bg-muted/50 mt-3 rounded p-2 font-mono text-xs">
                Receive: 35¢ × 10 contracts = $3.50
                <br />
                If Lakers win: You owe $0 → $3.50 profit
                <br />
                If Lakers lose: You owe $10.00 → $6.50 loss
              </div>
            </div>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 5: EQUIVALENT POSITIONS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="Why Both Exist?">
            If "Buy YES" and "Sell NO" are economically similar, why have both? Because they trade
            in different order books with potentially different prices, spreads, and liquidity.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="Equivalent Positions"
          subtitle="Different paths to the same outcome"
        />

        <p className="text-muted-foreground leading-relaxed">
          Notice something interesting: "Buy YES" and "Sell NO" both profit if the event happens.
          "Sell YES" and "Buy NO" both profit if it doesn't. These pairs are{' '}
          <em>economically equivalent</em> but exist as separate order types:
        </p>

        <div className="mt-6 overflow-x-auto">
          <table className="w-full text-sm">
            <thead>
              <tr className="border-b">
                <th className="py-3 text-left font-medium">If you're bullish (event happens)</th>
                <th className="py-3 text-left font-medium">
                  If you're bearish (event doesn't happen)
                </th>
              </tr>
            </thead>
            <tbody>
              <tr>
                <td className="py-3">
                  <span className="rounded bg-emerald-500/20 px-2 py-1 text-xs font-medium text-emerald-400">
                    Buy YES
                  </span>
                  <span className="text-muted-foreground mx-2">or</span>
                  <span className="rounded bg-emerald-500/20 px-2 py-1 text-xs font-medium text-emerald-400">
                    Sell NO
                  </span>
                </td>
                <td className="py-3">
                  <span className="rounded bg-rose-500/20 px-2 py-1 text-xs font-medium text-rose-400">
                    Buy NO
                  </span>
                  <span className="text-muted-foreground mx-2">or</span>
                  <span className="rounded bg-rose-500/20 px-2 py-1 text-xs font-medium text-rose-400">
                    Sell YES
                  </span>
                </td>
              </tr>
            </tbody>
          </table>
        </div>

        <div className="bg-muted/50 mt-4 rounded-lg p-3">
          <p className="text-muted-foreground text-sm">
            <strong>Why does this matter?</strong> Market makers exploit the fact that these
            equivalent positions trade in separate order books. The YES book might have better
            prices than the NO book, or vice versa.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 6: SHORT SELLING EXPLAINED */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="Short = Obligation">
            When you sell contracts you don't own, you're creating an obligation. If the contract
            pays out, YOU have to pay. This is real risk.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="Short Selling: Selling What You Don't Own"
          subtitle="This is not like selling on eBay"
        />

        <p className="text-muted-foreground leading-relaxed">
          On Kalshi, "Sell" doesn't check if you own contracts first. If you sell 10 YES contracts
          and you own 0, you now have a position of <strong>-10 YES contracts</strong>. This is
          called being "short."
        </p>

        <ExampleBox title="Going Short">
          <div className="space-y-4">
            <p className="text-sm">
              <strong>Scenario:</strong> You think the Lakers will lose. YES contracts are trading
              at 65¢.
            </p>

            <div className="space-y-2">
              <div className="flex items-center gap-2 text-sm">
                <span className="text-muted-foreground w-24">Before:</span>
                <span>You own 0 YES contracts</span>
              </div>
              <div className="flex items-center gap-2 text-sm">
                <span className="text-muted-foreground w-24">Action:</span>
                <span className="text-rose-400">Sell 10 YES @ 65¢</span>
              </div>
              <div className="flex items-center gap-2 text-sm">
                <span className="text-muted-foreground w-24">After:</span>
                <span>
                  You own <strong>-10</strong> YES contracts, received $6.50
                </span>
              </div>
            </div>

            <div className="border-t pt-3">
              <p className="text-muted-foreground text-sm">
                <strong>If Lakers win:</strong> Each YES contract pays $1. You're short 10, so you
                owe $10. Net: -$3.50
              </p>
              <p className="text-muted-foreground mt-1 text-sm">
                <strong>If Lakers lose:</strong> YES contracts pay $0. You owe nothing. Net: +$6.50
              </p>
            </div>
          </div>
        </ExampleBox>

        <div className="bg-card mt-6 rounded-lg border p-4">
          <h4 className="mb-3 font-semibold">Long vs Short Positions</h4>
          <div className="grid gap-4 sm:grid-cols-2">
            <div>
              <div className="mb-2 text-sm font-medium text-emerald-400">
                Long (Positive Position)
              </div>
              <ul className="text-muted-foreground space-y-1 text-sm">
                <li>• You OWN contracts</li>
                <li>• Created by BUYING</li>
                <li>• You paid money upfront</li>
                <li>• You receive payout if contract wins</li>
              </ul>
            </div>
            <div>
              <div className="mb-2 text-sm font-medium text-rose-400">
                Short (Negative Position)
              </div>
              <ul className="text-muted-foreground space-y-1 text-sm">
                <li>• You OWE contracts</li>
                <li>• Created by SELLING (without owning)</li>
                <li>• You received money upfront</li>
                <li>• You pay out if contract wins</li>
              </ul>
            </div>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 7: HOW IT MAPS TO THE API */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="For Developers">
            If you're building trading software, understanding these API fields is essential. The UI
            abstracts this, but the API requires you to specify both fields explicitly.
          </TipBlock>
        }
      >
        <SectionHeader
          title="How This Maps to the Kalshi API"
          subtitle="The technical implementation"
        />

        <p className="text-muted-foreground leading-relaxed">
          When you place an order through the Kalshi API, you specify two separate fields:
        </p>

        <div className="mt-6 overflow-x-auto rounded-lg border bg-zinc-900 p-4 font-mono text-sm">
          <div className="text-muted-foreground mb-3 flex items-center gap-2">
            <Code className="h-4 w-4" />
            <span>Order Request Body</span>
          </div>
          <pre className="text-emerald-400">
            {`{
  "ticker": "LAKERS-WIN-2024",
  "side": "yes",      // or "no"
  "action": "buy",    // or "sell"
  "type": "limit",
  "count": 10,
  "price": 65
}`}
          </pre>
        </div>

        <div className="mt-6 space-y-3">
          <div className="bg-muted/30 flex items-start gap-3 rounded-lg p-3">
            <code className="bg-primary/20 text-primary rounded px-2 py-0.5 font-mono text-xs">
              side
            </code>
            <p className="text-muted-foreground text-sm">
              Which contract type: <code className="text-foreground">"yes"</code> or{' '}
              <code className="text-foreground">"no"</code>
            </p>
          </div>
          <div className="bg-muted/30 flex items-start gap-3 rounded-lg p-3">
            <code className="bg-primary/20 text-primary rounded px-2 py-0.5 font-mono text-xs">
              action
            </code>
            <p className="text-muted-foreground text-sm">
              What you're doing: <code className="text-foreground">"buy"</code> or{' '}
              <code className="text-foreground">"sell"</code>
            </p>
          </div>
        </div>

        <div className="mt-6 overflow-x-auto">
          <table className="w-full text-sm">
            <thead>
              <tr className="border-b">
                <th className="py-3 text-left font-medium">UI Action</th>
                <th className="py-3 text-left font-medium">API side</th>
                <th className="py-3 text-left font-medium">API action</th>
              </tr>
            </thead>
            <tbody className="divide-y">
              <tr>
                <td className="py-3">Buy YES</td>
                <td className="py-3">
                  <code className="text-emerald-400">"yes"</code>
                </td>
                <td className="py-3">
                  <code className="text-emerald-400">"buy"</code>
                </td>
              </tr>
              <tr>
                <td className="py-3">Sell YES</td>
                <td className="py-3">
                  <code className="text-rose-400">"yes"</code>
                </td>
                <td className="py-3">
                  <code className="text-rose-400">"sell"</code>
                </td>
              </tr>
              <tr>
                <td className="py-3">Buy NO</td>
                <td className="py-3">
                  <code className="text-rose-400">"no"</code>
                </td>
                <td className="py-3">
                  <code className="text-emerald-400">"buy"</code>
                </td>
              </tr>
              <tr>
                <td className="py-3">Sell NO</td>
                <td className="py-3">
                  <code className="text-emerald-400">"no"</code>
                </td>
                <td className="py-3">
                  <code className="text-rose-400">"sell"</code>
                </td>
              </tr>
            </tbody>
          </table>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 8: COMMON CONFUSIONS */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="Common Confusions Addressed"
          subtitle="Let's clear these up"
        />

        <div className="space-y-6">
          <div className="bg-card rounded-lg border p-4">
            <h4 className="mb-2 flex items-center gap-2 font-semibold">
              <span className="text-amber-400">Q:</span>
              "Is Sell like short selling?"
            </h4>
            <p className="text-muted-foreground text-sm">
              <strong>It can be.</strong> If you own contracts and sell them, you're closing your
              position. If you don't own contracts and sell them, you're opening a short position.
              The API doesn't distinguish - it just adjusts your position accordingly.
            </p>
          </div>

          <div className="bg-card rounded-lg border p-4">
            <h4 className="mb-2 flex items-center gap-2 font-semibold">
              <span className="text-amber-400">Q:</span>
              "Is Buy YES the opposite of Buy NO?"
            </h4>
            <p className="text-muted-foreground text-sm">
              <strong>No.</strong> They're both "Buy" actions on different contract types. "Buy YES"
              profits if the event happens. "Buy NO" profits if it doesn't. The opposite of "Buy
              YES" is "Sell YES".
            </p>
          </div>

          <div className="bg-card rounded-lg border p-4">
            <h4 className="mb-2 flex items-center gap-2 font-semibold">
              <span className="text-amber-400">Q:</span>
              "Why would I Sell NO instead of Buy YES?"
            </h4>
            <p className="text-muted-foreground text-sm">
              <strong>Different order books.</strong> Both profit if the event happens, but they
              trade in separate markets. One might have better prices, more liquidity, or shorter
              queue times than the other.
            </p>
          </div>

          <div className="bg-card rounded-lg border p-4">
            <h4 className="mb-2 flex items-center gap-2 font-semibold">
              <span className="text-amber-400">Q:</span>
              "What happens to my short position when the market settles?"
            </h4>
            <p className="text-muted-foreground text-sm">
              <strong>You pay or you don't.</strong> If you're short YES and the event happens, you
              pay $1 per contract. If it doesn't happen, you pay nothing (and keep the premium you
              received when you sold).
            </p>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 9: MENTAL MODEL */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="The Grid">
            Draw a 2×2 grid: Actions (Buy/Sell) on one axis, Sides (YES/NO) on the other. Each cell
            is a distinct order type with distinct meaning.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="The Mental Model"
          subtitle="How to think about this going forward"
        />

        <p className="text-muted-foreground mb-6 leading-relaxed">
          Think of every Kalshi order as answering two questions:
        </p>

        <div className="bg-card rounded-lg border p-6">
          <div className="grid gap-6 sm:grid-cols-2">
            <div className="bg-muted/30 rounded-lg p-4 text-center">
              <div className="mb-2 text-2xl">1️⃣</div>
              <div className="mb-1 font-semibold">Which contract?</div>
              <div className="text-muted-foreground text-sm">YES or NO</div>
              <div className="text-muted-foreground mt-2 text-xs">
                (What pays out if the event happens vs doesn't)
              </div>
            </div>
            <div className="bg-muted/30 rounded-lg p-4 text-center">
              <div className="mb-2 text-2xl">2️⃣</div>
              <div className="mb-1 font-semibold">What are you doing?</div>
              <div className="text-muted-foreground text-sm">Buy or Sell</div>
              <div className="text-muted-foreground mt-2 text-xs">
                (Acquiring contracts vs disposing of them)
              </div>
            </div>
          </div>

          <div className="mt-6 border-t pt-6 text-center">
            <p className="text-muted-foreground text-sm">
              <ArrowLeftRight className="mr-1 inline-block h-4 w-4" />
              These two choices are <strong>independent</strong>. Don't conflate them.
            </p>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 10: NEXT STEPS */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="What's Next?"
          subtitle="Now that you understand order mechanics"
        />

        <p className="text-muted-foreground mb-6 leading-relaxed">
          With this foundation, you can now understand more advanced topics like how equivalent
          positions trade in different order books (covered in YES/NO Markets Explained), and how
          market makers use these mechanics strategically.
        </p>

        {nextGuide && (
          <NextStepBlock
            href={`/${nextGuide.slug}`}
            title={nextGuide.title}
            description={nextGuide.description}
            buttonText="Continue Learning"
          />
        )}
      </LessonRow>
    </LessonLayout>
  );
}
