import { useState } from 'react';
import {
  DollarSign,
  Calculator,
  TrendingUp,
  AlertTriangle,
  Percent,
  Scale,
  History,
  Receipt,
} from 'lucide-react';
import { LessonRow, SectionHeader } from '../LessonLayout';
import { GlossaryTooltip } from '@/components/atoms/GlossaryTooltip';
import { Formula } from '@/components/atoms/Formula';
import { TipBlock, InsightBlock, WarningBlock } from '../blocks';

// Custom glossary terms for this lesson
const terms = {
  expectedValue: {
    term: 'Expected Value',
    definition:
      'The average profit/loss per trade over time. Positive EV means profitable long-term after accounting for all costs.',
    category: 'Strategy',
  },
  spread: {
    term: 'Spread',
    definition:
      'The difference between the bid (buy) and ask (sell) price. Tighter spreads mean lower trading costs.',
    category: 'Order Book',
  },
  liquidity: {
    term: 'Liquidity',
    definition:
      'How easily you can buy/sell without moving the price. More orders on the book means better liquidity.',
    category: 'Market',
  },
  marketMaker: {
    term: 'Market Maker',
    definition:
      'A trader who provides liquidity by posting both buy and sell orders, profiting from the spread.',
    category: 'Trading',
  },
};

// ===========================================================================
// FEE CALCULATOR WIDGET
// Interactive widget for calculating trading costs
// ===========================================================================

function FeeCalculatorWidget() {
  const [contracts, setContracts] = useState(10);
  const [entryPrice, setEntryPrice] = useState(50);
  const [exitPrice, setExitPrice] = useState(65);

  const FEE_PER_CONTRACT = 0.02; // 2 cents per contract per side

  const entryCost = contracts * entryPrice;
  const exitProceeds = contracts * exitPrice;
  const grossProfit = exitProceeds - entryCost;
  const totalFees = contracts * FEE_PER_CONTRACT * 2; // Entry and exit
  const netProfit = grossProfit - totalFees;
  const feeAsPercentOfProfit = grossProfit > 0 ? (totalFees / grossProfit) * 100 : 0;

  return (
    <div className="bg-card space-y-4 rounded-lg border p-4">
      <div className="text-primary flex items-center gap-2">
        <Calculator className="h-5 w-5" />
        <span className="font-medium">Fee Impact Calculator</span>
      </div>

      <div className="grid grid-cols-3 gap-4">
        <div>
          <label className="text-muted-foreground text-sm">Contracts</label>
          <input
            type="range"
            min="1"
            max="100"
            value={contracts}
            onChange={(e) => setContracts(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center font-mono">{contracts}</div>
        </div>

        <div>
          <label className="text-muted-foreground text-sm">Entry Price</label>
          <input
            type="range"
            min="1"
            max="99"
            value={entryPrice}
            onChange={(e) => setEntryPrice(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center font-mono">{entryPrice}c</div>
        </div>

        <div>
          <label className="text-muted-foreground text-sm">Exit Price</label>
          <input
            type="range"
            min="1"
            max="100"
            value={exitPrice}
            onChange={(e) => setExitPrice(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center font-mono">{exitPrice}c</div>
        </div>
      </div>

      <div className="grid grid-cols-2 gap-4 border-t pt-4">
        <div className="space-y-2">
          <div className="flex justify-between text-sm">
            <span className="text-muted-foreground">Entry cost:</span>
            <span className="font-mono">${(entryCost / 100).toFixed(2)}</span>
          </div>
          <div className="flex justify-between text-sm">
            <span className="text-muted-foreground">Exit proceeds:</span>
            <span className="font-mono">${(exitProceeds / 100).toFixed(2)}</span>
          </div>
          <div className="flex justify-between text-sm">
            <span className="text-muted-foreground">Gross profit:</span>
            <span className={`font-mono ${grossProfit >= 0 ? 'text-green-400' : 'text-red-400'}`}>
              ${(grossProfit / 100).toFixed(2)}
            </span>
          </div>
        </div>

        <div className="space-y-2">
          <div className="flex justify-between text-sm">
            <span className="text-muted-foreground">Entry fees:</span>
            <span className="font-mono text-yellow-400">
              -${(contracts * FEE_PER_CONTRACT).toFixed(2)}
            </span>
          </div>
          <div className="flex justify-between text-sm">
            <span className="text-muted-foreground">Exit fees:</span>
            <span className="font-mono text-yellow-400">
              -${(contracts * FEE_PER_CONTRACT).toFixed(2)}
            </span>
          </div>
          <div className="flex justify-between text-sm font-medium">
            <span>Net profit:</span>
            <span className={`font-mono ${netProfit >= 0 ? 'text-green-400' : 'text-red-400'}`}>
              ${(netProfit / 100).toFixed(2)}
            </span>
          </div>
        </div>
      </div>

      {grossProfit > 0 && (
        <div className="rounded border border-yellow-500/30 bg-yellow-500/10 p-3">
          <p className="text-sm text-yellow-400">
            Fees consume <strong>{feeAsPercentOfProfit.toFixed(1)}%</strong> of your gross profit on
            this trade.
          </p>
        </div>
      )}
    </div>
  );
}

// ===========================================================================
// BREAKEVEN CALCULATOR WIDGET
// Shows how much price needs to move to cover fees
// ===========================================================================

function BreakevenWidget() {
  const [contracts, setContracts] = useState(10);
  const [entryPrice, setEntryPrice] = useState(50);

  const FEE_PER_CONTRACT = 0.02;
  const totalFees = contracts * FEE_PER_CONTRACT * 2;
  const centsNeeded = Math.ceil((totalFees / contracts) * 100) / 100;
  const breakevenPrice = entryPrice + Math.ceil(centsNeeded * 100);

  return (
    <div className="bg-card space-y-4 rounded-lg border p-4">
      <div className="text-primary flex items-center gap-2">
        <Scale className="h-5 w-5" />
        <span className="font-medium">Breakeven Calculator</span>
      </div>

      <div className="grid grid-cols-2 gap-4">
        <div>
          <label className="text-muted-foreground text-sm">Contracts</label>
          <input
            type="range"
            min="1"
            max="100"
            value={contracts}
            onChange={(e) => setContracts(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center font-mono">{contracts}</div>
        </div>

        <div>
          <label className="text-muted-foreground text-sm">Entry Price</label>
          <input
            type="range"
            min="1"
            max="99"
            value={entryPrice}
            onChange={(e) => setEntryPrice(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center font-mono">{entryPrice}c</div>
        </div>
      </div>

      <div className="bg-muted/50 rounded p-4 text-center">
        <p className="text-muted-foreground mb-2 text-sm">To break even, price must reach:</p>
        <div className="text-primary text-3xl font-bold">{breakevenPrice}c</div>
        <p className="text-muted-foreground mt-2 text-sm">
          ({breakevenPrice - entryPrice} cent move needed to cover ${totalFees.toFixed(2)} in fees)
        </p>
      </div>

      <div className="text-muted-foreground text-xs">
        With more contracts, fees per contract stay constant, but breakeven remains the same
        percentage move.
      </div>
    </div>
  );
}

// ===========================================================================
// MAIN LESSON COMPONENT
// ===========================================================================

export function FeesAndCostsLesson() {
  return (
    <div className="space-y-12">
      {/* =================================================================== */}
      {/* SECTION 1: INTRODUCTION - THE HIDDEN COSTS OF TRADING */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="The Fee Paradox">
            Professional traders obsess over fees. Beginners ignore them. But beginners with smaller
            edges are actually MORE affected by fees than pros who can negotiate lower rates.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="The Hidden Costs of Trading"
          subtitle="Understanding what you actually pay"
        />

        <div className="text-muted-foreground space-y-4">
          <p>
            Every trade you make on Kalshi costs money beyond the price of the contract itself.
            These costs might seem small on a single trade, but they compound dramatically over
            hundreds or thousands of trades. Understanding fees is essential for:
          </p>

          <ul className="ml-4 list-inside list-disc space-y-2">
            <li>
              Calculating your true{' '}
              <GlossaryTooltip term={terms.expectedValue}>expected value</GlossaryTooltip> on each
              trade
            </li>
            <li>Determining whether a small edge is actually profitable</li>
            <li>Comparing different trading strategies</li>
            <li>
              Understanding why tight <GlossaryTooltip term={terms.spread}>spreads</GlossaryTooltip>{' '}
              matter
            </li>
          </ul>
        </div>

        <div className="rounded-lg border border-yellow-500/30 bg-yellow-500/10 p-4">
          <div className="flex items-start gap-3">
            <AlertTriangle className="mt-0.5 h-5 w-5 flex-shrink-0 text-yellow-400" />
            <div>
              <h4 className="font-medium text-yellow-400">The fee paradox</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                Professional traders obsess over fees. Beginners ignore them. This is backwards:
                beginners with smaller accounts and tighter edges are actually <em>more</em>{' '}
                affected by fees than professionals who can negotiate lower rates.
              </p>
            </div>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 2: KALSHI'S FEE STRUCTURE */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="Kalshi's Fee Structure"
          subtitle="What you pay per transaction"
        />

        <div className="text-muted-foreground space-y-4">
          <p>
            Kalshi charges a simple, transparent fee structure. Unlike traditional brokerages with
            complex commission tiers, Kalshi uses a flat per-contract fee:
          </p>
        </div>

        <div className="grid gap-6 md:grid-cols-2">
          <div className="bg-card rounded-lg border p-4">
            <div className="mb-3 flex items-center gap-2">
              <Receipt className="text-primary h-5 w-5" />
              <h4 className="font-medium">Exchange Fee</h4>
            </div>
            <div className="text-primary mb-2 text-4xl font-bold">2c</div>
            <p className="text-muted-foreground text-sm">
              Per contract, per side. You pay 2 cents when you enter a position and 2 cents when you
              exit (or when the contract settles).
            </p>
          </div>

          <div className="bg-card rounded-lg border p-4">
            <div className="mb-3 flex items-center gap-2">
              <DollarSign className="h-5 w-5 text-green-400" />
              <h4 className="font-medium">Settlement Fee</h4>
            </div>
            <div className="mb-2 text-4xl font-bold text-green-400">2c</div>
            <p className="text-muted-foreground text-sm">
              If you hold to expiration, you pay the 2-cent fee when the contract settles instead of
              when you exit. Same cost, different timing.
            </p>
          </div>
        </div>

        <div className="bg-muted/50 space-y-3 rounded-lg p-4">
          <h4 className="font-medium">What this means in practice:</h4>
          <ul className="text-muted-foreground ml-4 list-inside list-disc space-y-2 text-sm">
            <li>
              <strong>Round trip cost:</strong> 4 cents per contract (2c entry + 2c exit)
            </li>
            <li>
              <strong>10 contracts:</strong> 40 cents total fees
            </li>
            <li>
              <strong>100 contracts:</strong> $4.00 total fees
            </li>
            <li>
              <strong>No fee on losing contracts:</strong> If a contract settles at $0, you still
              pay the settlement fee
            </li>
          </ul>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 3: INTERACTIVE FEE CALCULATOR */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="See Fee Impact in Action"
          subtitle="How fees eat into profits"
        />

        <div className="text-muted-foreground space-y-4">
          <p>
            Use this calculator to see how fees affect your actual profit. Pay attention to how the
            percentage of profit consumed by fees changes based on your trade size and profit
            margin:
          </p>
        </div>

        <FeeCalculatorWidget />

        <div className="bg-card rounded-lg border p-4">
          <h4 className="mb-3 font-medium">Key observations:</h4>
          <ul className="text-muted-foreground ml-4 list-inside list-disc space-y-2 text-sm">
            <li>
              Small profit margins are devastated by fees (a 5-cent profit loses 80% to fees on 10
              contracts)
            </li>
            <li>Larger profit margins are less affected percentage-wise</li>
            <li>
              The number of contracts doesn't change the percentage impact - it's about the{' '}
              <em>margin</em>
            </li>
          </ul>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 4: THE SPREAD AS A HIDDEN COST */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="Spread + Fees">
            The spread is a cost ON TOP of exchange fees. In a 4-cent spread market, your real cost
            to enter and exit is 8 cents (4c spread + 4c fees), not just 4 cents.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="The Spread: Your Hidden Cost"
          subtitle="The cost of immediacy"
        />

        <div className="text-muted-foreground space-y-4">
          <p>
            Beyond exchange fees, the{' '}
            <GlossaryTooltip term={terms.spread}>bid-ask spread</GlossaryTooltip> represents a real
            cost that many traders overlook. When you want to trade immediately, you cross the
            spread, effectively paying this cost.
          </p>
        </div>

        <div className="bg-card rounded-lg border p-4">
          <h4 className="mb-4 font-medium">Example: Crossing the spread</h4>

          <div className="mb-4 grid grid-cols-3 gap-4 text-center">
            <div className="rounded bg-green-500/10 p-3">
              <div className="text-muted-foreground text-sm">Best Bid</div>
              <div className="text-2xl font-bold text-green-400">48c</div>
            </div>
            <div className="bg-muted/50 rounded p-3">
              <div className="text-muted-foreground text-sm">Spread</div>
              <div className="text-2xl font-bold">4c</div>
            </div>
            <div className="rounded bg-red-500/10 p-3">
              <div className="text-muted-foreground text-sm">Best Ask</div>
              <div className="text-2xl font-bold text-red-400">52c</div>
            </div>
          </div>

          <div className="text-muted-foreground space-y-2 text-sm">
            <p>
              If you buy at the ask (52c) and immediately sell at the bid (48c), you lose 4 cents
              per contract just from the spread - before any fees!
            </p>
            <p>
              On 10 contracts:{' '}
              <Formula>
                {'10 \\times 4\\text{c (spread)} + 10 \\times 4\\text{c (fees)} = 80\\text{c}'}
              </Formula>{' '}
              to enter and exit.
            </p>
          </div>
        </div>

        <div className="grid gap-6 md:grid-cols-2">
          <div className="rounded-lg border border-green-500/30 bg-green-500/10 p-4">
            <h4 className="mb-2 font-medium text-green-400">Patient traders</h4>
            <p className="text-muted-foreground text-sm">
              Place limit orders at the bid or ask and wait to get filled. You might provide
              <GlossaryTooltip term={terms.liquidity}>liquidity</GlossaryTooltip> instead of taking
              it, avoiding the spread cost entirely.
            </p>
          </div>

          <div className="rounded-lg border border-red-500/30 bg-red-500/10 p-4">
            <h4 className="mb-2 font-medium text-red-400">Aggressive traders</h4>
            <p className="text-muted-foreground text-sm">
              Cross the spread with market orders for immediate execution. Pay the spread cost but
              guarantee you get filled at the current price.
            </p>
          </div>
        </div>

        <div className="rounded-lg border border-blue-500/30 bg-blue-500/10 p-4">
          <h4 className="mb-2 font-medium text-blue-400">The market maker's advantage</h4>
          <p className="text-muted-foreground text-sm">
            <GlossaryTooltip term={terms.marketMaker}>Market makers</GlossaryTooltip> profit by
            capturing the spread. They buy at the bid and sell at the ask. In a market with a 4-cent
            spread, a market maker who trades both sides profits{' '}
            <Formula>{'4\\text{c (spread)} - 4\\text{c (fees)} = 0\\text{c}'}</Formula>. This is why
            market makers need volume and tight spreads to make money.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 5: CALCULATING BREAKEVEN */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="Know Before You Trade">
            Always calculate your breakeven BEFORE entering a trade. If you need a 5-cent move to
            break even but your edge is only 3 cents, that's a losing trade.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="Calculating Your Breakeven"
          subtitle="How much does price need to move?"
        />

        <div className="text-muted-foreground space-y-4">
          <p>
            Before entering any trade, you should know exactly how much the price needs to move in
            your favor just to cover your costs. This is your breakeven point.
          </p>
        </div>

        <BreakevenWidget />

        <div className="bg-card rounded-lg border p-4">
          <h4 className="mb-3 font-medium">The breakeven formula:</h4>
          <div className="bg-muted/50 rounded p-3">
            <Formula block>
              {'\\text{Breakeven Move} = \\frac{\\text{Total Fees}}{\\text{Number of Contracts}}'}
            </Formula>
          </div>
          <p className="text-muted-foreground mt-3 text-sm">
            With 4 cents total fees per contract (round trip), you need price to move at least 1
            cent in your favor just to break even. Any edge smaller than 1 cent is unprofitable.
          </p>
        </div>

        <div className="rounded-lg border border-yellow-500/30 bg-yellow-500/10 p-4">
          <div className="flex items-start gap-3">
            <AlertTriangle className="mt-0.5 h-5 w-5 flex-shrink-0 text-yellow-400" />
            <div>
              <h4 className="font-medium text-yellow-400">Don't forget the spread</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                If you cross a 4-cent spread to enter, your true breakeven includes that cost too.
                Entering at 52c when the bid is 48c means you need to see 56c just to break even
                after fees (52c + 4c fees, then sell at bid of 56c - 4c spread - 4c fees = 52c).
              </p>
            </div>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 6: FEE IMPACT BY STRATEGY */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="The Volume Trade-off">
            High-frequency strategies generate more fees but also more opportunities. Low-frequency
            strategies pay less fees but need larger edges. Match your strategy to your edge size.
          </TipBlock>
        }
      >
        <SectionHeader
          title="Fee Impact by Strategy"
          subtitle="Different strategies, different fee burdens"
        />

        <div className="text-muted-foreground space-y-4">
          <p>
            Your trading strategy dramatically affects how much fees impact your returns.
            High-frequency strategies with small edges are most vulnerable to fee erosion.
          </p>
        </div>

        <div className="grid gap-4">
          <div className="bg-card rounded-lg border p-4">
            <div className="mb-2 flex items-center gap-2">
              <div className="h-3 w-3 rounded-full bg-green-400"></div>
              <h4 className="font-medium">Low frequency, high conviction</h4>
            </div>
            <p className="text-muted-foreground mb-2 text-sm">
              Trade only when you have a large edge (10+ cents). Make 5-10 trades per week.
            </p>
            <div className="text-muted-foreground text-xs">
              <strong>Fee impact:</strong> Minimal. Large profit margins easily absorb the 4-cent
              round trip.
            </div>
          </div>

          <div className="bg-card rounded-lg border p-4">
            <div className="mb-2 flex items-center gap-2">
              <div className="h-3 w-3 rounded-full bg-yellow-400"></div>
              <h4 className="font-medium">Medium frequency, medium edge</h4>
            </div>
            <p className="text-muted-foreground mb-2 text-sm">
              Trade 20-50 times per week with 5-10 cent average edges.
            </p>
            <div className="text-muted-foreground text-xs">
              <strong>Fee impact:</strong> Moderate. Fees take 10-20% of gross profits. Need good
              execution.
            </div>
          </div>

          <div className="bg-card rounded-lg border p-4">
            <div className="mb-2 flex items-center gap-2">
              <div className="h-3 w-3 rounded-full bg-red-400"></div>
              <h4 className="font-medium">High frequency, small edge</h4>
            </div>
            <p className="text-muted-foreground mb-2 text-sm">
              Scalping or market making with 2-5 cent target edges. 100+ trades per week.
            </p>
            <div className="text-muted-foreground text-xs">
              <strong>Fee impact:</strong> Critical. Fees can consume 50-100% of gross profits.
              Requires perfect execution and avoiding the spread.
            </div>
          </div>
        </div>

        <div className="bg-muted/50 rounded-lg p-4">
          <h4 className="mb-3 font-medium">Annual fee projection:</h4>
          <div className="overflow-x-auto">
            <table className="w-full text-sm">
              <thead>
                <tr className="border-b">
                  <th className="p-2 text-left">Strategy</th>
                  <th className="p-2 text-right">Trades/Week</th>
                  <th className="p-2 text-right">Contracts/Trade</th>
                  <th className="p-2 text-right">Annual Fees</th>
                </tr>
              </thead>
              <tbody className="text-muted-foreground">
                <tr className="border-b">
                  <td className="p-2">Casual</td>
                  <td className="p-2 text-right font-mono">5</td>
                  <td className="p-2 text-right font-mono">10</td>
                  <td className="p-2 text-right font-mono">$104</td>
                </tr>
                <tr className="border-b">
                  <td className="p-2">Active</td>
                  <td className="p-2 text-right font-mono">25</td>
                  <td className="p-2 text-right font-mono">20</td>
                  <td className="p-2 text-right font-mono">$1,040</td>
                </tr>
                <tr className="border-b">
                  <td className="p-2">Professional</td>
                  <td className="p-2 text-right font-mono">100</td>
                  <td className="p-2 text-right font-mono">50</td>
                  <td className="p-2 text-right font-mono">$10,400</td>
                </tr>
              </tbody>
            </table>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 7: COMPARING TO OTHER MARKETS */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="How Kalshi Compares"
          subtitle="Fees in context"
        />

        <div className="text-muted-foreground space-y-4">
          <p>
            Understanding how Kalshi's fees compare to other markets helps put the costs in
            perspective.
          </p>
        </div>

        <div className="grid gap-4">
          <div className="bg-card rounded-lg border p-4">
            <div className="flex items-center justify-between">
              <div>
                <h4 className="font-medium">Kalshi</h4>
                <p className="text-muted-foreground text-sm">Prediction markets</p>
              </div>
              <div className="text-right">
                <div className="font-mono text-lg">4c / contract</div>
                <p className="text-muted-foreground text-sm">Round trip</p>
              </div>
            </div>
          </div>

          <div className="bg-card rounded-lg border p-4">
            <div className="flex items-center justify-between">
              <div>
                <h4 className="font-medium">Sports Betting</h4>
                <p className="text-muted-foreground text-sm">Traditional sportsbooks</p>
              </div>
              <div className="text-right">
                <div className="font-mono text-lg">~4.5% vig</div>
                <p className="text-muted-foreground text-sm">Built into odds</p>
              </div>
            </div>
          </div>

          <div className="bg-card rounded-lg border p-4">
            <div className="flex items-center justify-between">
              <div>
                <h4 className="font-medium">Stock Options</h4>
                <p className="text-muted-foreground text-sm">Major brokers</p>
              </div>
              <div className="text-right">
                <div className="font-mono text-lg">$0.50-0.65 / contract</div>
                <p className="text-muted-foreground text-sm">Per side</p>
              </div>
            </div>
          </div>

          <div className="bg-card rounded-lg border p-4">
            <div className="flex items-center justify-between">
              <div>
                <h4 className="font-medium">Crypto Futures</h4>
                <p className="text-muted-foreground text-sm">Major exchanges</p>
              </div>
              <div className="text-right">
                <div className="font-mono text-lg">0.02-0.05%</div>
                <p className="text-muted-foreground text-sm">Of notional</p>
              </div>
            </div>
          </div>
        </div>

        <div className="rounded-lg border border-green-500/30 bg-green-500/10 p-4">
          <h4 className="mb-2 font-medium text-green-400">The Kalshi advantage</h4>
          <p className="text-muted-foreground text-sm">
            Compared to sports betting's 4.5% vig, Kalshi's 4-cent fee on a 50-cent contract is only{' '}
            <Formula>{'\\frac{4\\text{c}}{50\\text{c}} = 8\\%'}</Formula> - significantly lower.
            This makes Kalshi more attractive for sophisticated bettors who can find and exploit
            edges.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 8: STRATEGIES TO MINIMIZE FEES */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="Limit Orders Save Money">
            Using limit orders instead of market orders can save you the entire spread cost — often
            larger than the exchange fees themselves. Patience pays.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="Minimizing Your Fee Burden"
          subtitle="Practical strategies"
        />

        <div className="space-y-4">
          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">1</span>
            </div>
            <div>
              <h4 className="font-medium">Trade less frequently with larger edges</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                Wait for high-conviction opportunities rather than scalping small edges. One trade
                with a 20-cent edge beats four trades with 5-cent edges after fees.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">2</span>
            </div>
            <div>
              <h4 className="font-medium">Use limit orders</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                Avoid crossing the spread when possible. Place limit orders at favorable prices and
                let the market come to you. You save the spread cost on both entry and exit.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">3</span>
            </div>
            <div>
              <h4 className="font-medium">Choose liquid markets</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                Markets with tight spreads (1-2 cents) have lower effective costs than illiquid
                markets with 5-10 cent spreads. The exchange fee is fixed, but spread costs vary.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">4</span>
            </div>
            <div>
              <h4 className="font-medium">Factor fees into your edge calculation</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                Always calculate your <em>net</em> expected value after fees. A 3-cent edge that
                looks profitable is actually marginal after the 4-cent round trip fee.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">5</span>
            </div>
            <div>
              <h4 className="font-medium">Consider holding to settlement</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                If you're confident in your position, holding to settlement avoids the exit spread
                cost. You pay the same 2-cent fee at settlement, but don't lose the spread.
              </p>
            </div>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 9: HISTORICAL CONTEXT */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="The History of Trading Fees"
          subtitle="How we got here"
        />

        <div className="bg-muted/50 rounded-lg p-4">
          <div className="mb-3 flex items-center gap-2">
            <History className="text-primary h-5 w-5" />
            <h4 className="font-medium">From floor trading to zero commissions</h4>
          </div>

          <div className="text-muted-foreground space-y-4 text-sm">
            <p>
              For most of financial history, trading was expensive. Before 1975, stock commissions
              were fixed by the NYSE at high rates. The deregulation of commissions led to discount
              brokers like Charles Schwab, and eventually to the "zero commission" era we know
              today.
            </p>

            <p>
              But "zero commission" is misleading. Modern brokers make money through payment for
              order flow (PFOF) - selling your orders to market makers who profit from the spread.
              The cost is hidden, not eliminated.
            </p>

            <p>
              Kalshi's transparent 2-cent fee is actually more honest than "free" trading. You know
              exactly what you're paying, and there's no hidden spread extraction.
            </p>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 10: KEY TAKEAWAYS */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="Key Takeaways"
          subtitle="What to remember about fees"
        />

        <div className="space-y-4">
          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <Percent className="text-primary h-4 w-4" />
            </div>
            <div>
              <h4 className="font-medium">4 cents round trip</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                Every contract costs 2 cents to enter and 2 cents to exit or settle. This is fixed
                regardless of the contract price.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <TrendingUp className="text-primary h-4 w-4" />
            </div>
            <div>
              <h4 className="font-medium">Small edges get crushed</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                A 5-cent gross profit becomes a 1-cent net profit after fees:{' '}
                <Formula>{'5\\text{c} - 4\\text{c} = 1\\text{c}'}</Formula> (80% fee burden). Seek
                larger edges to make fees negligible.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <Scale className="text-primary h-4 w-4" />
            </div>
            <div>
              <h4 className="font-medium">The spread is a cost too</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                Crossing the bid-ask spread costs money just like fees. Use limit orders when
                possible to avoid this hidden cost.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <Calculator className="text-primary h-4 w-4" />
            </div>
            <div>
              <h4 className="font-medium">Always calculate net EV</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                Before every trade, know your breakeven point. Subtract fees from your edge to see
                if the trade is truly profitable.
              </p>
            </div>
          </div>
        </div>
      </LessonRow>
    </div>
  );
}
