/**
 * Lesson: Glossary
 *
 * Complete reference of trading and betting terminology.
 */

import { useState, useMemo } from 'react';
import { Search, BookOpen, ChevronRight, ExternalLink } from 'lucide-react';
import { Link } from 'react-router-dom';
import { cn } from '@/lib/utils';
import { getGuide } from '@/data/guides';
import { tradingGlossary, getAllCategories, type GlossaryEntry } from '@/data/guides/glossary';
import { LessonLayout, LessonRow, LessonHeader, SectionHeader } from '../LessonLayout';
import { ObjectiveBlock, TipBlock } from '../blocks';

export function GlossaryLesson() {
  const guide = getGuide('glossary')!;
  const [searchQuery, setSearchQuery] = useState('');
  const [selectedCategory, setSelectedCategory] = useState<string | null>(null);
  const [expandedTerm, setExpandedTerm] = useState<string | null>(null);

  const categories = getAllCategories();

  // Filter and sort terms
  const filteredTerms = useMemo(() => {
    return Object.entries(tradingGlossary)
      .filter(([, entry]) => {
        const matchesSearch =
          searchQuery === '' ||
          entry.term.toLowerCase().includes(searchQuery.toLowerCase()) ||
          entry.definition.toLowerCase().includes(searchQuery.toLowerCase());
        const matchesCategory = !selectedCategory || entry.category === selectedCategory;
        return matchesSearch && matchesCategory;
      })
      .sort(([, a], [, b]) => a.term.localeCompare(b.term));
  }, [searchQuery, selectedCategory]);

  // Group by first letter
  const groupedTerms = useMemo(() => {
    const groups: Record<string, [string, GlossaryEntry][]> = {};
    filteredTerms.forEach(([key, entry]) => {
      const letter = entry.term[0].toUpperCase();
      if (!groups[letter]) groups[letter] = [];
      groups[letter].push([key, entry]);
    });
    return groups;
  }, [filteredTerms]);

  return (
    <LessonLayout>
      {/* Header */}
      <LessonRow>
        <LessonHeader
          title={guide.title}
          description={guide.description}
          duration={guide.duration}
          category="Reference"
        />
      </LessonRow>

      {/* Objective */}
      <LessonRow
        aside={
          <TipBlock title="How to Use">
            Search for any term or browse by category. Click a term to see its full definition,
            etymology, and related terms.
          </TipBlock>
        }
      >
        <ObjectiveBlock>
          A comprehensive reference of all trading, betting, and market terminology used throughout
          these guides. Each term includes its definition, origin, and context.
        </ObjectiveBlock>
      </LessonRow>

      {/* Search and Filter */}
      <LessonRow>
        <div className="bg-card rounded-lg border p-4">
          {/* Search */}
          <div className="relative mb-4">
            <Search className="text-muted-foreground absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2" />
            <input
              type="text"
              placeholder="Search terms..."
              value={searchQuery}
              onChange={(e) => setSearchQuery(e.target.value)}
              className="bg-muted border-border focus:ring-primary/50 w-full rounded-lg border py-2 pl-10 pr-4 text-sm focus:outline-none focus:ring-2"
            />
          </div>

          {/* Category Filters */}
          <div className="flex flex-wrap items-center gap-2">
            <span className="text-muted-foreground text-xs">Filter:</span>
            <button
              onClick={() => setSelectedCategory(null)}
              className={cn(
                'rounded px-2 py-1 text-xs transition-colors',
                !selectedCategory
                  ? 'bg-primary text-primary-foreground'
                  : 'bg-muted text-muted-foreground hover:bg-muted/80'
              )}
            >
              All ({Object.keys(tradingGlossary).length})
            </button>
            {categories.map((cat) => {
              const count = Object.values(tradingGlossary).filter((t) => t.category === cat).length;
              return (
                <button
                  key={cat}
                  onClick={() => setSelectedCategory(cat)}
                  className={cn(
                    'rounded px-2 py-1 text-xs transition-colors',
                    selectedCategory === cat
                      ? 'bg-primary text-primary-foreground'
                      : 'bg-muted text-muted-foreground hover:bg-muted/80'
                  )}
                >
                  {cat} ({count})
                </button>
              );
            })}
          </div>
        </div>
      </LessonRow>

      {/* Results Count */}
      <LessonRow>
        <div className="text-muted-foreground flex items-center gap-2 text-sm">
          <BookOpen className="h-4 w-4" />
          <span>
            {filteredTerms.length} term{filteredTerms.length !== 1 ? 's' : ''} found
          </span>
        </div>
      </LessonRow>

      {/* Terms List */}
      <LessonRow>
        <div className="space-y-6">
          {Object.keys(groupedTerms).length === 0 ? (
            <div className="text-muted-foreground py-8 text-center">
              No terms found matching "{searchQuery}"
            </div>
          ) : (
            Object.entries(groupedTerms).map(([letter, terms]) => (
              <div key={letter}>
                {/* Letter Header */}
                <div className="bg-background sticky top-0 z-10 mb-2 border-b py-2">
                  <span className="text-primary text-lg font-bold">{letter}</span>
                </div>

                {/* Terms */}
                <div className="space-y-2">
                  {terms.map(([key, entry]) => (
                    <TermCard
                      key={key}
                      entry={entry}
                      isExpanded={expandedTerm === key}
                      onToggle={() => setExpandedTerm(expandedTerm === key ? null : key)}
                    />
                  ))}
                </div>
              </div>
            ))
          )}
        </div>
      </LessonRow>

      {/* Quick Category Navigation */}
      <LessonRow>
        <SectionHeader
          title="Browse by Category"
          subtitle="Jump to terms in a specific area"
        />
        <div className="grid gap-4 sm:grid-cols-2 lg:grid-cols-3">
          {categories.map((cat) => {
            const terms = Object.values(tradingGlossary).filter((t) => t.category === cat);
            return (
              <button
                key={cat}
                onClick={() => {
                  setSelectedCategory(cat);
                  setSearchQuery('');
                  window.scrollTo({ top: 0, behavior: 'smooth' });
                }}
                className="bg-card hover:border-primary/50 rounded-lg border p-4 text-left transition-colors"
              >
                <div className="font-medium">{cat}</div>
                <div className="text-muted-foreground mt-1 text-sm">
                  {terms.length} term{terms.length !== 1 ? 's' : ''}
                </div>
              </button>
            );
          })}
        </div>
      </LessonRow>
    </LessonLayout>
  );
}

interface TermCardProps {
  entry: GlossaryEntry;
  isExpanded: boolean;
  onToggle: () => void;
}

function TermCard({ entry, isExpanded, onToggle }: TermCardProps) {
  return (
    <div className="bg-card overflow-hidden rounded-lg border">
      <button
        onClick={onToggle}
        className="hover:bg-muted/50 flex w-full items-center justify-between px-4 py-3 text-left transition-colors"
      >
        <div className="flex items-center gap-3">
          <span className="font-medium">{entry.term}</span>
          <span className="bg-muted text-muted-foreground rounded px-2 py-0.5 text-[10px]">
            {entry.category}
          </span>
        </div>
        <ChevronRight
          className={cn(
            'text-muted-foreground h-4 w-4 transition-transform',
            isExpanded && 'rotate-90'
          )}
        />
      </button>

      {isExpanded && (
        <div className="bg-muted/20 border-t px-4 pb-4 pt-0">
          {/* Definition */}
          <div className="mt-3">
            <div className="text-muted-foreground mb-1 text-xs font-semibold uppercase tracking-wider">
              Definition
            </div>
            <p className="text-sm">{entry.definition}</p>
          </div>

          {/* Etymology */}
          {entry.etymology && (
            <div className="mt-4">
              <div className="text-muted-foreground mb-1 text-xs font-semibold uppercase tracking-wider">
                Etymology & History
              </div>
              <p className="text-muted-foreground text-sm">{entry.etymology}</p>
            </div>
          )}

          {/* See Also */}
          {entry.seeAlso && entry.seeAlso.length > 0 && (
            <div className="mt-4">
              <div className="text-muted-foreground mb-1 text-xs font-semibold uppercase tracking-wider">
                Related Terms
              </div>
              <div className="flex flex-wrap gap-2">
                {entry.seeAlso.map((key) => {
                  const related = tradingGlossary[key];
                  if (!related) return null;
                  return (
                    <span
                      key={key}
                      className="bg-muted text-muted-foreground rounded px-2 py-1 text-xs"
                    >
                      {related.term}
                    </span>
                  );
                })}
              </div>
            </div>
          )}

          {/* Used In */}
          {entry.usedIn && entry.usedIn.length > 0 && (
            <div className="mt-4">
              <div className="text-muted-foreground mb-1 text-xs font-semibold uppercase tracking-wider">
                Used In Lessons
              </div>
              <div className="flex flex-wrap gap-2">
                {entry.usedIn.map((slug) => {
                  const guide = getGuide(slug);
                  if (!guide) return null;
                  return (
                    <Link
                      key={slug}
                      to={`/${slug}`}
                      className="bg-primary/10 text-primary hover:bg-primary/20 flex items-center gap-1 rounded px-2 py-1 text-xs transition-colors"
                    >
                      {guide.title}
                      <ExternalLink className="h-3 w-3" />
                    </Link>
                  );
                })}
              </div>
            </div>
          )}
        </div>
      )}
    </div>
  );
}
