/**
 * Lesson: Kalshi API Explorer
 *
 * An interactive API reference for developers building on Kalshi.
 * More traditional API documentation style but interactive and digestible.
 */

import { useState, useMemo } from 'react';
import { getGuide, getNextGuide } from '@/data/guides';
import { LessonLayout, LessonRow, LessonHeader, SectionHeader } from '../LessonLayout';
import { ObjectiveBlock, TipBlock, InsightBlock, WarningBlock, NextStepBlock } from '../blocks';
import {
  Search,
  ChevronRight,
  ChevronDown,
  Copy,
  Check,
  ExternalLink,
  Server,
  Lock,
  Zap,
  Database,
  Globe,
} from 'lucide-react';

// =============================================================================
// TYPES
// =============================================================================

type HttpMethod = 'GET' | 'POST' | 'PUT' | 'DELETE';

type EndpointParam = {
  name: string;
  type: string;
  required: boolean;
  description: string;
  example?: string;
};

type EndpointResponse = {
  field: string;
  type: string;
  description: string;
};

type Endpoint = {
  id: string;
  method: HttpMethod;
  path: string;
  title: string;
  description: string;
  category: string;
  auth: boolean;
  params?: EndpointParam[];
  queryParams?: EndpointParam[];
  bodyParams?: EndpointParam[];
  response?: EndpointResponse[];
  example?: {
    request?: string;
    response?: string;
  };
  notes?: string[];
  rateLimit?: string;
};

type EndpointCategory = {
  id: string;
  name: string;
  description: string;
  icon: React.ReactNode;
  endpoints: Endpoint[];
};

// =============================================================================
// API DATA
// =============================================================================

const apiCategories: EndpointCategory[] = [
  {
    id: 'markets',
    name: 'Markets',
    description: 'Query markets, orderbooks, and candlestick data',
    icon: <Database className="h-4 w-4" />,
    endpoints: [
      {
        id: 'get-markets',
        method: 'GET',
        path: '/markets',
        title: 'Get Markets',
        description:
          'Retrieve a list of markets with optional filters. Supports filtering by status, series, event, and more.',
        category: 'markets',
        auth: false,
        queryParams: [
          {
            name: 'limit',
            type: 'integer',
            required: false,
            description: 'Maximum results to return (1-1000)',
            example: '100',
          },
          {
            name: 'cursor',
            type: 'string',
            required: false,
            description: 'Pagination cursor from previous response',
          },
          {
            name: 'status',
            type: 'string',
            required: false,
            description: 'Filter by status: unopened, open, closed, settled',
          },
          {
            name: 'series_ticker',
            type: 'string',
            required: false,
            description: 'Filter by series ticker',
          },
          {
            name: 'event_ticker',
            type: 'string',
            required: false,
            description: 'Filter by event ticker',
          },
          {
            name: 'tickers',
            type: 'string',
            required: false,
            description: 'Comma-separated list of market tickers',
          },
        ],
        response: [
          { field: 'markets', type: 'Market[]', description: 'Array of market objects' },
          {
            field: 'cursor',
            type: 'string',
            description: 'Cursor for next page (if more results)',
          },
        ],
        example: {
          request: 'GET /markets?status=open&limit=10',
          response: `{
  "markets": [
    {
      "ticker": "INXD-25JAN21-B5900",
      "event_ticker": "INXD-25JAN21",
      "status": "open",
      "yes_bid": 62,
      "yes_ask": 64,
      "last_price": 63,
      "volume": 15420,
      "open_interest": 8350
    }
  ],
  "cursor": "eyJsYXN0X2lkIjo..."
}`,
        },
        notes: [
          'No authentication required for public market data',
          'Rate limited to 10 requests/second',
          'Use pagination for large result sets',
        ],
      },
      {
        id: 'get-market',
        method: 'GET',
        path: '/markets/{ticker}',
        title: 'Get Market',
        description: 'Retrieve detailed data for a single market by its ticker.',
        category: 'markets',
        auth: false,
        params: [
          {
            name: 'ticker',
            type: 'string',
            required: true,
            description: 'Market ticker (e.g., "INXD-25JAN21-B5900")',
          },
        ],
        response: [
          { field: 'ticker', type: 'string', description: 'Unique market identifier' },
          { field: 'event_ticker', type: 'string', description: 'Parent event ticker' },
          {
            field: 'status',
            type: 'string',
            description: 'Market status: unopened | open | closed | settled',
          },
          { field: 'yes_bid', type: 'integer', description: 'Best bid price for YES (1-99 cents)' },
          { field: 'yes_ask', type: 'integer', description: 'Best ask price for YES (1-99 cents)' },
          { field: 'last_price', type: 'integer', description: 'Most recent trade price' },
          { field: 'volume', type: 'integer', description: 'Total contracts traded' },
          { field: 'volume_24h', type: 'integer', description: 'Volume in last 24 hours' },
          { field: 'open_interest', type: 'integer', description: 'Outstanding contracts' },
          { field: 'result', type: 'string?', description: '"yes" | "no" | null (if not settled)' },
        ],
        example: {
          request: 'GET /markets/INXD-25JAN21-B5900',
          response: `{
  "market": {
    "ticker": "INXD-25JAN21-B5900",
    "event_ticker": "INXD-25JAN21",
    "status": "open",
    "yes_bid": 62,
    "yes_ask": 64,
    "last_price": 63,
    "volume": 15420,
    "volume_24h": 3200,
    "open_interest": 8350,
    "result": null
  }
}`,
        },
      },
      {
        id: 'get-orderbook',
        method: 'GET',
        path: '/markets/{ticker}/orderbook',
        title: 'Get Orderbook',
        description: 'Retrieve the current orderbook (bid/ask levels) for a market.',
        category: 'markets',
        auth: false,
        params: [{ name: 'ticker', type: 'string', required: true, description: 'Market ticker' }],
        queryParams: [
          {
            name: 'depth',
            type: 'integer',
            required: false,
            description: 'Number of price levels (default: all)',
          },
        ],
        response: [
          {
            field: 'yes',
            type: '[[price, qty]]',
            description: 'YES side bids: array of [price, quantity] pairs',
          },
          {
            field: 'no',
            type: '[[price, qty]]',
            description: 'NO side bids: array of [price, quantity] pairs',
          },
        ],
        example: {
          response: `{
  "orderbook": {
    "yes": [[62, 500], [61, 1200], [60, 800]],
    "no": [[37, 450], [36, 900], [35, 600]]
  }
}`,
        },
        notes: [
          'YES bid at 62¢ is equivalent to NO ask at 38¢',
          'Prices are in cents (1-99)',
          'Quantities are number of contracts',
        ],
      },
      {
        id: 'get-candlesticks',
        method: 'GET',
        path: '/markets/{ticker}/candlesticks',
        title: 'Get Candlesticks',
        description: 'Retrieve OHLCV candlestick data for charting.',
        category: 'markets',
        auth: false,
        params: [{ name: 'ticker', type: 'string', required: true, description: 'Market ticker' }],
        queryParams: [
          {
            name: 'start_ts',
            type: 'integer',
            required: false,
            description: 'Start timestamp (Unix seconds)',
          },
          {
            name: 'end_ts',
            type: 'integer',
            required: false,
            description: 'End timestamp (Unix seconds)',
          },
          {
            name: 'period_interval',
            type: 'integer',
            required: false,
            description: 'Interval in minutes: 1, 60, or 1440',
          },
        ],
        response: [
          { field: 'candlesticks', type: 'Candlestick[]', description: 'Array of OHLCV data' },
        ],
        example: {
          response: `{
  "candlesticks": [
    {
      "end_period_ts": 1705788000,
      "yes_price": { "open": 60, "high": 65, "low": 58, "close": 63 },
      "volume": 1250,
      "open_interest": 8200
    }
  ]
}`,
        },
      },
      {
        id: 'get-trades',
        method: 'GET',
        path: '/markets/trades',
        title: 'Get Trades',
        description: 'Retrieve public trade history across all markets.',
        category: 'markets',
        auth: false,
        queryParams: [
          {
            name: 'ticker',
            type: 'string',
            required: false,
            description: 'Filter by market ticker',
          },
          {
            name: 'limit',
            type: 'integer',
            required: false,
            description: 'Max results (1-1000, default 100)',
          },
          { name: 'cursor', type: 'string', required: false, description: 'Pagination cursor' },
          { name: 'min_ts', type: 'integer', required: false, description: 'Minimum timestamp' },
          { name: 'max_ts', type: 'integer', required: false, description: 'Maximum timestamp' },
        ],
        response: [
          { field: 'trades', type: 'Trade[]', description: 'Array of trade records' },
          { field: 'cursor', type: 'string', description: 'Pagination cursor' },
        ],
        example: {
          response: `{
  "trades": [
    {
      "trade_id": "abc123",
      "ticker": "INXD-25JAN21-B5900",
      "count": 50,
      "yes_price": 63,
      "taker_side": "yes",
      "created_time": "2025-01-21T14:30:00Z"
    }
  ]
}`,
        },
      },
    ],
  },
  {
    id: 'events',
    name: 'Events',
    description: 'Query events and series that contain markets',
    icon: <Globe className="h-4 w-4" />,
    endpoints: [
      {
        id: 'get-events',
        method: 'GET',
        path: '/events',
        title: 'Get Events',
        description: 'Retrieve a list of events. Events are containers for related markets.',
        category: 'events',
        auth: false,
        queryParams: [
          {
            name: 'limit',
            type: 'integer',
            required: false,
            description: 'Maximum results (1-200)',
          },
          { name: 'cursor', type: 'string', required: false, description: 'Pagination cursor' },
          { name: 'status', type: 'string', required: false, description: 'Filter by status' },
          {
            name: 'series_ticker',
            type: 'string',
            required: false,
            description: 'Filter by series',
          },
          {
            name: 'with_nested_markets',
            type: 'boolean',
            required: false,
            description: 'Include markets array',
          },
        ],
        response: [
          { field: 'events', type: 'Event[]', description: 'Array of event objects' },
          { field: 'cursor', type: 'string', description: 'Pagination cursor' },
        ],
        example: {
          response: `{
  "events": [
    {
      "event_ticker": "INXD-25JAN21",
      "series_ticker": "INXD",
      "title": "S&P 500 Index on January 21, 2025",
      "strike_date": "2025-01-21T21:00:00Z",
      "mutually_exclusive": true,
      "markets": [...]
    }
  ]
}`,
        },
      },
      {
        id: 'get-event',
        method: 'GET',
        path: '/events/{event_ticker}',
        title: 'Get Event',
        description: 'Retrieve a single event by its ticker.',
        category: 'events',
        auth: false,
        params: [
          { name: 'event_ticker', type: 'string', required: true, description: 'Event ticker' },
        ],
        queryParams: [
          {
            name: 'with_nested_markets',
            type: 'boolean',
            required: false,
            description: 'Include markets array',
          },
        ],
        response: [
          { field: 'event_ticker', type: 'string', description: 'Unique event identifier' },
          { field: 'series_ticker', type: 'string', description: 'Parent series' },
          { field: 'title', type: 'string', description: 'Human-readable title' },
          {
            field: 'strike_date',
            type: 'string',
            description: 'When outcome is determined (ISO 8601)',
          },
          {
            field: 'mutually_exclusive',
            type: 'boolean',
            description: 'If true, exactly one market settles YES',
          },
          { field: 'markets', type: 'Market[]?', description: 'Nested markets (if requested)' },
        ],
      },
      {
        id: 'get-series',
        method: 'GET',
        path: '/series/{series_ticker}',
        title: 'Get Series',
        description: 'Retrieve a series template by its ticker.',
        category: 'events',
        auth: false,
        params: [
          {
            name: 'series_ticker',
            type: 'string',
            required: true,
            description: 'Series ticker (e.g., "INXD")',
          },
        ],
        response: [
          { field: 'ticker', type: 'string', description: 'Series identifier' },
          { field: 'title', type: 'string', description: 'Series name' },
          {
            field: 'category',
            type: 'string',
            description: 'Category (economics, politics, sports)',
          },
          { field: 'frequency', type: 'string', description: 'How often events are created' },
          {
            field: 'settlement_sources',
            type: 'string[]',
            description: 'Data sources for settlement',
          },
        ],
      },
      {
        id: 'get-series-list',
        method: 'GET',
        path: '/series',
        title: 'Get Series List',
        description: 'Browse all available series templates.',
        category: 'events',
        auth: false,
        queryParams: [
          { name: 'limit', type: 'integer', required: false, description: 'Maximum results' },
          { name: 'cursor', type: 'string', required: false, description: 'Pagination cursor' },
        ],
      },
    ],
  },
  {
    id: 'orders',
    name: 'Orders',
    description: 'Create, modify, and cancel orders',
    icon: <Zap className="h-4 w-4" />,
    endpoints: [
      {
        id: 'create-order',
        method: 'POST',
        path: '/portfolio/orders',
        title: 'Create Order',
        description: 'Submit a new order. Supports limit and market orders.',
        category: 'orders',
        auth: true,
        bodyParams: [
          { name: 'ticker', type: 'string', required: true, description: 'Market ticker' },
          { name: 'action', type: 'string', required: true, description: '"buy" or "sell"' },
          { name: 'side', type: 'string', required: true, description: '"yes" or "no"' },
          { name: 'type', type: 'string', required: true, description: '"limit" or "market"' },
          { name: 'count', type: 'integer', required: true, description: 'Number of contracts' },
          {
            name: 'yes_price',
            type: 'integer',
            required: false,
            description: 'Limit price in cents (for limit orders)',
          },
          {
            name: 'client_order_id',
            type: 'string',
            required: false,
            description: 'Your custom order ID',
          },
          {
            name: 'expiration_ts',
            type: 'integer',
            required: false,
            description: 'Order expiration (Unix timestamp)',
          },
        ],
        response: [{ field: 'order', type: 'Order', description: 'Created order object' }],
        example: {
          request: `POST /portfolio/orders
{
  "ticker": "INXD-25JAN21-B5900",
  "action": "buy",
  "side": "yes",
  "type": "limit",
  "count": 100,
  "yes_price": 60
}`,
          response: `{
  "order": {
    "order_id": "ord_abc123",
    "ticker": "INXD-25JAN21-B5900",
    "action": "buy",
    "side": "yes",
    "type": "limit",
    "count": 100,
    "remaining_count": 100,
    "yes_price": 60,
    "status": "resting",
    "created_time": "2025-01-21T14:30:00Z"
  }
}`,
        },
        notes: [
          'Maximum 200,000 open orders per user',
          'For market orders, omit yes_price',
          'Use client_order_id for idempotency',
        ],
        rateLimit: '100 requests/second',
      },
      {
        id: 'get-orders',
        method: 'GET',
        path: '/portfolio/orders',
        title: 'Get Orders',
        description: 'Retrieve your orders with optional filters.',
        category: 'orders',
        auth: true,
        queryParams: [
          {
            name: 'ticker',
            type: 'string',
            required: false,
            description: 'Filter by market ticker',
          },
          {
            name: 'status',
            type: 'string',
            required: false,
            description: 'resting, canceled, or executed',
          },
          { name: 'limit', type: 'integer', required: false, description: 'Max results' },
          { name: 'cursor', type: 'string', required: false, description: 'Pagination cursor' },
        ],
        response: [
          { field: 'orders', type: 'Order[]', description: 'Array of order objects' },
          { field: 'cursor', type: 'string', description: 'Pagination cursor' },
        ],
      },
      {
        id: 'get-order',
        method: 'GET',
        path: '/portfolio/orders/{order_id}',
        title: 'Get Order',
        description: 'Retrieve a single order by ID.',
        category: 'orders',
        auth: true,
        params: [{ name: 'order_id', type: 'string', required: true, description: 'Order ID' }],
      },
      {
        id: 'amend-order',
        method: 'PUT',
        path: '/portfolio/orders/{order_id}',
        title: 'Amend Order',
        description: "Modify an existing order's price and/or quantity.",
        category: 'orders',
        auth: true,
        params: [
          { name: 'order_id', type: 'string', required: true, description: 'Order ID to amend' },
        ],
        bodyParams: [
          { name: 'count', type: 'integer', required: false, description: 'New contract count' },
          {
            name: 'yes_price',
            type: 'integer',
            required: false,
            description: 'New price in cents',
          },
        ],
        notes: ['Amending price may change queue position', 'Can only amend resting orders'],
      },
      {
        id: 'cancel-order',
        method: 'DELETE',
        path: '/portfolio/orders/{order_id}',
        title: 'Cancel Order',
        description: 'Cancel a resting order.',
        category: 'orders',
        auth: true,
        params: [
          { name: 'order_id', type: 'string', required: true, description: 'Order ID to cancel' },
        ],
        response: [
          { field: 'order', type: 'Order', description: 'Canceled order with status=canceled' },
        ],
      },
      {
        id: 'batch-create-orders',
        method: 'POST',
        path: '/portfolio/orders/batched',
        title: 'Batch Create Orders',
        description: 'Submit up to 20 orders in a single request.',
        category: 'orders',
        auth: true,
        bodyParams: [
          {
            name: 'orders',
            type: 'CreateOrderRequest[]',
            required: true,
            description: 'Array of order requests (max 20)',
          },
        ],
        response: [{ field: 'orders', type: 'Order[]', description: 'Array of created orders' }],
        notes: [
          'Atomic: all orders succeed or all fail',
          'More efficient than individual requests',
        ],
        rateLimit: '20 requests/second',
      },
      {
        id: 'batch-cancel-orders',
        method: 'DELETE',
        path: '/portfolio/orders/batched',
        title: 'Batch Cancel Orders',
        description: 'Cancel up to 20 orders in a single request.',
        category: 'orders',
        auth: true,
        bodyParams: [
          {
            name: 'order_ids',
            type: 'string[]',
            required: true,
            description: 'Array of order IDs (max 20)',
          },
        ],
      },
    ],
  },
  {
    id: 'portfolio',
    name: 'Portfolio',
    description: 'Positions, fills, balance, and settlements',
    icon: <Server className="h-4 w-4" />,
    endpoints: [
      {
        id: 'get-balance',
        method: 'GET',
        path: '/portfolio/balance',
        title: 'Get Balance',
        description: 'Retrieve your account balance and portfolio value.',
        category: 'portfolio',
        auth: true,
        response: [
          { field: 'balance', type: 'integer', description: 'Available cash in cents' },
          {
            field: 'portfolio_value',
            type: 'integer',
            description: 'Total account value in cents',
          },
          { field: 'payout', type: 'integer', description: 'Pending payouts in cents' },
        ],
        example: {
          response: `{
  "balance": 150000,
  "portfolio_value": 182500,
  "payout": 0
}`,
        },
        notes: ['All monetary values are in cents (divide by 100 for dollars)'],
      },
      {
        id: 'get-positions',
        method: 'GET',
        path: '/portfolio/positions',
        title: 'Get Positions',
        description: 'Retrieve your current market positions.',
        category: 'portfolio',
        auth: true,
        queryParams: [
          {
            name: 'ticker',
            type: 'string',
            required: false,
            description: 'Filter by market ticker',
          },
          {
            name: 'event_ticker',
            type: 'string',
            required: false,
            description: 'Filter by event ticker',
          },
          { name: 'limit', type: 'integer', required: false, description: 'Max results' },
          { name: 'cursor', type: 'string', required: false, description: 'Pagination cursor' },
        ],
        response: [
          {
            field: 'market_positions',
            type: 'Position[]',
            description: 'Array of position objects',
          },
          { field: 'cursor', type: 'string', description: 'Pagination cursor' },
        ],
        example: {
          response: `{
  "market_positions": [
    {
      "ticker": "INXD-25JAN21-B5900",
      "position": 100,
      "market_exposure": 6000,
      "realized_pnl": 250,
      "total_traded": 150,
      "resting_orders_count": 2
    }
  ]
}`,
        },
        notes: [
          'Positive position = long YES',
          'Negative position = long NO',
          'market_exposure is in cents',
        ],
      },
      {
        id: 'get-fills',
        method: 'GET',
        path: '/portfolio/fills',
        title: 'Get Fills',
        description: 'Retrieve your trade execution history.',
        category: 'portfolio',
        auth: true,
        queryParams: [
          {
            name: 'ticker',
            type: 'string',
            required: false,
            description: 'Filter by market ticker',
          },
          { name: 'order_id', type: 'string', required: false, description: 'Filter by order ID' },
          { name: 'min_ts', type: 'integer', required: false, description: 'Minimum timestamp' },
          { name: 'max_ts', type: 'integer', required: false, description: 'Maximum timestamp' },
          { name: 'limit', type: 'integer', required: false, description: 'Max results' },
          { name: 'cursor', type: 'string', required: false, description: 'Pagination cursor' },
        ],
        response: [
          { field: 'fills', type: 'Fill[]', description: 'Array of fill objects' },
          { field: 'cursor', type: 'string', description: 'Pagination cursor' },
        ],
        example: {
          response: `{
  "fills": [
    {
      "trade_id": "fill_xyz789",
      "order_id": "ord_abc123",
      "ticker": "INXD-25JAN21-B5900",
      "side": "yes",
      "action": "buy",
      "count": 50,
      "yes_price": 60,
      "created_time": "2025-01-21T14:30:05Z",
      "is_taker": true
    }
  ]
}`,
        },
      },
      {
        id: 'get-settlements',
        method: 'GET',
        path: '/portfolio/settlements',
        title: 'Get Settlements',
        description: 'Retrieve your settlement history.',
        category: 'portfolio',
        auth: true,
        queryParams: [
          { name: 'limit', type: 'integer', required: false, description: 'Max results' },
          { name: 'cursor', type: 'string', required: false, description: 'Pagination cursor' },
        ],
        response: [
          {
            field: 'settlements',
            type: 'Settlement[]',
            description: 'Array of settlement records',
          },
        ],
        example: {
          response: `{
  "settlements": [
    {
      "ticker": "INXD-25JAN20-B5850",
      "settled_position": 100,
      "market_result": "yes",
      "revenue": 10000,
      "settled_time": "2025-01-20T21:15:00Z"
    }
  ]
}`,
        },
      },
    ],
  },
  {
    id: 'websocket',
    name: 'WebSocket',
    description: 'Real-time streaming data',
    icon: <Zap className="h-4 w-4" />,
    endpoints: [
      {
        id: 'ws-connect',
        method: 'GET',
        path: 'wss://trading-api.kalshi.com/trade-api/ws/v2',
        title: 'WebSocket Connection',
        description: 'Connect to the real-time WebSocket feed for streaming updates.',
        category: 'websocket',
        auth: true,
        notes: [
          'Authentication via token in connection URL or first message',
          'Ping/Pong frames every 10 seconds for keep-alive',
          'Automatic reconnection recommended',
        ],
      },
      {
        id: 'ws-ticker',
        method: 'GET',
        path: 'subscribe: ticker',
        title: 'Ticker Channel',
        description: 'Subscribe to price, volume, and open interest updates.',
        category: 'websocket',
        auth: false,
        example: {
          request: `{
  "type": "subscribe",
  "channels": ["ticker"],
  "market_tickers": ["INXD-25JAN21-B5900"]
}`,
          response: `{
  "type": "ticker",
  "msg": {
    "ticker": "INXD-25JAN21-B5900",
    "yes_bid": 62,
    "yes_ask": 64,
    "last_price": 63,
    "volume": 15421
  }
}`,
        },
      },
      {
        id: 'ws-orderbook',
        method: 'GET',
        path: 'subscribe: orderbook_delta',
        title: 'Orderbook Channel',
        description: 'Subscribe to orderbook updates (snapshots + deltas).',
        category: 'websocket',
        auth: false,
        example: {
          request: `{
  "type": "subscribe",
  "channels": ["orderbook_delta"],
  "market_tickers": ["INXD-25JAN21-B5900"]
}`,
          response: `{
  "type": "orderbook_delta",
  "msg": {
    "ticker": "INXD-25JAN21-B5900",
    "side": "yes",
    "price": 62,
    "delta": 100
  }
}`,
        },
        notes: [
          'First message is a snapshot, then deltas',
          'Delta can be positive (add) or negative (remove)',
        ],
      },
      {
        id: 'ws-trade',
        method: 'GET',
        path: 'subscribe: trade',
        title: 'Trade Channel',
        description: 'Subscribe to public trade notifications.',
        category: 'websocket',
        auth: false,
      },
      {
        id: 'ws-fill',
        method: 'GET',
        path: 'subscribe: fill',
        title: 'Fill Channel',
        description: 'Subscribe to your order fill notifications.',
        category: 'websocket',
        auth: true,
        notes: ['Requires authentication', 'Only receives your own fills'],
      },
      {
        id: 'ws-order',
        method: 'GET',
        path: 'subscribe: order',
        title: 'Order Channel',
        description: 'Subscribe to your order status updates.',
        category: 'websocket',
        auth: true,
        notes: ['Requires authentication', 'Receives order state changes'],
      },
    ],
  },
  {
    id: 'auth',
    name: 'Authentication',
    description: 'API key management and auth',
    icon: <Lock className="h-4 w-4" />,
    endpoints: [
      {
        id: 'login',
        method: 'POST',
        path: '/login',
        title: 'Login',
        description: 'Authenticate with email/password to get a session token.',
        category: 'auth',
        auth: false,
        bodyParams: [
          { name: 'email', type: 'string', required: true, description: 'Account email' },
          { name: 'password', type: 'string', required: true, description: 'Account password' },
        ],
        response: [
          { field: 'token', type: 'string', description: 'Session token for API calls' },
          { field: 'member_id', type: 'string', description: 'Your user ID' },
        ],
        notes: [
          'Session tokens expire after inactivity',
          'Use API keys for long-running applications',
        ],
      },
      {
        id: 'create-api-key',
        method: 'POST',
        path: '/users/api-keys',
        title: 'Create API Key',
        description: 'Generate an RSA key pair for API authentication.',
        category: 'auth',
        auth: true,
        response: [
          { field: 'api_key', type: 'string', description: 'API key ID' },
          {
            field: 'private_key',
            type: 'string',
            description: 'RSA private key (PEM format) - SAVE THIS!',
          },
        ],
        notes: [
          'Private key is only returned once',
          'Store private key securely',
          'Required for "Premier" or "Market Maker" access',
        ],
      },
      {
        id: 'get-api-keys',
        method: 'GET',
        path: '/users/api-keys',
        title: 'Get API Keys',
        description: 'List all API keys for your account.',
        category: 'auth',
        auth: true,
      },
      {
        id: 'delete-api-key',
        method: 'DELETE',
        path: '/users/api-keys/{api_key_id}',
        title: 'Delete API Key',
        description: 'Permanently delete an API key.',
        category: 'auth',
        auth: true,
        params: [
          { name: 'api_key_id', type: 'string', required: true, description: 'API key to delete' },
        ],
      },
    ],
  },
];

// =============================================================================
// HELPER COMPONENTS
// =============================================================================

function MethodBadge({ method }: { method: HttpMethod }) {
  const colors: Record<HttpMethod, string> = {
    GET: 'bg-green-500/20 text-green-400 border-green-500/30',
    POST: 'bg-blue-500/20 text-blue-400 border-blue-500/30',
    PUT: 'bg-amber-500/20 text-amber-400 border-amber-500/30',
    DELETE: 'bg-red-500/20 text-red-400 border-red-500/30',
  };

  return (
    <span
      className={`rounded border px-2 py-0.5 font-mono text-xs font-semibold ${colors[method]}`}
    >
      {method}
    </span>
  );
}

function CopyButton({ text }: { text: string }) {
  const [copied, setCopied] = useState(false);

  const handleCopy = () => {
    navigator.clipboard.writeText(text);
    setCopied(true);
    setTimeout(() => setCopied(false), 2000);
  };

  return (
    <button
      onClick={handleCopy}
      className="hover:bg-muted/50 rounded p-1 transition-colors"
      title="Copy to clipboard"
    >
      {copied ? (
        <Check className="h-4 w-4 text-green-400" />
      ) : (
        <Copy className="text-muted-foreground h-4 w-4" />
      )}
    </button>
  );
}

function ParamTable({ params, title }: { params: EndpointParam[]; title: string }) {
  return (
    <div className="mt-4">
      <div className="text-muted-foreground mb-2 text-sm font-semibold">{title}</div>
      <div className="overflow-hidden rounded-lg border">
        <table className="w-full text-sm">
          <thead className="bg-muted/50">
            <tr>
              <th className="px-3 py-2 text-left font-medium">Name</th>
              <th className="px-3 py-2 text-left font-medium">Type</th>
              <th className="px-3 py-2 text-left font-medium">Required</th>
              <th className="px-3 py-2 text-left font-medium">Description</th>
            </tr>
          </thead>
          <tbody className="divide-border divide-y">
            {params.map((param) => (
              <tr key={param.name}>
                <td className="text-primary px-3 py-2 font-mono">{param.name}</td>
                <td className="text-muted-foreground px-3 py-2 font-mono text-xs">{param.type}</td>
                <td className="px-3 py-2">
                  {param.required ? (
                    <span className="text-xs text-amber-400">Required</span>
                  ) : (
                    <span className="text-muted-foreground text-xs">Optional</span>
                  )}
                </td>
                <td className="text-muted-foreground px-3 py-2">
                  {param.description}
                  {param.example && (
                    <span className="text-foreground/70 ml-1">(e.g., {param.example})</span>
                  )}
                </td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
    </div>
  );
}

function ResponseTable({ response }: { response: EndpointResponse[] }) {
  return (
    <div className="mt-4">
      <div className="text-muted-foreground mb-2 text-sm font-semibold">Response Fields</div>
      <div className="overflow-hidden rounded-lg border">
        <table className="w-full text-sm">
          <thead className="bg-muted/50">
            <tr>
              <th className="px-3 py-2 text-left font-medium">Field</th>
              <th className="px-3 py-2 text-left font-medium">Type</th>
              <th className="px-3 py-2 text-left font-medium">Description</th>
            </tr>
          </thead>
          <tbody className="divide-border divide-y">
            {response.map((field) => (
              <tr key={field.field}>
                <td className="text-primary px-3 py-2 font-mono">{field.field}</td>
                <td className="text-muted-foreground px-3 py-2 font-mono text-xs">{field.type}</td>
                <td className="text-muted-foreground px-3 py-2">{field.description}</td>
              </tr>
            ))}
          </tbody>
        </table>
      </div>
    </div>
  );
}

function CodeBlock({ code, title }: { code: string; title?: string }) {
  return (
    <div className="mt-4">
      {title && <div className="text-muted-foreground mb-2 text-sm font-semibold">{title}</div>}
      <div className="relative">
        <div className="absolute right-2 top-2">
          <CopyButton text={code} />
        </div>
        <pre className="bg-muted/50 overflow-x-auto rounded-lg p-4 font-mono text-xs">{code}</pre>
      </div>
    </div>
  );
}

// =============================================================================
// ENDPOINT DETAIL COMPONENT
// =============================================================================

function EndpointDetail({ endpoint }: { endpoint: Endpoint }) {
  const [isExpanded, setIsExpanded] = useState(false);

  return (
    <div className="overflow-hidden rounded-lg border">
      {/* Header */}
      <button
        onClick={() => setIsExpanded(!isExpanded)}
        className="hover:bg-muted/30 flex w-full items-center gap-3 px-4 py-3 text-left transition-colors"
      >
        {isExpanded ? (
          <ChevronDown className="text-muted-foreground h-4 w-4 flex-shrink-0" />
        ) : (
          <ChevronRight className="text-muted-foreground h-4 w-4 flex-shrink-0" />
        )}
        <MethodBadge method={endpoint.method} />
        <code className="text-foreground font-mono text-sm">{endpoint.path}</code>
        {endpoint.auth && <Lock className="h-3 w-3 flex-shrink-0 text-amber-400" />}
        <span className="text-muted-foreground ml-auto text-sm">{endpoint.title}</span>
      </button>

      {/* Expanded content */}
      {isExpanded && (
        <div className="bg-card/50 border-t px-4 pb-4">
          <p className="text-muted-foreground mt-4 text-sm">{endpoint.description}</p>

          {endpoint.auth && (
            <div className="mt-3 flex items-center gap-2 rounded border border-amber-500/30 bg-amber-500/10 px-3 py-2 text-sm text-amber-400">
              <Lock className="h-4 w-4" />
              Authentication required
            </div>
          )}

          {endpoint.rateLimit && (
            <div className="mt-3 rounded border border-blue-500/30 bg-blue-500/10 px-3 py-2 text-sm text-blue-400">
              Rate limit: {endpoint.rateLimit}
            </div>
          )}

          {endpoint.params && endpoint.params.length > 0 && (
            <ParamTable
              params={endpoint.params}
              title="Path Parameters"
            />
          )}

          {endpoint.queryParams && endpoint.queryParams.length > 0 && (
            <ParamTable
              params={endpoint.queryParams}
              title="Query Parameters"
            />
          )}

          {endpoint.bodyParams && endpoint.bodyParams.length > 0 && (
            <ParamTable
              params={endpoint.bodyParams}
              title="Request Body"
            />
          )}

          {endpoint.response && endpoint.response.length > 0 && (
            <ResponseTable response={endpoint.response} />
          )}

          {endpoint.example?.request && (
            <CodeBlock
              code={endpoint.example.request}
              title="Example Request"
            />
          )}

          {endpoint.example?.response && (
            <CodeBlock
              code={endpoint.example.response}
              title="Example Response"
            />
          )}

          {endpoint.notes && endpoint.notes.length > 0 && (
            <div className="mt-4">
              <div className="text-muted-foreground mb-2 text-sm font-semibold">Notes</div>
              <ul className="space-y-1">
                {endpoint.notes.map((note, i) => (
                  <li
                    key={i}
                    className="text-muted-foreground flex items-start gap-2 text-sm"
                  >
                    <span className="text-primary">•</span>
                    {note}
                  </li>
                ))}
              </ul>
            </div>
          )}
        </div>
      )}
    </div>
  );
}

// =============================================================================
// CATEGORY SECTION COMPONENT
// =============================================================================

function CategorySection({ category }: { category: EndpointCategory }) {
  const [isExpanded, setIsExpanded] = useState(true);

  return (
    <div className="mb-8">
      <button
        onClick={() => setIsExpanded(!isExpanded)}
        className="mb-4 flex w-full items-center gap-3 text-left"
      >
        {isExpanded ? (
          <ChevronDown className="text-muted-foreground h-5 w-5" />
        ) : (
          <ChevronRight className="text-muted-foreground h-5 w-5" />
        )}
        <div className="bg-primary/10 flex h-8 w-8 items-center justify-center rounded-lg">
          {category.icon}
        </div>
        <div>
          <h3 className="font-semibold">{category.name}</h3>
          <p className="text-muted-foreground text-sm">{category.description}</p>
        </div>
        <span className="text-muted-foreground ml-auto text-sm">
          {category.endpoints.length} endpoints
        </span>
      </button>

      {isExpanded && (
        <div className="ml-11 space-y-2">
          {category.endpoints.map((endpoint) => (
            <EndpointDetail
              key={endpoint.id}
              endpoint={endpoint}
            />
          ))}
        </div>
      )}
    </div>
  );
}

// =============================================================================
// QUICK REFERENCE WIDGET
// =============================================================================

function QuickReferenceWidget() {
  const [selectedCategory, setSelectedCategory] = useState<string | null>(null);

  const categories = [
    { id: 'markets', name: 'Markets', color: 'bg-green-500' },
    { id: 'events', name: 'Events', color: 'bg-blue-500' },
    { id: 'orders', name: 'Orders', color: 'bg-amber-500' },
    { id: 'portfolio', name: 'Portfolio', color: 'bg-purple-500' },
    { id: 'websocket', name: 'WebSocket', color: 'bg-cyan-500' },
    { id: 'auth', name: 'Auth', color: 'bg-red-500' },
  ];

  return (
    <div className="bg-card rounded-lg border p-6">
      <div className="text-muted-foreground mb-4 text-sm font-semibold uppercase">
        Quick Reference
      </div>

      <div className="mb-4 grid grid-cols-3 gap-2">
        {categories.map((cat) => (
          <button
            key={cat.id}
            onClick={() => setSelectedCategory(selectedCategory === cat.id ? null : cat.id)}
            className={`rounded-lg border p-2 text-sm transition-all ${
              selectedCategory === cat.id
                ? 'border-primary bg-primary/10'
                : 'border-muted hover:border-muted-foreground/50'
            }`}
          >
            <div className={`h-2 w-2 rounded-full ${cat.color} mr-2 inline-block`} />
            {cat.name}
          </button>
        ))}
      </div>

      {selectedCategory && (
        <div className="bg-muted/30 rounded-lg p-4">
          <div className="mb-2 text-sm font-medium">
            {categories.find((c) => c.id === selectedCategory)?.name} Endpoints
          </div>
          <div className="space-y-1">
            {apiCategories
              .find((c) => c.id === selectedCategory)
              ?.endpoints.map((ep) => (
                <div
                  key={ep.id}
                  className="flex items-center gap-2 text-xs"
                >
                  <MethodBadge method={ep.method} />
                  <code className="text-muted-foreground">{ep.path}</code>
                </div>
              ))}
          </div>
        </div>
      )}
    </div>
  );
}

// =============================================================================
// BASE URL WIDGET
// =============================================================================

function BaseUrlWidget() {
  const [environment, setEnvironment] = useState<'production' | 'demo'>('production');

  const urls = {
    production: {
      rest: 'https://trading-api.kalshi.com/trade-api/v2',
      ws: 'wss://trading-api.kalshi.com/trade-api/ws/v2',
    },
    demo: {
      rest: 'https://demo-api.kalshi.co/trade-api/v2',
      ws: 'wss://demo-api.kalshi.co/trade-api/ws/v2',
    },
  };

  const current = urls[environment];

  return (
    <div className="bg-card rounded-lg border p-6">
      <div className="text-muted-foreground mb-4 text-sm font-semibold uppercase">
        API Base URLs
      </div>

      <div className="mb-4 flex gap-2">
        <button
          onClick={() => setEnvironment('production')}
          className={`rounded px-3 py-1.5 text-sm transition-colors ${
            environment === 'production'
              ? 'border border-green-500/30 bg-green-500/20 text-green-400'
              : 'bg-muted text-muted-foreground hover:text-foreground'
          }`}
        >
          Production
        </button>
        <button
          onClick={() => setEnvironment('demo')}
          className={`rounded px-3 py-1.5 text-sm transition-colors ${
            environment === 'demo'
              ? 'border border-blue-500/30 bg-blue-500/20 text-blue-400'
              : 'bg-muted text-muted-foreground hover:text-foreground'
          }`}
        >
          Demo
        </button>
      </div>

      <div className="space-y-3">
        <div>
          <div className="text-muted-foreground mb-1 text-xs">REST API</div>
          <div className="bg-muted/50 flex items-center gap-2 rounded px-3 py-2">
            <code className="text-foreground flex-1 font-mono text-sm">{current.rest}</code>
            <CopyButton text={current.rest} />
          </div>
        </div>
        <div>
          <div className="text-muted-foreground mb-1 text-xs">WebSocket</div>
          <div className="bg-muted/50 flex items-center gap-2 rounded px-3 py-2">
            <code className="text-foreground flex-1 font-mono text-sm">{current.ws}</code>
            <CopyButton text={current.ws} />
          </div>
        </div>
      </div>

      {environment === 'demo' && (
        <p className="text-muted-foreground mt-3 text-xs">
          Demo environment uses play money. Perfect for testing without risk.
        </p>
      )}
    </div>
  );
}

// =============================================================================
// AUTHENTICATION FLOW WIDGET
// =============================================================================

function AuthFlowWidget() {
  const [step, setStep] = useState(1);

  const steps = [
    {
      title: 'Generate API Key',
      description: 'Create an RSA key pair via the API or Kalshi dashboard',
      code: 'POST /users/api-keys\n→ Returns api_key + private_key (save this!)',
    },
    {
      title: 'Sign Requests',
      description: 'Sign each request using RSA-PSS with your private key',
      code: `const signature = sign({
  timestamp: Date.now(),
  method: 'POST',
  path: '/portfolio/orders',
  body: JSON.stringify(order)
}, privateKey)`,
    },
    {
      title: 'Add Headers',
      description: 'Include authentication headers with every request',
      code: `KALSHI-ACCESS-KEY: <your-api-key>
KALSHI-ACCESS-SIGNATURE: <signature>
KALSHI-ACCESS-TIMESTAMP: <timestamp>`,
    },
  ];

  return (
    <div className="bg-card rounded-lg border p-6">
      <div className="text-muted-foreground mb-4 text-sm font-semibold uppercase">
        Authentication Flow
      </div>

      <div className="mb-6 flex items-center justify-between">
        {steps.map((s, i) => (
          <div
            key={i}
            className="flex items-center"
          >
            <button
              onClick={() => setStep(i + 1)}
              className={`flex h-8 w-8 items-center justify-center rounded-full text-sm font-semibold transition-colors ${
                step === i + 1
                  ? 'bg-primary text-primary-foreground'
                  : step > i + 1
                    ? 'border border-green-500/30 bg-green-500/20 text-green-400'
                    : 'bg-muted text-muted-foreground'
              }`}
            >
              {step > i + 1 ? <Check className="h-4 w-4" /> : i + 1}
            </button>
            {i < steps.length - 1 && (
              <div className={`mx-2 h-0.5 w-16 ${step > i + 1 ? 'bg-green-500/50' : 'bg-muted'}`} />
            )}
          </div>
        ))}
      </div>

      <div className="bg-muted/30 rounded-lg p-4">
        <div className="mb-1 font-medium">{steps[step - 1].title}</div>
        <p className="text-muted-foreground mb-3 text-sm">{steps[step - 1].description}</p>
        <pre className="bg-muted/50 overflow-x-auto rounded p-3 font-mono text-xs">
          {steps[step - 1].code}
        </pre>
      </div>

      <div className="mt-4 flex justify-between">
        <button
          onClick={() => setStep(Math.max(1, step - 1))}
          disabled={step === 1}
          className="bg-muted hover:bg-muted/80 rounded px-3 py-1.5 text-sm disabled:cursor-not-allowed disabled:opacity-50"
        >
          Previous
        </button>
        <button
          onClick={() => setStep(Math.min(3, step + 1))}
          disabled={step === 3}
          className="bg-primary hover:bg-primary/90 text-primary-foreground rounded px-3 py-1.5 text-sm disabled:cursor-not-allowed disabled:opacity-50"
        >
          Next
        </button>
      </div>
    </div>
  );
}

// =============================================================================
// SEARCH/FILTER WIDGET
// =============================================================================

function ApiSearchWidget({ onSearch }: { onSearch: (query: string) => void }) {
  const [query, setQuery] = useState('');

  const handleSearch = (value: string) => {
    setQuery(value);
    onSearch(value);
  };

  return (
    <div className="relative">
      <Search className="text-muted-foreground absolute left-3 top-1/2 h-4 w-4 -translate-y-1/2" />
      <input
        type="text"
        value={query}
        onChange={(e) => handleSearch(e.target.value)}
        placeholder="Search endpoints... (e.g., 'orders', 'GET /markets')"
        className="bg-card text-foreground placeholder:text-muted-foreground focus:ring-primary/50 w-full rounded-lg border py-3 pl-10 pr-4 focus:outline-none focus:ring-2"
      />
    </div>
  );
}

// =============================================================================
// MAIN LESSON COMPONENT
// =============================================================================

export function KalshiApiExplorerLesson() {
  const guide = getGuide('kalshi-api-explorer')!;
  const nextGuide = getNextGuide('kalshi-api-explorer');
  const [searchQuery, setSearchQuery] = useState('');

  // Filter categories and endpoints based on search
  const filteredCategories = useMemo(() => {
    if (!searchQuery.trim()) return apiCategories;

    const query = searchQuery.toLowerCase();
    return apiCategories
      .map((category) => ({
        ...category,
        endpoints: category.endpoints.filter(
          (ep) =>
            ep.title.toLowerCase().includes(query) ||
            ep.path.toLowerCase().includes(query) ||
            ep.description.toLowerCase().includes(query) ||
            ep.method.toLowerCase().includes(query)
        ),
      }))
      .filter((category) => category.endpoints.length > 0);
  }, [searchQuery]);

  const totalEndpoints = apiCategories.reduce((sum, cat) => sum + cat.endpoints.length, 0);
  const filteredEndpoints = filteredCategories.reduce((sum, cat) => sum + cat.endpoints.length, 0);

  return (
    <LessonLayout>
      {/* Header */}
      <LessonRow>
        <LessonHeader
          title={guide.title}
          description={guide.description}
          duration="Reference"
          category="Developer"
        />
      </LessonRow>

      {/* Objective */}
      <LessonRow
        aside={
          <div className="space-y-4">
            <BaseUrlWidget />
          </div>
        }
      >
        <ObjectiveBlock>
          This is your interactive reference for the Kalshi REST and WebSocket APIs. Browse
          endpoints, view request/response schemas, and copy example code. Use the search to quickly
          find what you need.
        </ObjectiveBlock>

        <div className="mt-6 flex items-center gap-4">
          <a
            href="https://docs.kalshi.com"
            target="_blank"
            rel="noopener noreferrer"
            className="text-primary inline-flex items-center gap-2 text-sm hover:underline"
          >
            <ExternalLink className="h-4 w-4" />
            Official Kalshi Docs
          </a>
          <span className="text-muted-foreground">•</span>
          <span className="text-muted-foreground text-sm">
            {totalEndpoints} endpoints documented
          </span>
        </div>
      </LessonRow>

      {/* Quick Reference */}
      <LessonRow
        aside={
          <TipBlock title="Getting Started">
            If you're new to the Kalshi API, start with the{' '}
            <a
              href="/kalshi-api-entities"
              className="text-primary hover:underline"
            >
              API Entity Model
            </a>{' '}
            guide to understand the data structures first.
          </TipBlock>
        }
      >
        <SectionHeader
          title="Quick Reference"
          subtitle="Overview of available API categories"
        />

        <QuickReferenceWidget />
      </LessonRow>

      {/* Authentication */}
      <LessonRow
        aside={
          <WarningBlock title="Private Key Security">
            Your RSA private key is only shown once when created. Store it securely and never expose
            it in client-side code or commit it to version control.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="Authentication"
          subtitle="How to authenticate API requests"
        />

        <p className="text-muted-foreground mb-6">
          Kalshi uses RSA-PSS signature authentication for trading endpoints. Public market data
          endpoints don't require authentication.
        </p>

        <AuthFlowWidget />
      </LessonRow>

      {/* API Explorer */}
      <LessonRow
        aside={
          <InsightBlock title="REST vs WebSocket">
            Use REST for operations (placing orders, querying data). Use WebSocket for real-time
            streaming (price updates, fills). Most apps need both.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="API Explorer"
          subtitle="Browse and search all endpoints"
        />

        <div className="mb-6">
          <ApiSearchWidget onSearch={setSearchQuery} />
          {searchQuery && (
            <p className="text-muted-foreground mt-2 text-sm">
              Showing {filteredEndpoints} of {totalEndpoints} endpoints
            </p>
          )}
        </div>

        {filteredCategories.length === 0 ? (
          <div className="text-muted-foreground py-12 text-center">
            No endpoints match "{searchQuery}"
          </div>
        ) : (
          filteredCategories.map((category) => (
            <CategorySection
              key={category.id}
              category={category}
            />
          ))
        )}
      </LessonRow>

      {/* Common Patterns */}
      <LessonRow
        aside={
          <TipBlock title="Pagination">
            Most list endpoints use cursor-based pagination. Save the cursor from each response to
            fetch the next page.
          </TipBlock>
        }
      >
        <SectionHeader
          title="Common Patterns"
          subtitle="Frequently used API patterns"
        />

        <div className="space-y-6">
          {/* Pagination Pattern */}
          <div className="bg-card rounded-lg border p-4">
            <div className="mb-2 font-medium">Cursor Pagination</div>
            <p className="text-muted-foreground mb-3 text-sm">
              Fetch paginated results by passing the cursor from the previous response.
            </p>
            <CodeBlock
              code={`// First request
GET /markets?limit=100

// Response includes cursor
{ "markets": [...], "cursor": "abc123" }

// Next page
GET /markets?limit=100&cursor=abc123`}
            />
          </div>

          {/* Error Handling Pattern */}
          <div className="bg-card rounded-lg border p-4">
            <div className="mb-2 font-medium">Error Responses</div>
            <p className="text-muted-foreground mb-3 text-sm">
              Errors follow a consistent format with error codes and messages.
            </p>
            <CodeBlock
              code={`{
  "error": {
    "code": "insufficient_balance",
    "message": "Not enough funds to place this order"
  }
}`}
            />
          </div>

          {/* WebSocket Pattern */}
          <div className="bg-card rounded-lg border p-4">
            <div className="mb-2 font-medium">WebSocket Subscriptions</div>
            <p className="text-muted-foreground mb-3 text-sm">
              Subscribe to channels after connecting. Unsubscribe when no longer needed.
            </p>
            <CodeBlock
              code={`// Subscribe
{ "type": "subscribe", "channels": ["ticker"], "market_tickers": ["INXD-25JAN21-B5900"] }

// Unsubscribe
{ "type": "unsubscribe", "channels": ["ticker"], "market_tickers": ["INXD-25JAN21-B5900"] }`}
            />
          </div>
        </div>
      </LessonRow>

      {/* Rate Limits */}
      <LessonRow>
        <SectionHeader
          title="Rate Limits"
          subtitle="API request limits and best practices"
        />

        <div className="grid grid-cols-2 gap-4">
          <div className="bg-card rounded-lg border p-4">
            <div className="mb-2 font-medium">Public Endpoints</div>
            <div className="text-primary mb-1 text-2xl font-bold">10 req/sec</div>
            <p className="text-muted-foreground text-sm">Market data, events, trades, orderbooks</p>
          </div>
          <div className="bg-card rounded-lg border p-4">
            <div className="mb-2 font-medium">Trading Endpoints</div>
            <div className="text-primary mb-1 text-2xl font-bold">100 req/sec</div>
            <p className="text-muted-foreground text-sm">Orders, positions, fills</p>
          </div>
          <div className="bg-card rounded-lg border p-4">
            <div className="mb-2 font-medium">Batch Operations</div>
            <div className="text-primary mb-1 text-2xl font-bold">20 req/sec</div>
            <p className="text-muted-foreground text-sm">Batch order create/cancel</p>
          </div>
          <div className="bg-card rounded-lg border p-4">
            <div className="mb-2 font-medium">WebSocket</div>
            <div className="text-primary mb-1 text-2xl font-bold">No limit</div>
            <p className="text-muted-foreground text-sm">Streaming data, ping every 10s</p>
          </div>
        </div>

        <div className="mt-6 rounded-lg border border-amber-500/30 bg-amber-500/10 p-4">
          <div className="mb-2 font-medium text-amber-400">Rate Limit Headers</div>
          <p className="text-muted-foreground text-sm">
            API responses include <code className="text-foreground">X-RateLimit-Remaining</code> and{' '}
            <code className="text-foreground">X-RateLimit-Reset</code> headers. Use these to
            implement backoff when approaching limits.
          </p>
        </div>
      </LessonRow>

      {/* Next Step */}
      {nextGuide && (
        <LessonRow>
          <NextStepBlock
            title={nextGuide.title}
            description={nextGuide.description}
            href={`/${nextGuide.slug}`}
          />
        </LessonRow>
      )}
    </LessonLayout>
  );
}
