/**
 * Lesson: Market Making Strategy
 *
 * How professional traders provide liquidity on the NO side.
 * This is the capstone lesson bringing together all previous concepts.
 * Expanded with deep explanations, historical context, and practical examples.
 */

import {
  Coins,
  Users,
  Zap,
  Target,
  TrendingUp,
  History,
  AlertTriangle,
  Lightbulb,
} from 'lucide-react';
import { Formula } from '@/components/atoms/Formula';
import { getGuide } from '@/data/guides';
import { LessonLayout, LessonRow, LessonHeader, SectionHeader } from '../LessonLayout';
import {
  ObjectiveBlock,
  ConstraintBlock,
  TipBlock,
  InsightBlock,
  WarningBlock,
  ConceptCard,
  ConceptBlock,
  ExampleBox,
  NextStepBlock,
  Term,
} from '../blocks';
import { GlossaryTooltip } from '@/components/atoms/GlossaryTooltip';
import { MarketFlowVisualizer } from '../widgets/MarketFlowVisualizer';
import { MarketMakerDashboardPreview } from '../widgets/MarketMakerDashboardPreview';

// Custom glossary terms for this lesson
const terms = {
  marketMaker: {
    term: 'Market Maker',
    definition:
      'A trader who provides liquidity by posting both buy and sell orders, profiting from the spread between them.',
    category: 'Strategy',
  },
  marketTaker: {
    term: 'Market Taker',
    definition:
      'A trader who removes liquidity by filling existing orders. Wants immediate execution.',
    category: 'Strategy',
  },
  retailFlow: {
    term: 'Retail Flow',
    definition:
      'Order flow from recreational (non-professional) traders. Predictable - they typically bet YES on teams they like.',
    category: 'Strategy',
  },
  edge: {
    term: 'Edge',
    definition: 'Your advantage over the market. Positive edge = profitable over time.',
    category: 'Strategy',
  },
  ev: {
    term: 'Expected Value (EV)',
    definition:
      'The average outcome of a bet if repeated many times. Positive EV (+EV) bets are profitable long-term.',
    category: 'Strategy',
  },
  queuePosition: {
    term: 'Queue Position',
    definition:
      'Your place in line at a price level. First in, first out - early orders fill first.',
    category: 'Order Book',
  },
  liquidity: {
    term: 'Liquidity',
    definition:
      'How easily you can trade without moving the price. Market makers add liquidity; takers remove it.',
    category: 'Markets',
  },
  counterparty: {
    term: 'Counterparty',
    definition: 'The other side of your trade. When you buy, someone sells to you.',
    category: 'Trading',
  },
};

export function MarketMakingStrategyLesson() {
  const guide = getGuide('market-making-strategy')!;

  return (
    <LessonLayout>
      {/* Header */}
      <LessonRow>
        <LessonHeader
          title={guide.title}
          description={guide.description}
          duration="20 min"
          category="Strategies"
        />
      </LessonRow>

      {/* Objective */}
      <LessonRow
        aside={
          <ConstraintBlock
            title="What We'll Cover"
            items={[
              'What market makers do and how they profit',
              'Why recreational flow is predictable',
              'The NO side strategy for moneylines',
              'Queue position vs price tradeoffs',
              "When this strategy does and doesn't apply",
            ]}
          />
        }
      >
        <ObjectiveBlock>
          By the end of this lesson, you'll understand the core market making strategy used by
          professional traders: positioning on the NO side of the underdog's market to capture
          predictable recreational flow. This is the capstone lesson that ties everything together.
        </ObjectiveBlock>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 1: WHAT IS A MARKET MAKER */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="A Different Game">
            Market makers don't care who wins the game. Their edge comes from the spread and flow
            dynamics, not from predicting outcomes.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="What is a Market Maker?"
          subtitle="Understanding the professionals"
        />

        <div className="mb-6 flex items-start gap-3">
          <div className="bg-primary/10 flex h-10 w-10 flex-shrink-0 items-center justify-center rounded-full">
            <History className="text-primary h-5 w-5" />
          </div>
          <div>
            <h4 className="font-semibold">Etymology & History</h4>
            <p className="text-muted-foreground mt-1 text-sm">
              The term "<GlossaryTooltip term={terms.marketMaker}>market maker</GlossaryTooltip>"
              emerged with electronic trading in the 1980s-90s. On the NYSE, they were originally
              called "specialists" - professionals assigned to maintain orderly markets in specific
              stocks by always being willing to buy and sell.
            </p>
          </div>
        </div>

        <p className="text-muted-foreground leading-relaxed">
          A <GlossaryTooltip term={terms.marketMaker}>market maker</GlossaryTooltip> is
          fundamentally different from a bettor. While bettors have opinions about outcomes, market
          makers profit by <em>facilitating trades</em> for others - capturing the spread between
          buy and sell prices.
        </p>

        <div className="mt-6 grid gap-4 sm:grid-cols-2">
          <div className="bg-card rounded-lg border p-4">
            <div className="mb-3 flex items-center gap-2">
              <Users className="h-5 w-5 text-rose-400" />
              <h4 className="font-semibold">Market Takers (Recreational Bettors)</h4>
            </div>
            <p className="text-muted-foreground mb-3 text-sm">
              <GlossaryTooltip term={terms.marketTaker}>Takers</GlossaryTooltip> have opinions and
              want to express them immediately. They "take" liquidity from the order book.
            </p>
            <ul className="text-muted-foreground space-y-1 text-xs">
              <li className="flex items-start gap-2">
                <span className="text-rose-400">•</span>
                <span>Have a view: "Lakers will win tonight"</span>
              </li>
              <li className="flex items-start gap-2">
                <span className="text-rose-400">•</span>
                <span>Use market orders for immediate execution</span>
              </li>
              <li className="flex items-start gap-2">
                <span className="text-rose-400">•</span>
                <span>Almost always bet YES on their preferred team</span>
              </li>
              <li className="flex items-start gap-2">
                <span className="text-rose-400">•</span>
                <span>Pay the spread to get filled immediately</span>
              </li>
            </ul>
          </div>
          <div className="bg-card rounded-lg border p-4">
            <div className="mb-3 flex items-center gap-2">
              <Coins className="h-5 w-5 text-emerald-400" />
              <h4 className="font-semibold">Market Makers (Professionals)</h4>
            </div>
            <p className="text-muted-foreground mb-3 text-sm">
              <GlossaryTooltip term={terms.marketMaker}>Makers</GlossaryTooltip> provide liquidity
              by posting orders. They profit from spread capture, not outcome prediction.
            </p>
            <ul className="text-muted-foreground space-y-1 text-xs">
              <li className="flex items-start gap-2">
                <span className="text-emerald-400">•</span>
                <span>No strong directional opinion</span>
              </li>
              <li className="flex items-start gap-2">
                <span className="text-emerald-400">•</span>
                <span>Post limit orders and wait</span>
              </li>
              <li className="flex items-start gap-2">
                <span className="text-emerald-400">•</span>
                <span>Position to capture predictable flow</span>
              </li>
              <li className="flex items-start gap-2">
                <span className="text-emerald-400">•</span>
                <span>Earn the spread when orders get filled</span>
              </li>
            </ul>
          </div>
        </div>

        <div className="from-primary/10 via-primary/5 border-primary/30 mt-6 rounded-lg border bg-gradient-to-br to-transparent p-4">
          <h4 className="mb-2 font-semibold">The Key Distinction</h4>
          <p className="text-muted-foreground text-sm">
            <strong>Bettors ask:</strong> "Who will win?" → They take positions based on outcome
            predictions.
            <br />
            <strong>Market makers ask:</strong> "Where will flow go?" → They position to be the
            counterparty.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 2: THE RETAIL FLOW PATTERN */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <ConceptBlock title="Retail Flow">
            <GlossaryTooltip term={terms.retailFlow}>Retail flow</GlossaryTooltip> refers to orders
            from recreational bettors. It's predictable because fans think about teams, not
            contracts.
          </ConceptBlock>
        }
      >
        <SectionHeader
          title="The Retail Flow Pattern"
          subtitle="Why recreational bettors are predictable"
        />

        <p className="text-muted-foreground leading-relaxed">
          Here's the insight that makes market making profitable:{' '}
          <GlossaryTooltip term={terms.retailFlow}>retail flow</GlossaryTooltip> is predictable.
          When a Lakers fan wants to bet on the Lakers, they don't think about order books or
          equivalent positions. They think:
        </p>

        <div className="bg-muted/30 mt-6 rounded-lg p-5">
          <div className="text-center">
            <p className="mb-4 text-lg font-medium">"I like the Lakers, so I'll bet Lakers YES"</p>
            <p className="text-muted-foreground text-sm">
              They almost <em>never</em> think: "I'll bet Celtics NO" — even though it's the same
              thing.
            </p>
          </div>
        </div>

        <ExampleBox title="Why Fans Think in YES">
          <div className="space-y-4">
            <p className="text-sm">
              <strong>Psychology matters:</strong> Betting YES on your team feels like supporting
              them. Betting NO on the opponent feels... weird. It's the same position, but it
              doesn't
              <em>feel</em> the same.
            </p>

            <div className="grid gap-3 sm:grid-cols-2">
              <div className="rounded border border-emerald-500/30 bg-emerald-500/10 p-3">
                <div className="mb-2 text-sm font-medium text-emerald-400">What Fans Do</div>
                <ul className="text-muted-foreground space-y-1 text-xs">
                  <li>• "Lakers YES!" - feels like cheering</li>
                  <li>• Intuitive, emotionally satisfying</li>
                  <li>• "I'm betting on my team to win"</li>
                </ul>
              </div>
              <div className="rounded border border-rose-500/30 bg-rose-500/10 p-3">
                <div className="mb-2 text-sm font-medium text-rose-400">What Fans Don't Do</div>
                <ul className="text-muted-foreground space-y-1 text-xs">
                  <li>• "Celtics NO" - feels backwards</li>
                  <li>• Requires market knowledge</li>
                  <li>• "I'm betting against the other team?"</li>
                </ul>
              </div>
            </div>

            <p className="text-muted-foreground border-t pt-3 text-xs">
              This psychological pattern creates a flow asymmetry that market makers exploit.
            </p>
          </div>
        </ExampleBox>

        <div className="mt-6 space-y-4">
          <Term term="Flow Asymmetry">
            Because retail bettors gravitate to YES positions, order flow is uneven. The YES side of
            popular teams gets more action than the NO side of opponents - even though they're
            economically equivalent.
          </Term>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 3: THE NO SIDE STRATEGY */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="The Mental Model">
            Think of it like fishing: retail money flows into YES on popular teams. Market makers
            position their nets on the other side to catch that flow.
          </TipBlock>
        }
      >
        <SectionHeader
          title="The NO Side Strategy"
          subtitle="Positioning to capture retail flow"
        />

        <p className="text-muted-foreground leading-relaxed">
          If retail flow predictably goes to "Lakers YES," then a market maker can position on
          "Celtics NO" - the economically equivalent position that retail ignores.
        </p>

        <div className="from-primary/10 via-primary/5 border-primary/30 mt-6 rounded-lg border bg-gradient-to-br to-transparent p-5">
          <div className="flex items-start gap-4">
            <div className="bg-primary/20 flex h-10 w-10 flex-shrink-0 items-center justify-center rounded-full">
              <Target className="text-primary h-5 w-5" />
            </div>
            <div>
              <h3 className="text-lg font-semibold">The Core Strategy</h3>
              <p className="text-muted-foreground mt-2">
                Post limit orders on the <strong>NO side</strong> of the{' '}
                <strong>less popular team's market</strong>. When retail flow comes in as YES on the
                popular team, you become their{' '}
                <GlossaryTooltip term={terms.counterparty}>counterparty</GlossaryTooltip>.
              </p>
            </div>
          </div>
        </div>

        <ExampleBox title="Step-by-Step: Lakers (Favorite) vs Celtics (Underdog)">
          <ol className="space-y-4">
            <li className="flex gap-3">
              <span className="bg-primary/20 text-primary flex h-7 w-7 flex-shrink-0 items-center justify-center rounded-full text-sm font-semibold">
                1
              </span>
              <div>
                <strong>Identify the flow direction</strong>
                <p className="text-muted-foreground mt-1 text-sm">
                  Lakers are the popular team. Retail will bet "Lakers YES." They won't think about
                  "Celtics NO" even though it's the same position.
                </p>
              </div>
            </li>
            <li className="flex gap-3">
              <span className="bg-primary/20 text-primary flex h-7 w-7 flex-shrink-0 items-center justify-center rounded-full text-sm font-semibold">
                2
              </span>
              <div>
                <strong>Position on Celtics NO</strong>
                <p className="text-muted-foreground mt-1 text-sm">
                  Post a limit order to buy "Celtics NO" at your target price (say, 35¢). This is
                  equivalent to selling Lakers YES at 65¢.
                </p>
              </div>
            </li>
            <li className="flex gap-3">
              <span className="bg-primary/20 text-primary flex h-7 w-7 flex-shrink-0 items-center justify-center rounded-full text-sm font-semibold">
                3
              </span>
              <div>
                <strong>Wait for retail flow</strong>
                <p className="text-muted-foreground mt-1 text-sm">
                  A Lakers fan comes to the market and buys "Lakers YES" at 65¢ with a market order.
                  They need a counterparty.
                </p>
              </div>
            </li>
            <li className="flex gap-3">
              <span className="bg-primary/20 text-primary flex h-7 w-7 flex-shrink-0 items-center justify-center rounded-full text-sm font-semibold">
                4
              </span>
              <div>
                <strong>Orders match</strong>
                <p className="text-muted-foreground mt-1 text-sm">
                  The exchange matches the "Lakers YES" buyer with your "Celtics NO" order. You're
                  now the counterparty - you profit if Lakers lose, they profit if Lakers win.
                </p>
              </div>
            </li>
            <li className="flex gap-3">
              <span className="bg-primary/20 text-primary flex h-7 w-7 flex-shrink-0 items-center justify-center rounded-full text-sm font-semibold">
                5
              </span>
              <div>
                <strong>Your edge</strong>
                <p className="text-muted-foreground mt-1 text-sm">
                  Your <GlossaryTooltip term={terms.edge}>edge</GlossaryTooltip> comes from queue
                  position (getting filled before other market makers) and capturing the spread
                  between where you posted and where retail was willing to trade.
                </p>
              </div>
            </li>
          </ol>
        </ExampleBox>

        <MarketFlowVisualizer />
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 4: WHY THE NO SIDE */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="Competition Exists">
            Other market makers know this strategy. The edge comes from execution quality - being
            first in queue, having better pricing, and managing risk well.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="Why the NO Side?"
          subtitle="Three strategic advantages"
        />

        <p className="text-muted-foreground mb-6 leading-relaxed">
          Specifically targeting the NO side of the opponent's market (not just any position)
          provides three key advantages:
        </p>

        <div className="grid gap-4">
          <div className="bg-card flex items-start gap-4 rounded-lg border p-4">
            <div className="flex h-10 w-10 flex-shrink-0 items-center justify-center rounded-full bg-amber-500/20">
              <Users className="h-5 w-5 text-amber-400" />
            </div>
            <div>
              <h4 className="font-semibold">1. Less Competition</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                Retail traders crowd the YES side of popular teams. By going to the NO side of the
                opponent, you're in a less crowded order book with better{' '}
                <GlossaryTooltip term={terms.queuePosition}>queue position</GlossaryTooltip>.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-4 rounded-lg border p-4">
            <div className="flex h-10 w-10 flex-shrink-0 items-center justify-center rounded-full bg-amber-500/20">
              <Coins className="h-5 w-5 text-amber-400" />
            </div>
            <div>
              <h4 className="font-semibold">2. Capture Predictable Flow</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                <GlossaryTooltip term={terms.retailFlow}>Retail flow</GlossaryTooltip> is
                predictable. You know where it's going (YES on popular teams), so you can position
                to be the counterparty before it arrives.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-4 rounded-lg border p-4">
            <div className="flex h-10 w-10 flex-shrink-0 items-center justify-center rounded-full bg-amber-500/20">
              <Zap className="h-5 w-5 text-amber-400" />
            </div>
            <div>
              <h4 className="font-semibold">3. Faster Fills</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                Better queue position means you get matched first when flow comes in. Time-to-fill
                matters because the market can move, and sitting in queue ties up capital.
              </p>
            </div>
          </div>
        </div>

        <div className="bg-primary/5 border-primary/20 mt-6 rounded-lg border p-4">
          <h4 className="mb-2 flex items-center gap-2 font-semibold">
            <Lightbulb className="text-primary h-4 w-4" />
            Queue Position vs Price
          </h4>
          <p className="text-muted-foreground text-sm">
            Sometimes getting a <em>slightly worse price</em> with{' '}
            <em>much better queue position</em>
            is the right trade. If you're 10th in queue at 35¢ but could be 1st in queue at 34¢, the
            1¢ worse price might be worth it to actually get filled.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 5: WHEN THIS APPLIES */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="Structure Matters">
            The dual-market structure is unique to moneylines. Spreads and totals work differently.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="When This Strategy Applies"
          subtitle="Moneylines vs spreads and totals"
        />

        <p className="text-muted-foreground mb-6 leading-relaxed">
          This NO-side strategy specifically applies to <strong>moneyline markets</strong> because
          only moneylines have Kalshi's dual-market structure. Point spreads and totals work
          differently.
        </p>

        <div className="overflow-x-auto">
          <table className="w-full text-sm">
            <thead>
              <tr className="border-b">
                <th className="py-3 text-left font-medium">Bet Type</th>
                <th className="py-3 text-left font-medium">Market Structure</th>
                <th className="py-3 text-left font-medium">Strategy Applicability</th>
              </tr>
            </thead>
            <tbody className="divide-y">
              <tr>
                <td className="py-3 font-medium">Moneyline</td>
                <td className="text-muted-foreground py-3">
                  <span className="font-medium text-emerald-400">TWO markets</span> per game
                  <div className="mt-1 text-xs">"Will Lakers win?" + "Will Celtics win?"</div>
                </td>
                <td className="py-3">
                  <span className="font-medium text-emerald-400">✓ Full strategy applies</span>
                  <div className="text-muted-foreground mt-1 text-xs">
                    Focus on NO side of less popular team
                  </div>
                </td>
              </tr>
              <tr>
                <td className="py-3 font-medium">Point Spread</td>
                <td className="text-muted-foreground py-3">
                  <span className="font-medium text-amber-400">ONE market</span>
                  <div className="mt-1 text-xs">"Will Lakers win by 5.5+ points?"</div>
                </td>
                <td className="py-3">
                  <span className="text-amber-400">Different dynamics</span>
                  <div className="text-muted-foreground mt-1 text-xs">
                    YES = favorite covers, NO = underdog covers
                  </div>
                </td>
              </tr>
              <tr>
                <td className="py-3 font-medium">Total (O/U)</td>
                <td className="text-muted-foreground py-3">
                  <span className="font-medium text-amber-400">ONE market</span>
                  <div className="mt-1 text-xs">"Will total be over 227.5?"</div>
                </td>
                <td className="py-3">
                  <span className="text-amber-400">Different dynamics</span>
                  <div className="text-muted-foreground mt-1 text-xs">
                    No team-based flow asymmetry
                  </div>
                </td>
              </tr>
            </tbody>
          </table>
        </div>

        <div className="bg-muted/50 mt-6 rounded-lg border p-4">
          <h4 className="mb-2 font-semibold">Why Spreads and Totals Are Different</h4>
          <p className="text-muted-foreground text-sm">
            For spreads and totals, there's only one market with YES/NO representing opposite views
            on the same question. There's no "equivalent position in another order book" to exploit.
            The flow dynamics and market making strategies are different.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 6: EXPECTED VALUE */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <ConceptBlock title="Expected Value">
            <GlossaryTooltip term={terms.ev}>EV</GlossaryTooltip> is the average outcome if you
            repeated the trade infinitely. Positive EV means profitable long-term, even with
            individual losses.
          </ConceptBlock>
        }
      >
        <SectionHeader
          title="Understanding Expected Value"
          subtitle="Why edge matters more than individual wins"
        />

        <p className="text-muted-foreground leading-relaxed">
          Market makers don't need to "win" every trade. They need positive{' '}
          <GlossaryTooltip term={terms.ev}>expected value (EV)</GlossaryTooltip> over many trades.
          Small edges compound over hundreds of positions.
        </p>

        <div className="mt-6 space-y-4">
          <Term term="Edge">
            Your <GlossaryTooltip term={terms.edge}>edge</GlossaryTooltip> is the difference between
            the price you trade at and fair value. If you buy at 35¢ and fair value is 36¢, you have
            a 1¢ edge per contract.
          </Term>

          <Term term="Expected Value">
            <Formula>
              EV = (Win probability × Win amount) - (Loss probability × Loss amount)
            </Formula>
            . Positive EV means profitable long-term.
          </Term>
        </div>

        <ExampleBox title="EV Calculation Example">
          <div className="space-y-4">
            <p className="text-sm">
              <strong>Scenario:</strong> You buy Celtics NO at 35¢ (equivalent to selling Lakers YES
              at 65¢). You believe the true probability of Celtics losing is 65% (Lakers winning is
              65%).
            </p>

            <div className="bg-muted/50 rounded-lg p-4 text-sm">
              <div className="mb-2">If Lakers win (65% chance): You lose 35¢</div>
              <div className="mb-3">If Celtics win (35% chance): You win 65¢</div>
              <div className="space-y-2 border-t pt-3">
                <Formula block>EV = (35\% \times 65¢) - (65\% \times 35¢)</Formula>
                <Formula block>EV = 22.75¢ - 22.75¢ = 0¢</Formula>
              </div>
            </div>

            <p className="text-muted-foreground text-xs">
              At fair value, EV is zero. Market makers look for situations where they can trade at
              better-than-fair prices, creating positive EV.
            </p>

            <div className="rounded-lg border border-emerald-500/30 bg-emerald-500/10 p-4">
              <p className="text-sm">
                <strong className="text-emerald-400">Where does edge come from?</strong>
              </p>
              <ul className="text-muted-foreground mt-2 space-y-1 text-sm">
                <li>• Spread capture: buying at bid, selling at ask</li>
                <li>• Queue position: getting filled before price moves</li>
                <li>• Information advantage: knowing where flow will go</li>
              </ul>
            </div>
          </div>
        </ExampleBox>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 7: PRACTICAL CONSIDERATIONS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="Risk Exists">
            Market making isn't risk-free. You're taking real positions that can lose money. The
            edge comes from consistent execution, not guaranteed profits.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="Practical Considerations"
          subtitle="What you need to know before trading"
        />

        <p className="text-muted-foreground mb-6 leading-relaxed">
          Understanding the strategy is one thing. Executing it profitably requires attention to
          several practical factors.
        </p>

        <div className="grid gap-4 sm:grid-cols-2">
          <ConceptCard
            title="Fees"
            description="Kalshi charges fees per trade. Your edge must exceed fees to be profitable."
            icon={<Coins className="h-4 w-4" />}
          />
          <ConceptCard
            title="Capital Efficiency"
            description="Positions tie up capital. Consider how long you'll wait for fills."
            icon={<TrendingUp className="h-4 w-4" />}
          />
          <ConceptCard
            title="Risk Management"
            description="Don't over-concentrate. Spread positions across multiple games."
            icon={<AlertTriangle className="h-4 w-4" />}
          />
          <ConceptCard
            title="Competition"
            description="Other market makers exist. Your edge comes from better execution."
            icon={<Users className="h-4 w-4" />}
          />
        </div>

        <div className="bg-card mt-6 rounded-lg border p-4">
          <h4 className="mb-3 font-semibold">Key Questions to Ask</h4>
          <ul className="text-muted-foreground space-y-2 text-sm">
            <li className="flex items-start gap-2">
              <span className="text-primary">→</span>
              <span>Where is the retail flow likely to go for this game?</span>
            </li>
            <li className="flex items-start gap-2">
              <span className="text-primary">→</span>
              <span>What's my queue position in the target order book?</span>
            </li>
            <li className="flex items-start gap-2">
              <span className="text-primary">→</span>
              <span>Is the spread wide enough to cover fees and provide edge?</span>
            </li>
            <li className="flex items-start gap-2">
              <span className="text-primary">→</span>
              <span>How much capital am I tying up and for how long?</span>
            </li>
          </ul>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 8: THE DASHBOARD VIEW */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="Reading the Dashboard"
          subtitle="What market makers monitor"
        />

        <p className="text-muted-foreground mb-4">
          Professional market makers compare prices and liquidity across multiple venues and
          markets. Here's what a typical monitoring view looks like:
        </p>

        <MarketMakerDashboardPreview />
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 9: KEY TAKEAWAYS */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="Key Takeaways"
          subtitle="What to remember from this lesson"
        />

        <div className="space-y-4">
          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">1</span>
            </div>
            <div>
              <h4 className="font-medium">Market makers profit from flow, not predictions</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                The goal isn't to predict who wins. It's to position where you know flow will go and
                be the counterparty at favorable prices.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">2</span>
            </div>
            <div>
              <h4 className="font-medium">Retail flow is predictable</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                Recreational bettors almost always bet YES on their preferred team. This creates
                asymmetric flow that market makers can exploit.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">3</span>
            </div>
            <div>
              <h4 className="font-medium">Focus on the NO side of the opponent's market</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                For moneylines, position on the NO side of the less popular team. You'll face less
                competition and better queue positions.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">4</span>
            </div>
            <div>
              <h4 className="font-medium">Queue position can matter more than price</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                A slightly worse price with much better queue position often wins. Getting filled
                matters more than optimizing the last cent.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">5</span>
            </div>
            <div>
              <h4 className="font-medium">This only applies to moneylines</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                Kalshi's dual-market structure only exists for moneylines. Spreads and totals have
                single markets with different dynamics.
              </p>
            </div>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 10: NEXT STEPS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="Check the Glossary">
            For detailed definitions of any terms covered, visit the{' '}
            <a
              href="/glossary"
              className="text-primary hover:underline"
            >
              Glossary
            </a>
            .
          </TipBlock>
        }
      >
        <SectionHeader
          title="What's Next?"
          subtitle="Applying what you've learned"
        />

        <p className="text-muted-foreground mb-6 leading-relaxed">
          You've completed the core curriculum! You now understand:
        </p>

        <ul className="text-muted-foreground mb-6 space-y-2 text-sm">
          <li className="flex items-center gap-2">
            <span className="text-emerald-400">✓</span>
            <span>What prediction markets and Kalshi are (Lesson 1)</span>
          </li>
          <li className="flex items-center gap-2">
            <span className="text-emerald-400">✓</span>
            <span>The three main bet types: moneylines, spreads, totals (Lesson 2)</span>
          </li>
          <li className="flex items-center gap-2">
            <span className="text-emerald-400">✓</span>
            <span>Kalshi's dual market structure and order books (Lesson 3)</span>
          </li>
          <li className="flex items-center gap-2">
            <span className="text-emerald-400">✓</span>
            <span>Market making strategy: NO side positioning (This lesson)</span>
          </li>
        </ul>

        <NextStepBlock
          href="/app"
          title="Ready to Trade?"
          description="Apply what you've learned in the trading dashboard. Start by observing order flow patterns before placing your own orders."
          buttonText="Go to Dashboard"
        />
      </LessonRow>
    </LessonLayout>
  );
}
