/**
 * Lesson: Understanding Moneylines
 *
 * A deep dive into what "moneyline" actually means, where the term comes from,
 * how to read it, common confusions, and when people use it incorrectly.
 */

import { useState } from 'react';
import {
  Trophy,
  History,
  Calculator,
  AlertTriangle,
  Scale,
  DollarSign,
  ArrowLeftRight,
  Globe,
  Users,
  Lightbulb,
  TrendingUp,
} from 'lucide-react';
import { getGuide, getNextGuide } from '@/data/guides';
import { LessonLayout, LessonRow, LessonHeader, SectionHeader } from '../LessonLayout';
import {
  ObjectiveBlock,
  ConstraintBlock,
  TipBlock,
  InsightBlock,
  WarningBlock,
  ConceptBlock,
  ConceptCard,
  NextStepBlock,
  ExampleBox,
  Term,
} from '../blocks';
import { GlossaryTooltip } from '@/components/atoms/GlossaryTooltip';

// Custom glossary terms for this lesson
const terms = {
  moneyline: {
    term: 'Moneyline',
    definition:
      'A bet on which team will win, regardless of score. The simplest bet type - just pick the winner.',
    category: 'Bet Types',
  },
  favorite: {
    term: 'Favorite',
    definition:
      'The team expected to win. They give points on a spread or cost more on a moneyline.',
    category: 'Bet Types',
  },
  underdog: {
    term: 'Underdog',
    definition:
      'The team expected to lose. They receive points on a spread and pay more on a moneyline.',
    category: 'Bet Types',
  },
  juice: {
    term: 'Juice (Vig)',
    definition:
      'The fee built into odds that gives the sportsbook its edge. Standard is -110 (bet $110 to win $100).',
    category: 'Fees',
  },
  impliedProbability: {
    term: 'Implied Probability',
    definition:
      "The probability suggested by the odds. Calculate from price: 65¢ = 65% chance. The market's estimate of likelihood.",
    category: 'Probability',
  },
  americanOdds: {
    term: 'American Odds',
    definition:
      'Odds format using + and - numbers. Negative shows how much to bet to win $100. Positive shows how much you win on a $100 bet.',
    category: 'Odds Formats',
  },
  decimalOdds: {
    term: 'Decimal Odds',
    definition:
      'European odds format. Multiply your stake by the decimal to get your total return (including stake).',
    category: 'Odds Formats',
  },
  fractionalOdds: {
    term: 'Fractional Odds',
    definition:
      'British odds format showing profit/stake ratio. 5/1 means £5 profit for every £1 staked.',
    category: 'Odds Formats',
  },
  pointSpread: {
    term: 'Point Spread',
    definition:
      'A handicap given to the underdog. Different from moneyline - spread bets on margin, not just winner.',
    category: 'Bet Types',
  },
};

// Interactive widget for converting between formats
function OddsConverterWidget() {
  const [kalshiPrice, setKalshiPrice] = useState(65);

  // Convert Kalshi price (cents) to other formats
  const impliedProb = kalshiPrice;
  const americanOdds =
    kalshiPrice >= 50
      ? Math.round((-100 * kalshiPrice) / (100 - kalshiPrice))
      : Math.round((100 * (100 - kalshiPrice)) / kalshiPrice);
  const decimalOdds = 100 / kalshiPrice;
  const profit = 100 - kalshiPrice;

  return (
    <div className="bg-card mt-6 rounded-lg border p-5">
      <div className="text-muted-foreground mb-4 text-xs font-semibold uppercase">
        Interactive: Odds Format Converter
      </div>

      <div className="space-y-6">
        {/* Kalshi price slider */}
        <div>
          <div className="mb-2 flex items-center justify-between">
            <label className="text-sm font-medium">Kalshi Price</label>
            <span className="text-primary font-mono text-lg font-bold">{kalshiPrice}¢</span>
          </div>
          <input
            type="range"
            min={5}
            max={95}
            value={kalshiPrice}
            onChange={(e) => setKalshiPrice(Number(e.target.value))}
            className="accent-primary w-full"
          />
          <div className="text-muted-foreground mt-1 flex justify-between text-xs">
            <span>Heavy underdog</span>
            <span>Heavy favorite</span>
          </div>
        </div>

        {/* Conversion results */}
        <div className="grid gap-3 sm:grid-cols-2">
          <div className="bg-muted/50 rounded-lg p-3">
            <div className="text-muted-foreground mb-1 text-xs">Implied Probability</div>
            <div className="font-mono text-lg font-semibold">{impliedProb}%</div>
          </div>
          <div className="bg-muted/50 rounded-lg p-3">
            <div className="text-muted-foreground mb-1 text-xs">American Odds</div>
            <div className="font-mono text-lg font-semibold">
              {americanOdds > 0 ? '+' : ''}
              {americanOdds}
            </div>
          </div>
          <div className="bg-muted/50 rounded-lg p-3">
            <div className="text-muted-foreground mb-1 text-xs">Decimal Odds</div>
            <div className="font-mono text-lg font-semibold">{decimalOdds.toFixed(2)}</div>
          </div>
          <div className="bg-muted/50 rounded-lg p-3">
            <div className="text-muted-foreground mb-1 text-xs">Profit per $1</div>
            <div className="font-mono text-lg font-semibold text-emerald-400">{profit}¢</div>
          </div>
        </div>

        {/* Plain English explanation */}
        <div className="text-muted-foreground border-t pt-4 text-sm">
          <strong>In plain English:</strong> At {kalshiPrice}¢, you risk {kalshiPrice}¢ to profit{' '}
          {profit}¢. The market thinks there's about a {impliedProb}% chance this happens.
          {kalshiPrice >= 50 ? (
            <span className="text-amber-400"> This is a favorite.</span>
          ) : (
            <span className="text-violet-400"> This is an underdog.</span>
          )}
        </div>
      </div>
    </div>
  );
}

// Interactive widget showing moneyline payouts
function MoneylinePayoutWidget() {
  const [stake, setStake] = useState(100);
  const [favoritePrice, setFavoritePrice] = useState(65);

  const underdogPrice = 100 - favoritePrice;
  const favoriteProfit = (stake * (100 - favoritePrice)) / favoritePrice;
  const underdogProfit = (stake * favoritePrice) / (100 - favoritePrice);

  return (
    <div className="bg-card mt-6 rounded-lg border p-5">
      <div className="text-muted-foreground mb-4 text-xs font-semibold uppercase">
        Interactive: Moneyline Payout Calculator
      </div>

      <div className="space-y-6">
        {/* Controls */}
        <div className="grid gap-4 sm:grid-cols-2">
          <div>
            <div className="mb-2 flex items-center justify-between">
              <label className="text-sm font-medium">Your Stake</label>
              <span className="text-primary font-mono">${stake}</span>
            </div>
            <input
              type="range"
              min={10}
              max={500}
              step={10}
              value={stake}
              onChange={(e) => setStake(Number(e.target.value))}
              className="accent-primary w-full"
            />
          </div>
          <div>
            <div className="mb-2 flex items-center justify-between">
              <label className="text-sm font-medium">Favorite Price</label>
              <span className="text-primary font-mono">{favoritePrice}¢</span>
            </div>
            <input
              type="range"
              min={51}
              max={95}
              value={favoritePrice}
              onChange={(e) => setFavoritePrice(Number(e.target.value))}
              className="accent-primary w-full"
            />
          </div>
        </div>

        {/* Payout comparison */}
        <div className="grid gap-4 sm:grid-cols-2">
          <div className="rounded-lg border border-amber-500/30 bg-amber-500/5 p-4">
            <div className="mb-3 flex items-center gap-2">
              <Trophy className="h-4 w-4 text-amber-400" />
              <span className="font-medium text-amber-400">Betting on Favorite</span>
            </div>
            <div className="space-y-2 text-sm">
              <div className="flex justify-between">
                <span className="text-muted-foreground">Price:</span>
                <span className="font-mono">{favoritePrice}¢</span>
              </div>
              <div className="flex justify-between">
                <span className="text-muted-foreground">You risk:</span>
                <span className="font-mono">${stake.toFixed(2)}</span>
              </div>
              <div className="flex justify-between border-t pt-2">
                <span className="text-muted-foreground">If they win:</span>
                <span className="font-mono text-emerald-400">+${favoriteProfit.toFixed(2)}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-muted-foreground">If they lose:</span>
                <span className="font-mono text-rose-400">-${stake.toFixed(2)}</span>
              </div>
            </div>
          </div>

          <div className="rounded-lg border border-violet-500/30 bg-violet-500/5 p-4">
            <div className="mb-3 flex items-center gap-2">
              <TrendingUp className="h-4 w-4 text-violet-400" />
              <span className="font-medium text-violet-400">Betting on Underdog</span>
            </div>
            <div className="space-y-2 text-sm">
              <div className="flex justify-between">
                <span className="text-muted-foreground">Price:</span>
                <span className="font-mono">{underdogPrice}¢</span>
              </div>
              <div className="flex justify-between">
                <span className="text-muted-foreground">You risk:</span>
                <span className="font-mono">${stake.toFixed(2)}</span>
              </div>
              <div className="flex justify-between border-t pt-2">
                <span className="text-muted-foreground">If they win:</span>
                <span className="font-mono text-emerald-400">+${underdogProfit.toFixed(2)}</span>
              </div>
              <div className="flex justify-between">
                <span className="text-muted-foreground">If they lose:</span>
                <span className="font-mono text-rose-400">-${stake.toFixed(2)}</span>
              </div>
            </div>
          </div>
        </div>

        <div className="text-muted-foreground border-t pt-4 text-sm">
          <strong>Key insight:</strong> With the same ${stake} stake, you win{' '}
          <span className="text-amber-400">${favoriteProfit.toFixed(2)}</span> on the favorite but{' '}
          <span className="text-violet-400">${underdogProfit.toFixed(2)}</span> on the underdog.
          Higher risk = lower reward. Lower risk = higher reward.
        </div>
      </div>
    </div>
  );
}

export function MoneylineLesson() {
  const guide = getGuide('moneyline')!;
  const nextGuide = getNextGuide('moneyline');

  return (
    <LessonLayout>
      {/* Header */}
      <LessonRow>
        <LessonHeader
          title={guide.title}
          description={guide.description}
          duration="20 min"
          category="Fundamentals"
        />
      </LessonRow>

      {/* Objective */}
      <LessonRow
        aside={
          <ConstraintBlock
            title="What We'll Cover"
            items={[
              'What "moneyline" actually means and why it\'s called that',
              'How to read moneyline odds in different formats',
              'The relationship between price and probability',
              'Common confusions and when people use the term wrong',
              'How moneyline differs from spreads and other bets',
            ]}
          />
        }
      >
        <ObjectiveBlock>
          By the end of this lesson, you'll understand exactly what a moneyline is, where the
          strange name comes from, how to interpret odds in any format, and - critically - when
          people are using the term incorrectly. You'll never be confused by "moneyline" again.
        </ObjectiveBlock>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 1: THE TERM ITSELF */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="The Physical Line">
            Before computers, bookmakers literally wrote odds on chalkboards. The "line" was the row
            of numbers. Different lines for different bet types: the money line, the point spread
            line, the total line.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="Why Is It Called 'Moneyline'?"
          subtitle="Etymology first - understanding the word helps everything else make sense"
        />

        <p className="text-muted-foreground leading-relaxed">
          Let's start with the name itself, because it's genuinely strange. Why "moneyline"? What
          money? What line? The term makes no intuitive sense to newcomers, which is why it trips
          people up.
        </p>

        <div className="bg-muted/50 my-6 flex items-start gap-3 rounded-lg border p-4">
          <div className="bg-primary/10 flex h-10 w-10 flex-shrink-0 items-center justify-center rounded-full">
            <History className="text-primary h-5 w-5" />
          </div>
          <div>
            <h4 className="font-semibold">The Origin Story</h4>
            <p className="text-muted-foreground mt-1 text-sm">
              The term comes from American bookmaking in the early 20th century. Bookmakers
              displayed odds on large chalkboards. Each type of bet had its own horizontal "line" of
              information written across the board. The <em>moneyline</em> was the line that showed
              how much <em>money</em> you needed to bet to win a fixed amount - or how much you'd
              win from a fixed bet. The "line" was literally a line on a board.
            </p>
          </div>
        </div>

        <p className="text-muted-foreground leading-relaxed">
          In Britain, they simply call this a "win bet" or "to win" market. In continental Europe,
          it's often called "1X2" (home/draw/away). The American term stuck in betting culture
          because American odds notation became standardized, and "moneyline" was baked into the
          terminology.
        </p>

        <div className="mt-6 space-y-4">
          <Term term="Line">
            In betting, a "line" means the odds or terms set by a bookmaker. The spread is "the
            line." The total is "the line." The moneyline is... the money line. You might hear "the
            line moved" - meaning the odds changed.
          </Term>

          <Term term="Money">
            Here, "money" refers to the stake/payout relationship. Unlike point spreads where both
            sides usually pay the same (bet $110 to win $100), moneyline odds vary based on who's
            favored. The "money" differs between the two teams.
          </Term>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 2: THE CORE CONCEPT */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="The Simplest Bet">
            If someone asks "what's the simplest sports bet?" - the answer is moneyline. No math
            about margins, no thinking about totals. Just: who wins?
          </TipBlock>
        }
      >
        <SectionHeader
          title="What Is a Moneyline Bet?"
          subtitle="Stripping away the jargon"
        />

        <p className="text-muted-foreground leading-relaxed">
          A <GlossaryTooltip term={terms.moneyline}>moneyline bet</GlossaryTooltip> answers the
          simplest question in sports: <strong>who wins the game?</strong>
        </p>

        <p className="text-muted-foreground mt-4 leading-relaxed">
          That's it. Nothing else matters:
        </p>

        <ul className="text-muted-foreground mt-4 space-y-2">
          <li className="flex items-start gap-2">
            <span className="mt-1 text-emerald-400">✓</span>
            <span>
              <strong>Not the score.</strong> Win by 1 or win by 50 - you get paid the same.
            </span>
          </li>
          <li className="flex items-start gap-2">
            <span className="mt-1 text-emerald-400">✓</span>
            <span>
              <strong>Not how they win.</strong> Blowout, buzzer-beater, overtime - doesn't matter.
            </span>
          </li>
          <li className="flex items-start gap-2">
            <span className="mt-1 text-emerald-400">✓</span>
            <span>
              <strong>Not when they take the lead.</strong> They can trail the whole game and win in
              the final second.
            </span>
          </li>
        </ul>

        <ExampleBox title="The Everyday Analogy">
          <p className="text-muted-foreground">
            Imagine you and a friend are watching a game. Your friend says: "I bet you $20 the
            Lakers win." You shake hands. That's a moneyline bet. The only question that matters at
            the end is: did the Lakers win?
          </p>
          <p className="text-muted-foreground mt-3">
            The complexity comes when you ask: what if one team is heavily favored? Should you both
            risk $20 when one side has a 90% chance of winning? That's where the "odds" part of
            moneyline betting comes in.
          </p>
        </ExampleBox>

        <div className="from-primary/10 via-primary/5 border-primary/30 mt-6 rounded-lg border bg-gradient-to-br to-transparent p-4">
          <div className="flex items-start gap-3">
            <Lightbulb className="text-primary mt-0.5 h-5 w-5 flex-shrink-0" />
            <div>
              <p className="font-medium">The Core Insight</p>
              <p className="text-muted-foreground mt-1 text-sm">
                A moneyline bet is really a <em>probability bet</em> expressed as a price. You're
                betting on whether an outcome will happen, and the price reflects how likely it is.
                On Kalshi, this is made explicit: a 65¢ price means roughly 65% probability.
              </p>
            </div>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 3: FAVORITES VS UNDERDOGS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <ConceptBlock title="Favorites & Underdogs">
            The <GlossaryTooltip term={terms.favorite}>favorite</GlossaryTooltip> is the team
            expected to win. They cost more to bet on. The{' '}
            <GlossaryTooltip term={terms.underdog}>underdog</GlossaryTooltip> is expected to lose -
            they cost less but pay more if they win.
          </ConceptBlock>
        }
      >
        <SectionHeader
          title="Favorites and Underdogs"
          subtitle="Why moneyline odds aren't always 50/50"
        />

        <p className="text-muted-foreground leading-relaxed">
          If two teams were evenly matched, betting on either would have the same risk and reward.
          But sports rarely work that way. One team is usually more likely to win. Moneyline odds
          reflect this imbalance.
        </p>

        <div className="mt-6 grid gap-4 sm:grid-cols-2">
          <div className="rounded-lg border border-amber-500/30 bg-amber-500/5 p-4">
            <div className="mb-3 flex items-center gap-2">
              <Trophy className="h-5 w-5 text-amber-400" />
              <h4 className="font-semibold text-amber-400">The Favorite</h4>
            </div>
            <ul className="text-muted-foreground space-y-2 text-sm">
              <li>• Expected to win</li>
              <li>• Costs more to bet on (higher price)</li>
              <li>• Smaller profit if they win</li>
              <li>• You risk a lot to win a little</li>
            </ul>
            <div className="mt-4 border-t border-amber-500/20 pt-3">
              <p className="text-muted-foreground text-xs">
                <strong>On Kalshi:</strong> 70¢ price means 70% chance. Risk 70¢ to profit 30¢.
              </p>
            </div>
          </div>

          <div className="rounded-lg border border-violet-500/30 bg-violet-500/5 p-4">
            <div className="mb-3 flex items-center gap-2">
              <TrendingUp className="h-5 w-5 text-violet-400" />
              <h4 className="font-semibold text-violet-400">The Underdog</h4>
            </div>
            <ul className="text-muted-foreground space-y-2 text-sm">
              <li>• Expected to lose</li>
              <li>• Costs less to bet on (lower price)</li>
              <li>• Larger profit if they win</li>
              <li>• You risk a little to win a lot</li>
            </ul>
            <div className="mt-4 border-t border-violet-500/20 pt-3">
              <p className="text-muted-foreground text-xs">
                <strong>On Kalshi:</strong> 30¢ price means 30% chance. Risk 30¢ to profit 70¢.
              </p>
            </div>
          </div>
        </div>

        <ExampleBox title="Lakers 70¢ vs Celtics 30¢">
          <div className="space-y-4">
            <p className="text-muted-foreground text-sm">
              The market thinks the Lakers have about a 70% chance of winning. Here's what each bet
              looks like for a $100 stake:
            </p>
            <div className="grid gap-3 text-sm sm:grid-cols-2">
              <div className="bg-muted/50 rounded p-3">
                <div className="mb-2 font-medium text-amber-400">Bet $100 on Lakers (favorite)</div>
                <ul className="text-muted-foreground space-y-1">
                  <li>• Buy 143 contracts at 70¢ each</li>
                  <li>
                    • Lakers win → <span className="text-emerald-400">+$43 profit</span>
                  </li>
                  <li>
                    • Lakers lose → <span className="text-rose-400">-$100 loss</span>
                  </li>
                </ul>
              </div>
              <div className="bg-muted/50 rounded p-3">
                <div className="mb-2 font-medium text-violet-400">
                  Bet $100 on Celtics (underdog)
                </div>
                <ul className="text-muted-foreground space-y-1">
                  <li>• Buy 333 contracts at 30¢ each</li>
                  <li>
                    • Celtics win → <span className="text-emerald-400">+$233 profit</span>
                  </li>
                  <li>
                    • Celtics lose → <span className="text-rose-400">-$100 loss</span>
                  </li>
                </ul>
              </div>
            </div>
            <p className="text-muted-foreground border-t pt-3 text-xs">
              Same $100 at risk, but very different potential profits. The favorite wins more often
              but pays less. The underdog wins less often but pays more. Over the long run, if the
              odds are accurate, both bets have the same expected value.
            </p>
          </div>
        </ExampleBox>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 4: ODDS FORMATS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="Don't Memorize Formulas">
            You don't need to memorize conversion formulas. Just understand that all formats express
            the same thing differently. Use the interactive tool below.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="Understanding Odds Formats"
          subtitle="American, decimal, fractional - they all mean the same thing"
        />

        <p className="text-muted-foreground leading-relaxed">
          When you hear people talk about moneylines, they might express the odds in different
          formats. This causes massive confusion, because "-150" and "1.67" and "2/3" all mean the
          same thing. Let's demystify this.
        </p>

        <div className="mt-6 grid gap-4 sm:grid-cols-3">
          <ConceptCard
            title="American Odds"
            description='Uses + and - signs. Standard in US sports betting. "-150" or "+200"'
            icon={<DollarSign className="h-4 w-4" />}
          />
          <ConceptCard
            title="Decimal Odds"
            description="Standard in Europe, Australia, Canada. Shows total return: 2.50"
            icon={<Calculator className="h-4 w-4" />}
          />
          <ConceptCard
            title="Fractional Odds"
            description='Traditional British format. Shows profit to stake ratio: "5/2"'
            icon={<Scale className="h-4 w-4" />}
          />
        </div>

        <div className="mt-6 space-y-4">
          <Term term="American Odds (-150, +200)">
            <p>
              <GlossaryTooltip term={terms.americanOdds}>American odds</GlossaryTooltip> are
              confusing because they work differently for favorites and underdogs:
            </p>
            <ul className="mt-2 space-y-1 text-sm">
              <li>
                <strong>Negative (-150):</strong> How much you bet to win $100. Bet $150 to win
                $100.
              </li>
              <li>
                <strong>Positive (+200):</strong> How much you win on a $100 bet. Bet $100 to win
                $200.
              </li>
            </ul>
          </Term>

          <Term term="Decimal Odds (1.67, 3.00)">
            <GlossaryTooltip term={terms.decimalOdds}>Decimal odds</GlossaryTooltip> show your total
            return per $1 staked. At 1.67, a $100 bet returns $167 total ($67 profit + your $100
            stake). At 3.00, a $100 bet returns $300 ($200 profit + $100 stake).
          </Term>

          <Term term="Fractional Odds (2/3, 2/1)">
            <GlossaryTooltip term={terms.fractionalOdds}>Fractional odds</GlossaryTooltip> show
            profit relative to stake. At 2/3, you profit $2 for every $3 staked. At 2/1, you profit
            $2 for every $1 staked.
          </Term>
        </div>

        <OddsConverterWidget />

        <div className="bg-primary/5 border-primary/20 mt-6 rounded-lg border p-4">
          <h4 className="mb-2 flex items-center gap-2 font-semibold">
            <Globe className="text-primary h-4 w-4" />
            Kalshi's Approach: The Clearest Format
          </h4>
          <p className="text-muted-foreground text-sm">
            Kalshi uses a price in cents (0-100¢) which directly equals probability. 65¢ = 65%
            chance. You don't need to convert anything. The price IS the implied probability. This
            is why Kalshi's format is by far the most intuitive for beginners.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 5: PAYOUTS DEEP DIVE */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="The Math Checks Out">
            If markets are efficient, the expected value of betting favorites equals the expected
            value of betting underdogs. The different payouts balance the different probabilities.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="How Payouts Work"
          subtitle="Understanding risk vs reward"
        />

        <p className="text-muted-foreground leading-relaxed">
          The core principle: <strong>higher probability = lower payout</strong>. This is true
          across all betting, investing, and probability. Safe bets pay less. Risky bets pay more.
        </p>

        <MoneylinePayoutWidget />

        <div className="bg-card mt-6 rounded-lg border p-4">
          <h4 className="mb-3 font-semibold">The Invariant Rule</h4>
          <p className="text-muted-foreground mb-4 text-sm">
            On Kalshi, every contract pays $1 if you're right, $0 if you're wrong. So:
          </p>
          <div className="bg-muted/30 rounded-lg p-4 text-center font-mono text-lg">
            <span className="text-primary">Cost</span>
            <span className="mx-2">+</span>
            <span className="text-emerald-400">Profit</span>
            <span className="mx-2">=</span>
            <span className="text-amber-400">$1.00</span>
          </div>
          <p className="text-muted-foreground mt-4 text-sm">
            If you pay 70¢, you profit 30¢. If you pay 30¢, you profit 70¢. Always adds to $1.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 6: COMMON CONFUSIONS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="Listen Carefully">
            When someone says "the moneyline moved" or "I bet the moneyline" - make sure you
            understand what they're actually saying. Context matters.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="Common Confusions"
          subtitle="When 'moneyline' gets used wrong or misunderstood"
        />

        <p className="text-muted-foreground leading-relaxed">
          Now let's address the confusions you'll encounter in the wild. These are mistakes even
          experienced bettors make.
        </p>

        <div className="mt-6 space-y-6">
          {/* Confusion 1 */}
          <div className="bg-card rounded-lg border p-5">
            <div className="flex items-start gap-3">
              <div className="flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full bg-rose-500/20">
                <AlertTriangle className="h-4 w-4 text-rose-400" />
              </div>
              <div>
                <h4 className="font-semibold text-rose-400">
                  Confusion #1: "Moneyline" vs "The Line"
                </h4>
                <p className="text-muted-foreground mt-2 text-sm">
                  People often say "the line" when they mean "the spread." When someone says "What's
                  the line on the Lakers game?" they might mean the point spread, not the moneyline.
                </p>
                <div className="bg-muted/50 mt-3 rounded p-3">
                  <p className="mb-1 text-xs font-medium">Context clue:</p>
                  <p className="text-muted-foreground text-xs">
                    If they mention a number like "-5.5" or "+3" they mean the spread. If they
                    mention odds like "-150" or "65¢" they mean the moneyline.
                  </p>
                </div>
              </div>
            </div>
          </div>

          {/* Confusion 2 */}
          <div className="bg-card rounded-lg border p-5">
            <div className="flex items-start gap-3">
              <div className="flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full bg-rose-500/20">
                <AlertTriangle className="h-4 w-4 text-rose-400" />
              </div>
              <div>
                <h4 className="font-semibold text-rose-400">
                  Confusion #2: Thinking Higher Number = Better Bet
                </h4>
                <p className="text-muted-foreground mt-2 text-sm">
                  In American odds, "-400" looks bigger than "-150" but it's actually a worse
                  payout. At -400, you bet $400 to win $100. At -150, you bet $150 to win $100.
                </p>
                <div className="bg-muted/50 mt-3 rounded p-3">
                  <p className="mb-1 text-xs font-medium">Rule of thumb:</p>
                  <p className="text-muted-foreground text-xs">
                    For favorites (negative odds): smaller absolute number = better payout. For
                    underdogs (positive odds): larger number = better payout.
                  </p>
                </div>
              </div>
            </div>
          </div>

          {/* Confusion 3 */}
          <div className="bg-card rounded-lg border p-5">
            <div className="flex items-start gap-3">
              <div className="flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full bg-rose-500/20">
                <AlertTriangle className="h-4 w-4 text-rose-400" />
              </div>
              <div>
                <h4 className="font-semibold text-rose-400">
                  Confusion #3: Moneyline for Draw Sports
                </h4>
                <p className="text-muted-foreground mt-2 text-sm">
                  In soccer and some other sports, games can end in draws. A "moneyline" in soccer
                  typically includes three options: Team A, Draw, or Team B. But some platforms
                  offer "2-way moneyline" where your bet pushes (returns) on a draw. Know what
                  you're betting.
                </p>
                <div className="bg-muted/50 mt-3 rounded p-3">
                  <p className="mb-1 text-xs font-medium">Ask yourself:</p>
                  <p className="text-muted-foreground text-xs">
                    "What happens if the game is tied after regulation?" Different platforms handle
                    this differently.
                  </p>
                </div>
              </div>
            </div>
          </div>

          {/* Confusion 4 */}
          <div className="bg-card rounded-lg border p-5">
            <div className="flex items-start gap-3">
              <div className="flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full bg-rose-500/20">
                <AlertTriangle className="h-4 w-4 text-rose-400" />
              </div>
              <div>
                <h4 className="font-semibold text-rose-400">
                  Confusion #4: Mixing Up Odds Formats
                </h4>
                <p className="text-muted-foreground mt-2 text-sm">
                  Someone says "Lakers are 1.5" - do they mean 1.5 decimal odds (very heavy
                  favorite) or are they confusing it with a 1.5 point spread? At 1.5 decimal, you'd
                  only profit 50¢ per dollar wagered.
                </p>
                <div className="bg-muted/50 mt-3 rounded p-3">
                  <p className="mb-1 text-xs font-medium">Context clue:</p>
                  <p className="text-muted-foreground text-xs">
                    Point spreads have a + or - sign and usually involve half-points (-1.5, +3.5).
                    Decimal odds don't use signs and are always greater than 1.00.
                  </p>
                </div>
              </div>
            </div>
          </div>

          {/* Confusion 5 */}
          <div className="bg-card rounded-lg border p-5">
            <div className="flex items-start gap-3">
              <div className="flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full bg-rose-500/20">
                <AlertTriangle className="h-4 w-4 text-rose-400" />
              </div>
              <div>
                <h4 className="font-semibold text-rose-400">
                  Confusion #5: "Better Odds" Ambiguity
                </h4>
                <p className="text-muted-foreground mt-2 text-sm">
                  When someone says they "got better odds" on a team, they might mean: (a) they got
                  a better payout, or (b) the probability shifted in favor of that team winning.
                  These are opposite things!
                </p>
                <div className="bg-muted/50 mt-3 rounded p-3">
                  <p className="mb-1 text-xs font-medium">Clarifying question:</p>
                  <p className="text-muted-foreground text-xs">
                    "Do you mean you're getting paid more, or the team became more likely to win?" A
                    team becoming more likely to win means you get paid LESS.
                  </p>
                </div>
              </div>
            </div>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 7: MONEYLINE VS SPREAD */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="Different Tools">
            Think of moneyline and spread as different tools for the same job. A hammer and
            screwdriver both help build things, but you use them in different situations.
          </TipBlock>
        }
      >
        <SectionHeader
          title="Moneyline vs Point Spread"
          subtitle="When to use which"
        />

        <p className="text-muted-foreground leading-relaxed">
          The most common confusion is between moneyline and{' '}
          <GlossaryTooltip term={terms.pointSpread}>point spread</GlossaryTooltip> bets. They're
          fundamentally different questions:
        </p>

        <div className="mt-6 grid gap-4 sm:grid-cols-2">
          <div className="rounded-lg border p-4">
            <div className="mb-3 flex items-center gap-2">
              <Trophy className="text-primary h-5 w-5" />
              <h4 className="font-semibold">Moneyline</h4>
            </div>
            <p className="mb-2 text-sm font-medium">"Who wins?"</p>
            <ul className="text-muted-foreground space-y-1 text-sm">
              <li>• Win by 1 or win by 100 - same payout</li>
              <li>• Favorites cost more, underdogs less</li>
              <li>• Simple binary outcome</li>
            </ul>
          </div>

          <div className="rounded-lg border p-4">
            <div className="mb-3 flex items-center gap-2">
              <ArrowLeftRight className="text-primary h-5 w-5" />
              <h4 className="font-semibold">Point Spread</h4>
            </div>
            <p className="mb-2 text-sm font-medium">"Who wins after the handicap?"</p>
            <ul className="text-muted-foreground space-y-1 text-sm">
              <li>• Margin of victory matters</li>
              <li>• Usually -110 both sides (similar pricing)</li>
              <li>• Underdog gets a head start</li>
            </ul>
          </div>
        </div>

        <ExampleBox title="Lakers -7 vs Celtics">
          <div className="space-y-4">
            <p className="text-sm font-medium">If Lakers win 105-100:</p>
            <div className="grid gap-3 text-sm sm:grid-cols-2">
              <div className="bg-muted/50 rounded p-3">
                <div className="mb-1 font-medium text-emerald-400">Moneyline on Lakers</div>
                <p className="text-muted-foreground">✓ Lakers won the game. You win your bet.</p>
              </div>
              <div className="bg-muted/50 rounded p-3">
                <div className="mb-1 font-medium text-rose-400">Spread -7 on Lakers</div>
                <p className="text-muted-foreground">
                  ✗ Lakers only won by 5. They needed to win by 7+. You lose.
                </p>
              </div>
            </div>
            <p className="text-muted-foreground border-t pt-3 text-xs">
              Same game, same outcome, different bet results. Spread bets care about margin;
              moneyline bets don't.
            </p>
          </div>
        </ExampleBox>

        <div className="bg-card mt-6 rounded-lg border p-4">
          <h4 className="mb-3 font-semibold">When to Choose Moneyline</h4>
          <ul className="text-muted-foreground space-y-2 text-sm">
            <li className="flex items-start gap-2">
              <span className="mt-1 text-emerald-400">✓</span>
              <span>You think a team will win but aren't sure by how much</span>
            </li>
            <li className="flex items-start gap-2">
              <span className="mt-1 text-emerald-400">✓</span>
              <span>
                You're betting on an underdog upset (moneyline pays more than covering the spread)
              </span>
            </li>
            <li className="flex items-start gap-2">
              <span className="mt-1 text-emerald-400">✓</span>
              <span>The game could be close and you don't want to sweat the margin</span>
            </li>
          </ul>
        </div>

        <div className="bg-card mt-4 rounded-lg border p-4">
          <h4 className="mb-3 font-semibold">When to Choose Spread</h4>
          <ul className="text-muted-foreground space-y-2 text-sm">
            <li className="flex items-start gap-2">
              <span className="mt-1 text-emerald-400">✓</span>
              <span>
                The moneyline on a heavy favorite is too expensive (risk $400 to win $100)
              </span>
            </li>
            <li className="flex items-start gap-2">
              <span className="mt-1 text-emerald-400">✓</span>
              <span>You have a strong opinion on the margin of victory</span>
            </li>
            <li className="flex items-start gap-2">
              <span className="mt-1 text-emerald-400">✓</span>
              <span>You want roughly 50/50 odds regardless of who's favored</span>
            </li>
          </ul>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 8: REAL-WORLD CONTEXT */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="Follow the Money">
            When you hear "sharp money came in on the underdog," that means professional bettors bet
            heavily on the less-favored team, often moving the line.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="How People Talk About Moneylines"
          subtitle="Understanding the lingo in context"
        />

        <p className="text-muted-foreground leading-relaxed">
          Now that you understand what moneylines are, let's cover how you'll hear people talk about
          them. This will help you follow conversations and not get confused.
        </p>

        <div className="mt-6 space-y-4">
          <div className="bg-card rounded-lg border p-4">
            <div className="mb-2 flex items-center gap-2">
              <Users className="text-muted-foreground h-4 w-4" />
              <span className="text-sm font-medium">"I took the Lakers moneyline"</span>
            </div>
            <p className="text-muted-foreground text-sm">
              Meaning: They bet on the Lakers to win (not on the spread or total).
            </p>
          </div>

          <div className="bg-card rounded-lg border p-4">
            <div className="mb-2 flex items-center gap-2">
              <Users className="text-muted-foreground h-4 w-4" />
              <span className="text-sm font-medium">"The moneyline moved from -150 to -180"</span>
            </div>
            <p className="text-muted-foreground text-sm">
              Meaning: The favorite became more expensive to bet on. More people are betting on
              them, or sharp money came in. The team is now seen as more likely to win.
            </p>
          </div>

          <div className="bg-card rounded-lg border p-4">
            <div className="mb-2 flex items-center gap-2">
              <Users className="text-muted-foreground h-4 w-4" />
              <span className="text-sm font-medium">"I'm getting +200 on the underdog"</span>
            </div>
            <p className="text-muted-foreground text-sm">
              Meaning: They're betting on the underdog at odds that would pay $200 profit on a $100
              bet. In Kalshi terms, this would be roughly a 33¢ price.
            </p>
          </div>

          <div className="bg-card rounded-lg border p-4">
            <div className="mb-2 flex items-center gap-2">
              <Users className="text-muted-foreground h-4 w-4" />
              <span className="text-sm font-medium">"The moneyline is too juiced"</span>
            </div>
            <p className="text-muted-foreground text-sm">
              Meaning: The <GlossaryTooltip term={terms.juice}>juice</GlossaryTooltip> (bookmaker's
              margin) is too high. If favorites are -150 and underdogs are +120, the gap between
              those is the "juice" - the house edge.
            </p>
          </div>

          <div className="bg-card rounded-lg border p-4">
            <div className="mb-2 flex items-center gap-2">
              <Users className="text-muted-foreground h-4 w-4" />
              <span className="text-sm font-medium">"I parlayed three moneylines"</span>
            </div>
            <p className="text-muted-foreground text-sm">
              Meaning: They combined three separate moneyline bets into one parlay. All three teams
              must win for the bet to pay out, but the payout multiplies.
            </p>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 9: KALSHI SPECIFIC */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="Price = Probability">
            The most elegant thing about Kalshi: you never need to convert odds. 65¢ means 65%. No
            mental math required.
          </TipBlock>
        }
      >
        <SectionHeader
          title="Moneylines on Kalshi"
          subtitle="How Kalshi makes this simpler"
        />

        <p className="text-muted-foreground leading-relaxed">
          On traditional sportsbooks, you see odds like "-150" or "+200" and have to do mental math
          to understand the actual probabilities. Kalshi makes this transparent.
        </p>

        <div className="bg-card mt-6 rounded-lg border p-5">
          <h4 className="mb-4 font-semibold">Kalshi's Approach</h4>
          <div className="space-y-4">
            <div className="flex items-start gap-3">
              <div className="flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full bg-emerald-500/20">
                <span className="font-bold text-emerald-400">1</span>
              </div>
              <div>
                <p className="font-medium">Price = Probability</p>
                <p className="text-muted-foreground text-sm">
                  A contract at 65¢ implies 65% probability. No conversion needed.
                </p>
              </div>
            </div>

            <div className="flex items-start gap-3">
              <div className="flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full bg-emerald-500/20">
                <span className="font-bold text-emerald-400">2</span>
              </div>
              <div>
                <p className="font-medium">Always pays $1</p>
                <p className="text-muted-foreground text-sm">
                  Every contract pays $1 if you're right. Your profit is always $1 minus your cost.
                </p>
              </div>
            </div>

            <div className="flex items-start gap-3">
              <div className="flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full bg-emerald-500/20">
                <span className="font-bold text-emerald-400">3</span>
              </div>
              <div>
                <p className="font-medium">Two Markets Per Game</p>
                <p className="text-muted-foreground text-sm">
                  Kalshi creates separate markets for "Will Lakers win?" and "Will Celtics win?"
                  This is covered in the YES/NO Markets lesson.
                </p>
              </div>
            </div>
          </div>
        </div>

        <ExampleBox title="Traditional vs Kalshi">
          <div className="grid gap-4 sm:grid-cols-2">
            <div>
              <div className="text-muted-foreground mb-2 text-sm font-medium">
                Traditional Sportsbook
              </div>
              <div className="font-mono text-lg">
                Lakers <span className="text-amber-400">-150</span>
              </div>
              <div className="font-mono text-lg">
                Celtics <span className="text-violet-400">+130</span>
              </div>
              <p className="text-muted-foreground mt-2 text-xs">
                "I need to bet $150 to win $100 on Lakers... and I'd win $130 on $100 for
                Celtics..."
              </p>
            </div>
            <div>
              <div className="text-muted-foreground mb-2 text-sm font-medium">Kalshi</div>
              <div className="font-mono text-lg">
                Lakers YES <span className="text-amber-400">60¢</span>
              </div>
              <div className="font-mono text-lg">
                Celtics YES <span className="text-violet-400">40¢</span>
              </div>
              <p className="text-muted-foreground mt-2 text-xs">
                "60% Lakers, 40% Celtics. I pay 60¢ to profit 40¢, or pay 40¢ to profit 60¢."
              </p>
            </div>
          </div>
        </ExampleBox>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 10: KEY TAKEAWAYS */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="Key Takeaways"
          subtitle="What to remember"
        />

        <div className="grid gap-4 sm:grid-cols-2 lg:grid-cols-3">
          <ConceptCard
            title="Simple Question"
            description="Moneyline = who wins? That's it. Margin doesn't matter."
            icon={<Trophy className="h-4 w-4" />}
          />
          <ConceptCard
            title="The Name"
            description="'Line' from chalkboard rows, 'money' from varying stakes."
            icon={<History className="h-4 w-4" />}
          />
          <ConceptCard
            title="Price = Probability"
            description="On Kalshi, 65¢ means 65% implied probability."
            icon={<Calculator className="h-4 w-4" />}
          />
          <ConceptCard
            title="Risk vs Reward"
            description="Favorites: risk more, win less. Underdogs: risk less, win more."
            icon={<Scale className="h-4 w-4" />}
          />
          <ConceptCard
            title="Not the Spread"
            description="Moneyline = who wins. Spread = margin of victory."
            icon={<ArrowLeftRight className="h-4 w-4" />}
          />
          <ConceptCard
            title="Many Formats"
            description="American, decimal, fractional - all mean the same thing."
            icon={<Globe className="h-4 w-4" />}
          />
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* NEXT STEPS */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="What's Next?"
          subtitle="Continue your learning journey"
        />

        <p className="text-muted-foreground mb-6 leading-relaxed">
          Now that you understand moneylines deeply, the next step is understanding how Kalshi
          structures these as YES/NO markets - which creates unique opportunities for traders.
        </p>

        {nextGuide && (
          <NextStepBlock
            href={`/${nextGuide.slug}`}
            title={nextGuide.title}
            description="Learn how Kalshi's unique dual market structure works and why it matters for your trading."
            buttonText="Continue to Next Lesson"
          />
        )}
      </LessonRow>
    </LessonLayout>
  );
}
