import { useState } from 'react';
import { LessonRow, SectionHeader } from '../LessonLayout';
import { GlossaryTooltip } from '@/components/atoms/GlossaryTooltip';
import { Formula } from '@/components/atoms/Formula';
import { TipBlock, InsightBlock, WarningBlock } from '../blocks';

const terms = {
  expectedValue: {
    term: 'Expected Value',
    definition:
      'The average outcome if you repeated a bet infinitely many times. Positive EV means you profit long-term.',
    category: 'Strategy' as const,
  },
  edge: {
    term: 'Edge',
    definition:
      'Your advantage over the market, measured as a percentage. A 5% edge means you expect to profit 5 cents per dollar wagered.',
    category: 'Strategy' as const,
  },
  impliedProbability: {
    term: 'Implied Probability',
    definition:
      'The probability embedded in market prices. A 40¢ YES contract implies a 40% chance of the event occurring.',
    category: 'Market' as const,
  },
  trueOdds: {
    term: 'True Odds',
    definition:
      'Your honest assessment of the actual probability of an event, independent of market prices.',
    category: 'Strategy' as const,
  },
};

// Interactive widget for building EV intuition
function EVSimulatorWidget() {
  const [probability, setProbability] = useState(55);
  const [marketPrice, setMarketPrice] = useState(50);
  const [numBets, setNumBets] = useState(100);
  const [simResults, setSimResults] = useState<number[] | null>(null);

  const winPayout = 100 - marketPrice; // cents profit on win
  const lossAmount = marketPrice; // cents lost on loss
  const ev = (probability / 100) * winPayout - ((100 - probability) / 100) * lossAmount;
  const edgePercent = (ev / marketPrice) * 100;

  const runSimulation = () => {
    const results: number[] = [];
    let bankroll = 0;

    for (let i = 0; i < numBets; i++) {
      const won = Math.random() < probability / 100;
      bankroll += won ? winPayout : -lossAmount;
      results.push(bankroll);
    }

    setSimResults(results);
  };

  const finalResult = simResults ? simResults[simResults.length - 1] : 0;
  const theoreticalResult = ev * numBets;

  return (
    <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-6">
      <h4 className="mb-4 text-lg font-semibold text-white">
        EV Simulator: See Expected Value in Action
      </h4>

      <div className="mb-6 grid grid-cols-1 gap-4 md:grid-cols-3">
        <div>
          <label className="mb-1 block text-sm text-slate-400">Your True Probability (%)</label>
          <input
            type="range"
            min="1"
            max="99"
            value={probability}
            onChange={(e) => setProbability(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center font-mono text-white">{probability}%</div>
        </div>

        <div>
          <label className="mb-1 block text-sm text-slate-400">Market Price (cents)</label>
          <input
            type="range"
            min="1"
            max="99"
            value={marketPrice}
            onChange={(e) => setMarketPrice(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center font-mono text-white">{marketPrice}¢</div>
        </div>

        <div>
          <label className="mb-1 block text-sm text-slate-400">Number of Bets</label>
          <input
            type="range"
            min="10"
            max="1000"
            step="10"
            value={numBets}
            onChange={(e) => setNumBets(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center font-mono text-white">{numBets} bets</div>
        </div>
      </div>

      <div className="mb-6 grid grid-cols-2 gap-4 md:grid-cols-4">
        <div className="rounded bg-slate-900/50 p-3 text-center">
          <div className="text-xs text-slate-400">Win Payout</div>
          <div className="text-xl font-bold text-green-400">+{winPayout}¢</div>
        </div>
        <div className="rounded bg-slate-900/50 p-3 text-center">
          <div className="text-xs text-slate-400">Loss Amount</div>
          <div className="text-xl font-bold text-red-400">-{lossAmount}¢</div>
        </div>
        <div className="rounded bg-slate-900/50 p-3 text-center">
          <div className="text-xs text-slate-400">EV per Bet</div>
          <div className={`text-xl font-bold ${ev >= 0 ? 'text-green-400' : 'text-red-400'}`}>
            {ev >= 0 ? '+' : ''}
            {ev.toFixed(2)}¢
          </div>
        </div>
        <div className="rounded bg-slate-900/50 p-3 text-center">
          <div className="text-xs text-slate-400">Your Edge</div>
          <div
            className={`text-xl font-bold ${edgePercent >= 0 ? 'text-green-400' : 'text-red-400'}`}
          >
            {edgePercent >= 0 ? '+' : ''}
            {edgePercent.toFixed(1)}%
          </div>
        </div>
      </div>

      <button
        onClick={runSimulation}
        className="mb-4 w-full rounded-lg bg-blue-600 py-3 font-semibold text-white hover:bg-blue-700"
      >
        Run {numBets} Bet Simulation
      </button>

      {simResults && (
        <div className="space-y-4">
          <div className="rounded bg-slate-900/50 p-4">
            <div className="mb-2 flex items-center justify-between">
              <span className="text-slate-400">Theoretical EV ({numBets} bets):</span>
              <span
                className={`font-bold ${theoreticalResult >= 0 ? 'text-green-400' : 'text-red-400'}`}
              >
                {theoreticalResult >= 0 ? '+' : ''}${(theoreticalResult / 100).toFixed(2)}
              </span>
            </div>
            <div className="flex items-center justify-between">
              <span className="text-slate-400">Actual Simulation Result:</span>
              <span className={`font-bold ${finalResult >= 0 ? 'text-green-400' : 'text-red-400'}`}>
                {finalResult >= 0 ? '+' : ''}${(finalResult / 100).toFixed(2)}
              </span>
            </div>
          </div>

          <div className="text-sm text-slate-400">
            {Math.abs(finalResult - theoreticalResult) > Math.abs(theoreticalResult) * 0.5 ? (
              <p className="text-amber-400">
                Notice how the actual result differs from theoretical EV? This is variance. Run more
                simulations or increase the number of bets to see EV converge to theory.
              </p>
            ) : (
              <p className="text-green-400">
                The actual result is close to theoretical EV. With enough bets, reality converges
                toward expected value.
              </p>
            )}
          </div>
        </div>
      )}
    </div>
  );
}

// Interactive widget for understanding implied probability
function ImpliedProbabilityWidget() {
  const [price, setPrice] = useState(45);
  const [yourEstimate, setYourEstimate] = useState(55);

  const impliedProb = price;
  const difference = yourEstimate - impliedProb;
  const ev = (yourEstimate / 100) * (100 - price) - ((100 - yourEstimate) / 100) * price;

  return (
    <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-6">
      <h4 className="mb-4 text-lg font-semibold text-white">
        Finding Edge: Your Probability vs Market Probability
      </h4>

      <div className="mb-6 grid grid-cols-1 gap-6 md:grid-cols-2">
        <div>
          <label className="mb-2 block text-sm text-slate-400">
            Market YES Price (Implied Probability)
          </label>
          <input
            type="range"
            min="5"
            max="95"
            value={price}
            onChange={(e) => setPrice(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center">
            <span className="text-2xl font-bold text-white">{price}¢</span>
            <span className="ml-2 text-slate-400">= {impliedProb}% implied</span>
          </div>
        </div>

        <div>
          <label className="mb-2 block text-sm text-slate-400">Your Estimated Probability</label>
          <input
            type="range"
            min="5"
            max="95"
            value={yourEstimate}
            onChange={(e) => setYourEstimate(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center">
            <span className="text-2xl font-bold text-blue-400">{yourEstimate}%</span>
            <span className="ml-2 text-slate-400">your estimate</span>
          </div>
        </div>
      </div>

      <div className="rounded-lg bg-slate-900/50 p-4">
        <div className="grid grid-cols-3 gap-4 text-center">
          <div>
            <div className="mb-1 text-xs text-slate-400">Probability Gap</div>
            <div
              className={`text-xl font-bold ${difference > 0 ? 'text-green-400' : difference < 0 ? 'text-red-400' : 'text-slate-400'}`}
            >
              {difference > 0 ? '+' : ''}
              {difference}%
            </div>
          </div>
          <div>
            <div className="mb-1 text-xs text-slate-400">EV per Contract</div>
            <div
              className={`text-xl font-bold ${ev > 0 ? 'text-green-400' : ev < 0 ? 'text-red-400' : 'text-slate-400'}`}
            >
              {ev > 0 ? '+' : ''}
              {ev.toFixed(1)}¢
            </div>
          </div>
          <div>
            <div className="mb-1 text-xs text-slate-400">Recommendation</div>
            <div
              className={`text-xl font-bold ${
                difference > 5
                  ? 'text-green-400'
                  : difference < -5
                    ? 'text-red-400'
                    : 'text-slate-400'
              }`}
            >
              {difference > 5 ? 'BUY YES' : difference < -5 ? 'BUY NO' : 'NO EDGE'}
            </div>
          </div>
        </div>

        <div className="mt-4 text-sm text-slate-400">
          {Math.abs(difference) < 5 && (
            <p>
              The gap between your estimate and market price is small. After accounting for fees and
              uncertainty, there&apos;s likely no edge here.
            </p>
          )}
          {difference >= 5 && (
            <p className="text-green-400">
              You believe this is more likely than the market thinks. If you&apos;re right, buying
              YES has positive expected value.
            </p>
          )}
          {difference <= -5 && (
            <p className="text-green-400">
              You believe this is less likely than the market thinks. If you&apos;re right, buying
              NO has positive expected value.
            </p>
          )}
        </div>
      </div>
    </div>
  );
}

// Widget showing the coin flip thought experiment
function CoinFlipIntuitionWidget() {
  const [flips, setFlips] = useState<boolean[]>([]);
  const [isFlipping, setIsFlipping] = useState(false);

  const flipCoin = async () => {
    setIsFlipping(true);
    const newFlips: boolean[] = [];

    for (let i = 0; i < 20; i++) {
      // This coin is biased - 60% heads
      newFlips.push(Math.random() < 0.6);
      setFlips([...newFlips]);
      await new Promise((r) => setTimeout(r, 150));
    }

    setIsFlipping(false);
  };

  const heads = flips.filter((f) => f).length;
  const tails = flips.length - heads;
  const profit = heads * 50 - tails * 60; // Win $50 on heads, lose $60 on tails

  return (
    <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-6">
      <h4 className="mb-2 text-lg font-semibold text-white">
        The Biased Coin: Building EV Intuition
      </h4>
      <p className="mb-4 text-sm text-slate-400">
        This coin lands heads 60% of the time. You win $50 on heads, lose $60 on tails. Is this a
        good bet?
      </p>

      <div className="mb-4 rounded bg-slate-900/50 p-4">
        <div className="mb-2 text-sm text-slate-400">EV Calculation:</div>
        <div className="text-white">
          <Formula block>
            {
              'EV = (60\\% \\times \\$50) - (40\\% \\times \\$60) = \\$30 - \\$24 = \\textcolor{#4ade80}{+\\$6 \\text{ per flip}}'
            }
          </Formula>
        </div>
      </div>

      <button
        onClick={flipCoin}
        disabled={isFlipping}
        className="mb-4 w-full rounded-lg bg-purple-600 py-3 font-semibold text-white hover:bg-purple-700 disabled:bg-slate-600"
      >
        {isFlipping ? 'Flipping...' : 'Flip 20 Times'}
      </button>

      {flips.length > 0 && (
        <div className="space-y-4">
          <div className="flex flex-wrap gap-1">
            {flips.map((isHeads, i) => (
              <div
                key={i}
                className={`flex h-8 w-8 items-center justify-center rounded-full text-xs font-bold ${
                  isHeads ? 'bg-green-600 text-white' : 'bg-red-600 text-white'
                }`}
              >
                {isHeads ? 'H' : 'T'}
              </div>
            ))}
          </div>

          <div className="grid grid-cols-3 gap-4 text-center">
            <div className="rounded bg-slate-900/50 p-3">
              <div className="text-xs text-slate-400">Heads</div>
              <div className="text-xl font-bold text-green-400">{heads}</div>
            </div>
            <div className="rounded bg-slate-900/50 p-3">
              <div className="text-xs text-slate-400">Tails</div>
              <div className="text-xl font-bold text-red-400">{tails}</div>
            </div>
            <div className="rounded bg-slate-900/50 p-3">
              <div className="text-xs text-slate-400">Profit</div>
              <div
                className={`text-xl font-bold ${profit >= 0 ? 'text-green-400' : 'text-red-400'}`}
              >
                {profit >= 0 ? '+' : ''}${profit}
              </div>
            </div>
          </div>

          <div className="text-sm text-slate-400">
            <p>
              Expected profit for 20 flips: <span className="text-green-400">+$120</span> (20 × $6)
            </p>
            <p>
              Actual profit:{' '}
              <span className={profit >= 0 ? 'text-green-400' : 'text-red-400'}>
                {profit >= 0 ? '+' : ''}${profit}
              </span>
            </p>
            {profit < 0 && (
              <p className="mt-2 text-amber-400">
                Even with a +EV bet, you lost money this time. This is variance. Over thousands of
                flips, you&apos;d be profitable.
              </p>
            )}
          </div>
        </div>
      )}
    </div>
  );
}

export function ProbabilityExpectedValueLesson() {
  return (
    <div className="space-y-8">
      {/* Introduction */}
      <section>
        <SectionHeader
          title="Why Expected Value is Everything"
          subtitle="The single most important concept in profitable trading"
        />
        <LessonRow
          aside={
            <InsightBlock title="The Mental Shift">
              Profitable trading isn't about being right on any single trade. It's about making +EV
              decisions repeatedly. The outcome of one trade tells you almost nothing.
            </InsightBlock>
          }
        >
          <p className="leading-relaxed text-slate-300">
            If you learn nothing else from this entire course, learn this:
            <strong className="text-white">
              {' '}
              profitable trading is about making positive expected value decisions, repeatedly, over
              time
            </strong>
            . Not about being right on any single trade. Not about predicting the future perfectly.
            Not about never losing.
          </p>
          <p className="mt-4 leading-relaxed text-slate-300">
            <GlossaryTooltip term={terms.expectedValue}>Expected value</GlossaryTooltip> (EV) is the
            mathematical answer to the question: &ldquo;If I made this exact bet 10,000 times, what
            would happen to my bankroll?&rdquo; This mental model—thinking in terms of repeated
            decisions rather than individual outcomes—is what separates professional traders from
            gamblers.
          </p>
        </LessonRow>
      </section>

      {/* The Coin Flip Intuition Builder */}
      <section>
        <SectionHeader
          title="Building Intuition: The Biased Coin"
          subtitle="Understanding EV through simulation"
        />
        <LessonRow>
          <p className="mb-6 leading-relaxed text-slate-300">
            Before we dive into formulas, let&apos;s build intuition. Imagine a coin that lands
            heads 60% of the time. Someone offers you a bet: $50 if it lands heads, you pay $60 if
            tails. Should you take this bet?
          </p>
          <CoinFlipIntuitionWidget />
          <p className="mt-6 leading-relaxed text-slate-300">
            The key insight: even though you sometimes lose (even several times in a row!), the
            mathematics guarantee that over enough flips, you&apos;ll profit. This is what
            &ldquo;positive EV&rdquo; means in practice. You don&apos;t need to win every flip. You
            just need the math to be in your favor.
          </p>
        </LessonRow>
      </section>

      {/* The EV Formula */}
      <section>
        <SectionHeader
          title="The Expected Value Formula"
          subtitle="From intuition to calculation"
        />
        <LessonRow
          aside={
            <TipBlock title="The Simple Test">
              Before any trade, do this quick math: (Your probability × Profit) minus ((1 - Your
              probability) × Loss). If it's positive, you have edge. If not, pass.
            </TipBlock>
          }
        >
          <div className="mb-6 rounded-lg border border-slate-700 bg-slate-800/50 p-6">
            <h4 className="mb-4 text-lg font-semibold text-white">The Formula</h4>
            <div className="mb-4 text-center text-xl text-white">
              <Formula block>{'EV = (P_{win} \\times Profit) - (P_{lose} \\times Loss)'}</Formula>
            </div>
            <div className="text-sm text-slate-400">
              <p>
                <strong>
                  <Formula>{'P_{win}'}</Formula>
                </strong>{' '}
                = Your estimated probability of winning
              </p>
              <p>
                <strong>Profit</strong> = What you gain if you win
              </p>
              <p>
                <strong>
                  <Formula>{'P_{lose}'}</Formula>
                </strong>{' '}
                = Your estimated probability of losing (<Formula>{'1 - P_{win}'}</Formula>)
              </p>
              <p>
                <strong>Loss</strong> = What you lose if you lose (your stake)
              </p>
            </div>
          </div>

          <p className="leading-relaxed text-slate-300">
            Let&apos;s apply this to Kalshi. You see a YES contract priced at 40¢. You believe the
            true probability is 55%. Should you buy?
          </p>

          <div className="mt-4 rounded-lg bg-slate-900/50 p-4 text-sm">
            <p className="text-slate-400">If you buy YES at 40¢:</p>
            <p className="text-white">- Win: You get $1.00, profit = 60¢</p>
            <p className="text-white">- Lose: You lose your 40¢ stake</p>
            <p className="mt-2 text-slate-400">EV calculation:</p>
            <div className="mt-2 text-white">
              <Formula block>{'EV = (55\\% \\times 60¢) - (45\\% \\times 40¢)'}</Formula>
              <Formula block>
                {'EV = 33¢ - 18¢ = \\textcolor{#4ade80}{+15¢ \\text{ per contract}}'}
              </Formula>
            </div>
          </div>

          <p className="mt-4 leading-relaxed text-slate-300">
            This bet has positive expected value. If your probability estimate is correct,
            you&apos;ll profit 15¢ per contract on average over many such trades.
          </p>
        </LessonRow>
      </section>

      {/* Implied Probability */}
      <section>
        <SectionHeader
          title="Reading Market Prices as Probabilities"
          subtitle="What the market is telling you"
        />
        <LessonRow
          aside={
            <InsightBlock title="Price = Probability">
              On Kalshi, prices directly equal implied probabilities. A 40¢ YES contract means the
              market believes there's a 40% chance. This makes edge calculation simple.
            </InsightBlock>
          }
        >
          <p className="leading-relaxed text-slate-300">
            On Kalshi, prices directly translate to probabilities. A YES contract at 40¢ means the
            market collectively believes there&apos;s a 40% chance of YES occurring. This is called
            the{' '}
            <GlossaryTooltip term={terms.impliedProbability}>implied probability</GlossaryTooltip>.
          </p>

          <div className="my-6 rounded-lg border border-slate-700 bg-slate-800/50 p-6">
            <h4 className="mb-4 text-lg font-semibold text-white">Price to Probability</h4>
            <div className="grid grid-cols-2 gap-4">
              <div>
                <p className="mb-2 text-sm text-slate-400">For YES contracts:</p>
                <p className="text-white">
                  <Formula>{'\\text{Implied Prob} = \\text{Price in cents}'}</Formula>
                </p>
                <p className="mt-2 text-sm text-slate-400">35¢ = 35% implied probability</p>
              </div>
              <div>
                <p className="mb-2 text-sm text-slate-400">For NO contracts:</p>
                <p className="text-white">
                  <Formula>{'\\text{Implied Prob} = 100 - \\text{YES price}'}</Formula>
                </p>
                <p className="mt-2 text-sm text-slate-400">35¢ YES = 65% NO probability</p>
              </div>
            </div>
          </div>

          <p className="leading-relaxed text-slate-300">
            <strong className="text-white">
              Your edge comes from disagreeing with the market.
            </strong>{' '}
            If you think the true probability is higher than implied, buy YES. If you think
            it&apos;s lower, buy NO. The size of your edge is the gap between your estimated
            probability and the market&apos;s implied probability.
          </p>
        </LessonRow>
      </section>

      {/* Interactive Edge Finder */}
      <section>
        <SectionHeader
          title="Finding Your Edge"
          subtitle="Comparing your probability to the market's"
        />
        <LessonRow>
          <p className="mb-6 leading-relaxed text-slate-300">
            The key question for every trade: &ldquo;Do I believe the probability is different from
            what the market implies, and by how much?&rdquo; Use this tool to explore how
            probability disagreements translate into edge.
          </p>
          <ImpliedProbabilityWidget />
        </LessonRow>
      </section>

      {/* EV Simulator */}
      <section>
        <SectionHeader
          title="Seeing EV Over Time"
          subtitle="How positive EV compounds"
        />
        <LessonRow>
          <p className="mb-6 leading-relaxed text-slate-300">
            The power of positive EV becomes clear over many bets. Even when individual bets lose,
            the mathematics pull your results toward the expected value over time. This simulator
            shows how EV works in practice.
          </p>
          <EVSimulatorWidget />
          <p className="mt-6 leading-relaxed text-slate-300">
            Notice how the results converge toward theoretical EV as you increase the number of
            bets. This is the <strong className="text-white">law of large numbers</strong> in
            action—the mathematical foundation of all profitable trading.
          </p>
        </LessonRow>
      </section>

      {/* Why Most Bettors Get This Wrong */}
      <section>
        <SectionHeader
          title="Why Most Traders Ignore EV"
          subtitle="Common mistakes and how to avoid them"
        />
        <LessonRow
          aside={
            <WarningBlock title="Outcome Bias">
              Judging a decision by its outcome instead of its process is the most common error in
              trading. A winning trade made for bad reasons is still a bad trade.
            </WarningBlock>
          }
        >
          <div className="space-y-6">
            <div className="rounded-lg border border-red-800/50 bg-red-900/20 p-4">
              <h4 className="mb-2 font-semibold text-red-400">
                Mistake #1: Outcome-Based Thinking
              </h4>
              <p className="text-sm text-slate-300">
                &ldquo;I lost, so it must have been a bad bet.&rdquo; Wrong. A positive EV bet that
                loses is still a good bet. Judging decisions by outcomes rather than process is the
                most common error in trading.
              </p>
            </div>

            <div className="rounded-lg border border-red-800/50 bg-red-900/20 p-4">
              <h4 className="mb-2 font-semibold text-red-400">Mistake #2: Seeking Certainty</h4>
              <p className="text-sm text-slate-300">
                &ldquo;I only bet when I&apos;m sure I&apos;m right.&rdquo; Certainty doesn&apos;t
                exist in prediction markets. If you wait for certainty, you&apos;ll either never bet
                or bet on prices that already reflect the information (no edge).
              </p>
            </div>

            <div className="rounded-lg border border-red-800/50 bg-red-900/20 p-4">
              <h4 className="mb-2 font-semibold text-red-400">
                Mistake #3: Ignoring the Other Side
              </h4>
              <p className="text-sm text-slate-300">
                &ldquo;Lakers will probably win, so I should bet YES.&rdquo; But at what price? If
                YES is priced at 80¢ and your estimate is 75%, you have negative EV despite Lakers
                being &ldquo;likely to win.&rdquo;
              </p>
            </div>

            <div className="rounded-lg border border-red-800/50 bg-red-900/20 p-4">
              <h4 className="mb-2 font-semibold text-red-400">
                Mistake #4: Small Sample Conclusions
              </h4>
              <p className="text-sm text-slate-300">
                &ldquo;I lost 5 of my last 7 bets, my strategy doesn&apos;t work.&rdquo; Variance is
                brutal in small samples. A 55% win rate strategy will have 7+ bet losing streaks
                regularly. You need hundreds of bets to evaluate a strategy.
              </p>
            </div>
          </div>
        </LessonRow>
      </section>

      {/* Thinking in Probability Distributions */}
      <section>
        <SectionHeader
          title="Advanced: Thinking in Distributions"
          subtitle="Moving beyond point estimates"
        />
        <LessonRow
          aside={
            <TipBlock title="Confidence Matters">
              Not all 55% estimates are equal. If you're confident the true probability is 53-57%,
              that's actionable. If it could be 40-70%, you don't really know enough to trade.
            </TipBlock>
          }
        >
          <p className="leading-relaxed text-slate-300">
            Professional traders don&apos;t think &ldquo;the probability is 55%.&rdquo; They think
            &ldquo;the probability is somewhere between 50-60%, most likely around 55%.&rdquo; This
            is called <strong className="text-white">thinking in distributions</strong>.
          </p>

          <div className="my-6 rounded-lg border border-slate-700 bg-slate-800/50 p-6">
            <h4 className="mb-4 text-lg font-semibold text-white">Why Distributions Matter</h4>
            <div className="space-y-4 text-slate-300">
              <p>
                <strong className="text-blue-400">Scenario A:</strong> You&apos;re highly confident
                the probability is 55% (maybe it&apos;s 53-57%).
              </p>
              <p>
                <strong className="text-purple-400">Scenario B:</strong> You&apos;re uncertain—it
                could be anywhere from 40-70%, best guess 55%.
              </p>
              <p>
                Both have the same point estimate (55%), but Scenario A is a much better trading
                opportunity. In Scenario B, your &ldquo;edge&rdquo; might not even exist—the true
                probability could be 45%, making your bet negative EV.
              </p>
            </div>
          </div>

          <p className="leading-relaxed text-slate-300">
            <strong className="text-white">Rule of thumb:</strong> The wider your uncertainty, the
            larger the gap needs to be between your estimate and market price before you have a
            tradeable edge. If you&apos;re uncertain, demand more edge to compensate.
          </p>
        </LessonRow>
      </section>

      {/* The EV Mindset */}
      <section>
        <SectionHeader
          title="Adopting the EV Mindset"
          subtitle="How professionals think about every trade"
        />
        <LessonRow>
          <p className="mb-6 leading-relaxed text-slate-300">
            Every time you consider a trade, run through this mental checklist:
          </p>

          <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-6">
            <ol className="space-y-4 text-slate-300">
              <li className="flex gap-3">
                <span className="flex h-6 w-6 flex-shrink-0 items-center justify-center rounded-full bg-blue-600 text-sm text-white">
                  1
                </span>
                <div>
                  <strong className="text-white">What does the market imply?</strong>
                  <p className="text-sm text-slate-400">
                    Convert the price to implied probability.
                  </p>
                </div>
              </li>
              <li className="flex gap-3">
                <span className="flex h-6 w-6 flex-shrink-0 items-center justify-center rounded-full bg-blue-600 text-sm text-white">
                  2
                </span>
                <div>
                  <strong className="text-white">What do I believe?</strong>
                  <p className="text-sm text-slate-400">
                    Form your own probability estimate. Be honest.
                  </p>
                </div>
              </li>
              <li className="flex gap-3">
                <span className="flex h-6 w-6 flex-shrink-0 items-center justify-center rounded-full bg-blue-600 text-sm text-white">
                  3
                </span>
                <div>
                  <strong className="text-white">How confident am I?</strong>
                  <p className="text-sm text-slate-400">
                    Is your estimate tight (<Formula>{'55\\% \\pm 3\\%'}</Formula>) or wide (
                    <Formula>{'55\\% \\pm 15\\%'}</Formula>)?
                  </p>
                </div>
              </li>
              <li className="flex gap-3">
                <span className="flex h-6 w-6 flex-shrink-0 items-center justify-center rounded-full bg-blue-600 text-sm text-white">
                  4
                </span>
                <div>
                  <strong className="text-white">What&apos;s my edge?</strong>
                  <p className="text-sm text-slate-400">Calculate EV. Is it positive after fees?</p>
                </div>
              </li>
              <li className="flex gap-3">
                <span className="flex h-6 w-6 flex-shrink-0 items-center justify-center rounded-full bg-blue-600 text-sm text-white">
                  5
                </span>
                <div>
                  <strong className="text-white">Is the edge large enough?</strong>
                  <p className="text-sm text-slate-400">
                    Account for uncertainty. Bigger uncertainty = need bigger edge.
                  </p>
                </div>
              </li>
            </ol>
          </div>

          <p className="mt-6 leading-relaxed text-slate-300">
            If you can&apos;t clearly articulate your edge on a trade—why you disagree with the
            market and by how much—you shouldn&apos;t make the trade.
            <strong className="text-white"> No edge, no trade.</strong>
          </p>
        </LessonRow>
      </section>

      {/* Key Takeaways */}
      <section>
        <SectionHeader
          title="Key Takeaways"
          subtitle="What to remember"
        />
        <LessonRow>
          <div className="rounded-lg border border-blue-800/50 bg-gradient-to-r from-blue-900/30 to-purple-900/30 p-6">
            <ul className="space-y-3 text-slate-300">
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>
                  <GlossaryTooltip term={terms.expectedValue}>Expected value</GlossaryTooltip> is
                  the average outcome over many repetitions—the only metric that matters for
                  profitability.
                </span>
              </li>
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>
                  Prices on Kalshi directly equal implied probabilities. 40¢ = 40% implied.
                </span>
              </li>
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>
                  Your <GlossaryTooltip term={terms.edge}>edge</GlossaryTooltip> is the gap between
                  your probability estimate and the market&apos;s implied probability.
                </span>
              </li>
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>A good bet that loses is still a good bet. Judge process, not outcomes.</span>
              </li>
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>The wider your uncertainty, the larger the gap you need before trading.</span>
              </li>
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>
                  <strong className="text-white">No edge, no trade.</strong> If you can&apos;t
                  articulate why you disagree with the market, don&apos;t trade.
                </span>
              </li>
            </ul>
          </div>
        </LessonRow>
      </section>
    </div>
  );
}
