/**
 * Lesson: Risk Management & Bankroll
 *
 * Essential concepts for protecting your capital and trading sustainably.
 * This lesson covers position sizing, bankroll management, and risk control.
 */

import {
  Shield,
  TrendingDown,
  PiggyBank,
  Calculator,
  AlertTriangle,
  Target,
  Percent,
  Scale,
} from 'lucide-react';
import { getGuide, getNextGuide } from '@/data/guides';
import { LessonLayout, LessonRow, LessonHeader, SectionHeader } from '../LessonLayout';
import {
  ObjectiveBlock,
  ConstraintBlock,
  TipBlock,
  InsightBlock,
  WarningBlock,
  ConceptCard,
  ConceptBlock,
  ExampleBox,
  NextStepBlock,
  Term,
} from '../blocks';
import { GlossaryTooltip } from '@/components/atoms/GlossaryTooltip';
import { Formula } from '@/components/atoms/Formula';

// Custom glossary terms for this lesson
const terms = {
  bankroll: {
    term: 'Bankroll',
    definition:
      'The total amount of money you have set aside specifically for trading. Never risk money you cannot afford to lose.',
    category: 'Portfolio',
  },
  positionSize: {
    term: 'Position Size',
    definition:
      'The amount of capital you allocate to a single trade. Proper sizing protects against catastrophic losses.',
    category: 'Risk',
  },
  drawdown: {
    term: 'Drawdown',
    definition:
      'The decline from your peak account value to the lowest point. A 50% drawdown requires a 100% gain to recover.',
    category: 'Risk',
  },
  kellycriterion: {
    term: 'Kelly Criterion',
    definition:
      'A formula for optimal bet sizing based on your edge and odds. Full Kelly is aggressive; most use fractional Kelly.',
    category: 'Risk',
  },
  ruin: {
    term: 'Risk of Ruin',
    definition:
      'The probability of losing your entire bankroll. Even with positive expected value, poor sizing can lead to ruin.',
    category: 'Risk',
  },
  variance: {
    term: 'Variance',
    definition:
      'The natural fluctuation in results even when making correct decisions. High variance means wild swings.',
    category: 'Risk',
  },
};

export function RiskManagementLesson() {
  const guide = getGuide('risk-management')!;
  const nextGuide = getNextGuide('risk-management');

  return (
    <LessonLayout>
      {/* Header */}
      <LessonRow>
        <LessonHeader
          title={guide.title}
          description={guide.description}
          duration="15 min"
          category="Fundamentals"
        />
      </LessonRow>

      {/* Objective */}
      <LessonRow
        aside={
          <ConstraintBlock
            title="What We'll Cover"
            items={[
              'Why risk management matters more than picking winners',
              'How to size your positions properly',
              'The mathematics of ruin and recovery',
              'Practical bankroll management rules',
            ]}
          />
        }
      >
        <ObjectiveBlock>
          By the end of this lesson, you'll understand why position sizing and bankroll management
          are the foundation of sustainable trading. The best strategy in the world is worthless if
          you blow up your account before it can work.
        </ObjectiveBlock>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 1: WHY RISK MANAGEMENT MATTERS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="The Hard Truth">
            Most failed traders had winning strategies. They failed because of poor risk management,
            not poor analysis.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="Why Risk Management Matters"
          subtitle="The most important skill you'll learn"
        />

        <p className="text-muted-foreground leading-relaxed">
          Here's a counterintuitive truth:{' '}
          <strong>risk management is more important than picking winners</strong>. You can have a
          positive edge and still go broke. You can have a mediocre strategy and still profit
          sustainably. The difference is how you manage risk.
        </p>

        <div className="mt-6 rounded-lg border border-rose-500/30 bg-gradient-to-br from-rose-500/10 via-rose-500/5 to-transparent p-5">
          <div className="flex items-start gap-4">
            <div className="flex h-10 w-10 flex-shrink-0 items-center justify-center rounded-full bg-rose-500/20">
              <AlertTriangle className="h-5 w-5 text-rose-400" />
            </div>
            <div>
              <h3 className="text-lg font-semibold text-rose-400">The Ruin Problem</h3>
              <p className="text-muted-foreground mt-2">
                Even with a 55% win rate (positive edge), betting your entire bankroll each time
                gives you a near-certain probability of going broke.{' '}
                <GlossaryTooltip term={terms.variance}>Variance</GlossaryTooltip> will eventually
                produce a losing streak that wipes you out.
              </p>
            </div>
          </div>
        </div>

        <ExampleBox title="The Math of Recovery">
          <div className="space-y-4">
            <p className="text-sm">
              <strong>Losses hurt more than wins help.</strong> Here's why:
            </p>

            <div className="overflow-x-auto">
              <table className="w-full text-sm">
                <thead>
                  <tr className="border-b">
                    <th className="py-2 text-left font-medium">Loss</th>
                    <th className="py-2 text-left font-medium">Gain Needed to Recover</th>
                    <th className="py-2 text-left font-medium">Difficulty</th>
                  </tr>
                </thead>
                <tbody className="divide-y">
                  <tr>
                    <td className="py-2 text-rose-400">-10%</td>
                    <td className="py-2">+11%</td>
                    <td className="py-2 text-emerald-400">Easy</td>
                  </tr>
                  <tr>
                    <td className="py-2 text-rose-400">-25%</td>
                    <td className="py-2">+33%</td>
                    <td className="py-2 text-amber-400">Moderate</td>
                  </tr>
                  <tr>
                    <td className="py-2 text-rose-400">-50%</td>
                    <td className="py-2">+100%</td>
                    <td className="py-2 text-rose-400">Hard</td>
                  </tr>
                  <tr>
                    <td className="py-2 text-rose-400">-75%</td>
                    <td className="py-2">+300%</td>
                    <td className="py-2 text-rose-400">Very Hard</td>
                  </tr>
                  <tr>
                    <td className="py-2 text-rose-400">-90%</td>
                    <td className="py-2">+900%</td>
                    <td className="py-2 text-rose-400">Nearly Impossible</td>
                  </tr>
                </tbody>
              </table>
            </div>

            <p className="text-muted-foreground border-t pt-3 text-xs">
              This asymmetry is why preventing large{' '}
              <GlossaryTooltip term={terms.drawdown}>drawdowns</GlossaryTooltip> is more important
              than maximizing individual gains.
            </p>
          </div>
        </ExampleBox>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 2: BANKROLL BASICS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="Separate Your Money">
            Your trading bankroll should be separate from your living expenses. Only trade with
            money you can afford to lose entirely.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="Bankroll Basics"
          subtitle="Setting up your trading capital"
        />

        <p className="text-muted-foreground leading-relaxed">
          Your <GlossaryTooltip term={terms.bankroll}>bankroll</GlossaryTooltip> is the dedicated
          capital you use for trading. It should be money you can afford to lose without affecting
          your life - not rent money, not emergency savings, not borrowed funds.
        </p>

        <div className="mt-6 space-y-4">
          <Term term="Bankroll">
            A dedicated pool of capital for trading. Treat it as completely separate from your other
            finances. If you lose it all, your life continues unchanged - that's the test.
          </Term>

          <Term term="Working Capital">
            The portion of your bankroll actively deployed in positions. You don't need to have your
            entire bankroll at risk at all times.
          </Term>
        </div>

        <div className="bg-card mt-6 rounded-lg border p-5">
          <h4 className="mb-4 font-semibold">Bankroll Rules</h4>
          <div className="space-y-3">
            <div className="flex items-start gap-3">
              <div className="flex h-6 w-6 flex-shrink-0 items-center justify-center rounded-full bg-emerald-500/20">
                <span className="text-xs font-bold text-emerald-400">1</span>
              </div>
              <div>
                <p className="text-sm font-medium">Never add to a losing bankroll</p>
                <p className="text-muted-foreground text-xs">
                  If you've lost a significant portion, stop trading and reassess. Don't chase
                  losses.
                </p>
              </div>
            </div>
            <div className="flex items-start gap-3">
              <div className="flex h-6 w-6 flex-shrink-0 items-center justify-center rounded-full bg-emerald-500/20">
                <span className="text-xs font-bold text-emerald-400">2</span>
              </div>
              <div>
                <p className="text-sm font-medium">Scale position sizes with bankroll</p>
                <p className="text-muted-foreground text-xs">
                  As your bankroll grows, you can increase sizes. As it shrinks, reduce them
                  proportionally.
                </p>
              </div>
            </div>
            <div className="flex items-start gap-3">
              <div className="flex h-6 w-6 flex-shrink-0 items-center justify-center rounded-full bg-emerald-500/20">
                <span className="text-xs font-bold text-emerald-400">3</span>
              </div>
              <div>
                <p className="text-sm font-medium">Have a stop-loss level for the session/week</p>
                <p className="text-muted-foreground text-xs">
                  Define maximum daily/weekly losses. When hit, stop trading until the next period.
                </p>
              </div>
            </div>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 3: POSITION SIZING */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="Start Small">
            When learning, use 0.5-1% position sizes. You can always increase later once you've
            proven your edge over hundreds of trades.
          </TipBlock>
        }
      >
        <SectionHeader
          title="Position Sizing"
          subtitle="How much to risk on each trade"
        />

        <p className="text-muted-foreground leading-relaxed">
          <GlossaryTooltip term={terms.positionSize}>Position sizing</GlossaryTooltip> determines
          how much of your bankroll you risk on each trade. It's the single most important decision
          you make - more important than which side to bet.
        </p>

        <div className="mt-6 grid gap-4 sm:grid-cols-3">
          <ConceptCard
            title="Conservative"
            description="1-2% per trade. Slow growth but protected from ruin. Best for beginners."
            icon={<Shield className="h-4 w-4" />}
          />
          <ConceptCard
            title="Moderate"
            description="2-5% per trade. Balanced growth and risk. For proven strategies."
            icon={<Scale className="h-4 w-4" />}
          />
          <ConceptCard
            title="Aggressive"
            description="5-10% per trade. Fast growth but high ruin risk. Only for large edges."
            icon={<Target className="h-4 w-4" />}
          />
        </div>

        <ExampleBox title="Position Sizing in Practice">
          <div className="space-y-4">
            <p className="text-sm">
              <strong>Scenario:</strong> You have a $1,000 bankroll and use 2% position sizing.
            </p>

            <div className="bg-muted/50 rounded-lg p-4">
              <div className="grid gap-4 sm:grid-cols-2">
                <div>
                  <div className="mb-2 text-sm font-medium">Maximum Risk Per Trade</div>
                  <div className="text-primary text-2xl font-bold">$20</div>
                  <p className="text-muted-foreground mt-1 text-xs">
                    2% × $1,000 = $20 maximum loss per position
                  </p>
                </div>
                <div>
                  <div className="mb-2 text-sm font-medium">Contract Calculation</div>
                  <div className="font-mono text-lg">
                    If YES costs 60¢, max risk is 60¢ per contract
                  </div>
                  <p className="text-muted-foreground mt-1 text-xs">
                    $20 ÷ $0.60 = 33 contracts maximum
                  </p>
                </div>
              </div>
            </div>

            <p className="text-muted-foreground border-t pt-3 text-xs">
              At 2% sizing, you could lose 50 trades in a row before losing your bankroll. This
              gives your edge time to play out.
            </p>
          </div>
        </ExampleBox>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 4: THE KELLY CRITERION */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <ConceptBlock title="Kelly Formula">
            <Formula block>{'f^{*} = \\frac{bp - q}{b}'}</Formula>
            Where <Formula>b</Formula> = odds, <Formula>p</Formula> = win probability,{' '}
            <Formula>q</Formula> = loss probability <Formula>{'(1-p)'}</Formula>
          </ConceptBlock>
        }
      >
        <SectionHeader
          title="The Kelly Criterion"
          subtitle="Optimal sizing based on your edge"
        />

        <p className="text-muted-foreground leading-relaxed">
          The <GlossaryTooltip term={terms.kellycriterion}>Kelly Criterion</GlossaryTooltip> is a
          formula that calculates the optimal bet size to maximize long-term growth. It was
          developed by John Kelly at Bell Labs in 1956 for information theory - but traders quickly
          adopted it.
        </p>

        <div className="mt-6 space-y-4">
          <Term term="Full Kelly">
            Betting the exact Kelly percentage. Mathematically optimal but very aggressive. Results
            in large swings and psychological discomfort.
          </Term>

          <Term term="Fractional Kelly">
            Betting a fraction (usually 1/4 to 1/2) of Kelly. Sacrifices some growth for much
            smoother ride and lower risk of ruin. Most professionals use fractional Kelly.
          </Term>
        </div>

        <ExampleBox title="Kelly Calculation">
          <div className="space-y-4">
            <p className="text-sm">
              <strong>Scenario:</strong> You're buying YES at 40¢ and believe the true probability
              is 50%.
            </p>

            <div className="bg-muted/50 space-y-2 rounded-lg p-4 text-sm">
              <div>
                Win probability <Formula>{'(p)'}</Formula> = 50% = 0.50
              </div>
              <div>
                Loss probability <Formula>{'(q)'}</Formula> = 50% = 0.50
              </div>
              <div>
                Odds <Formula>{'(b)'}</Formula> = Win/Cost = $0.60 / $0.40 = 1.5
              </div>
              <div className="mt-2 border-t pt-2">
                <Formula>{'Kelly = \\frac{bp - q}{b}'}</Formula>
              </div>
              <div>
                <Formula>{'Kelly = \\frac{1.5 \\times 0.50 - 0.50}{1.5}'}</Formula>
              </div>
              <div>
                <Formula>{'Kelly = \\frac{0.75 - 0.50}{1.5}'}</Formula> ={' '}
                <span className="text-primary font-bold">16.7%</span>
              </div>
            </div>

            <p className="text-muted-foreground text-sm">
              Full Kelly says bet 16.7% of bankroll. Most traders would use 1/4 Kelly (4.2%) or 1/2
              Kelly (8.3%) for a smoother experience.
            </p>
          </div>
        </ExampleBox>

        <div className="mt-6 rounded-lg border border-amber-500/30 bg-amber-500/10 p-4">
          <h4 className="mb-2 font-semibold text-amber-400">Important Caveat</h4>
          <p className="text-muted-foreground text-sm">
            Kelly assumes you know the true probabilities - which you don't. Your edge estimate is
            just that: an estimate. This is why fractional Kelly is strongly recommended.
            Overestimating your edge and using full Kelly can be devastating.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 5: MANAGING DRAWDOWNS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="Drawdowns Are Normal">
            Even with a 60% win rate, you'll experience 5+ loss streaks regularly. Plan for them.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="Managing Drawdowns"
          subtitle="Surviving the inevitable losing streaks"
        />

        <p className="text-muted-foreground leading-relaxed">
          <GlossaryTooltip term={terms.drawdown}>Drawdowns</GlossaryTooltip> are inevitable. Even
          the best traders experience them. The question isn't whether you'll have losing streaks -
          it's whether you'll survive them with enough capital to recover.
        </p>

        <div className="bg-card mt-6 rounded-lg border p-5">
          <h4 className="mb-4 font-semibold">Drawdown Management Rules</h4>
          <div className="space-y-4">
            <div className="flex items-start gap-3">
              <TrendingDown className="mt-0.5 h-5 w-5 flex-shrink-0 text-rose-400" />
              <div>
                <p className="text-sm font-medium">Set Maximum Drawdown Limits</p>
                <p className="text-muted-foreground text-xs">
                  Define the point at which you stop trading. Common levels: 20% daily, 30% weekly,
                  50% total. Hit the limit → step away.
                </p>
              </div>
            </div>
            <div className="flex items-start gap-3">
              <Calculator className="mt-0.5 h-5 w-5 flex-shrink-0 text-amber-400" />
              <div>
                <p className="text-sm font-medium">Reduce Size During Drawdowns</p>
                <p className="text-muted-foreground text-xs">
                  As your bankroll shrinks, reduce position sizes proportionally. This slows the
                  bleeding and gives you more attempts to recover.
                </p>
              </div>
            </div>
            <div className="flex items-start gap-3">
              <Shield className="mt-0.5 h-5 w-5 flex-shrink-0 text-emerald-400" />
              <div>
                <p className="text-sm font-medium">Don't Increase Size to "Win It Back"</p>
                <p className="text-muted-foreground text-xs">
                  Chasing losses is the fastest path to ruin. If you're down, the answer is smaller
                  sizes, not bigger ones. Discipline is everything.
                </p>
              </div>
            </div>
          </div>
        </div>

        <ExampleBox title="Surviving a Losing Streak">
          <div className="space-y-4">
            <p className="text-sm">
              <strong>Scenario:</strong> $1,000 bankroll, 2% sizing, you lose 10 trades in a row.
            </p>

            <div className="overflow-x-auto">
              <table className="w-full font-mono text-sm">
                <thead>
                  <tr className="border-b text-xs">
                    <th className="py-2 text-left">Trade</th>
                    <th className="py-2 text-right">Bankroll</th>
                    <th className="py-2 text-right">Risk (2%)</th>
                    <th className="py-2 text-right">After Loss</th>
                  </tr>
                </thead>
                <tbody className="divide-y text-xs">
                  <tr>
                    <td>1</td>
                    <td className="text-right">$1,000</td>
                    <td className="text-right text-rose-400">-$20</td>
                    <td className="text-right">$980</td>
                  </tr>
                  <tr>
                    <td>2</td>
                    <td className="text-right">$980</td>
                    <td className="text-right text-rose-400">-$19.60</td>
                    <td className="text-right">$960</td>
                  </tr>
                  <tr>
                    <td>3</td>
                    <td className="text-right">$960</td>
                    <td className="text-right text-rose-400">-$19.20</td>
                    <td className="text-right">$941</td>
                  </tr>
                  <tr>
                    <td>...</td>
                    <td></td>
                    <td></td>
                    <td></td>
                  </tr>
                  <tr>
                    <td>10</td>
                    <td className="text-right">$833</td>
                    <td className="text-right text-rose-400">-$16.66</td>
                    <td className="text-right">$817</td>
                  </tr>
                </tbody>
              </table>
            </div>

            <p className="text-muted-foreground border-t pt-3 text-xs">
              After 10 consecutive losses, you're down ~18% - painful but recoverable. With 10%
              sizing, you'd be down ~65% and facing ruin.
            </p>
          </div>
        </ExampleBox>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 6: PRACTICAL RULES */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="Write These Down">
            Create your own risk rules and keep them visible when trading. It's easy to forget
            discipline in the heat of the moment.
          </TipBlock>
        }
      >
        <SectionHeader
          title="Practical Risk Rules"
          subtitle="Guidelines for sustainable trading"
        />

        <div className="grid gap-4 sm:grid-cols-2">
          <div className="bg-card rounded-lg border p-4">
            <div className="mb-3 flex items-center gap-2">
              <div className="flex h-8 w-8 items-center justify-center rounded-full bg-emerald-500/20">
                <Percent className="h-4 w-4 text-emerald-400" />
              </div>
              <h4 className="font-semibold">Position Limits</h4>
            </div>
            <ul className="text-muted-foreground space-y-2 text-sm">
              <li>• Max 1-2% per position (beginners)</li>
              <li>• Max 5% per position (experienced)</li>
              <li>• Max 20% total exposure at once</li>
              <li>• Max 10% in any single event</li>
            </ul>
          </div>

          <div className="bg-card rounded-lg border p-4">
            <div className="mb-3 flex items-center gap-2">
              <div className="flex h-8 w-8 items-center justify-center rounded-full bg-amber-500/20">
                <AlertTriangle className="h-4 w-4 text-amber-400" />
              </div>
              <h4 className="font-semibold">Stop-Loss Levels</h4>
            </div>
            <ul className="text-muted-foreground space-y-2 text-sm">
              <li>• Daily loss limit: 5-10% of bankroll</li>
              <li>• Weekly loss limit: 15-20% of bankroll</li>
              <li>• Monthly loss limit: 30% of bankroll</li>
              <li>• Hit a limit → stop trading that period</li>
            </ul>
          </div>

          <div className="bg-card rounded-lg border p-4">
            <div className="mb-3 flex items-center gap-2">
              <div className="flex h-8 w-8 items-center justify-center rounded-full bg-violet-500/20">
                <PiggyBank className="h-4 w-4 text-violet-400" />
              </div>
              <h4 className="font-semibold">Bankroll Rules</h4>
            </div>
            <ul className="text-muted-foreground space-y-2 text-sm">
              <li>• Only trade money you can lose</li>
              <li>• Never add to a losing bankroll</li>
              <li>• Take profits periodically</li>
              <li>• Scale sizes with bankroll changes</li>
            </ul>
          </div>

          <div className="bg-card rounded-lg border p-4">
            <div className="mb-3 flex items-center gap-2">
              <div className="flex h-8 w-8 items-center justify-center rounded-full bg-rose-500/20">
                <Shield className="h-4 w-4 text-rose-400" />
              </div>
              <h4 className="font-semibold">Psychological Rules</h4>
            </div>
            <ul className="text-muted-foreground space-y-2 text-sm">
              <li>• Never chase losses</li>
              <li>• Don't trade when emotional</li>
              <li>• Take breaks after big wins/losses</li>
              <li>• Review decisions, not outcomes</li>
            </ul>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 7: KEY TAKEAWAYS */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="Key Takeaways"
          subtitle="Remember these principles"
        />

        <div className="space-y-4">
          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">1</span>
            </div>
            <div>
              <h4 className="font-medium">Risk management beats picking winners</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                You can have a positive edge and still go broke with poor sizing. Position sizing
                determines survival.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">2</span>
            </div>
            <div>
              <h4 className="font-medium">Losses are asymmetric</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                A 50% loss requires 100% gain to recover. Preventing large drawdowns is more
                important than maximizing gains.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">3</span>
            </div>
            <div>
              <h4 className="font-medium">Use 1-2% position sizing as a beginner</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                Conservative sizing gives your edge time to play out across many trades. You can
                increase later once proven.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="bg-primary/20 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
              <span className="text-primary font-bold">4</span>
            </div>
            <div>
              <h4 className="font-medium">Have predefined stop-loss rules</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                Set daily/weekly loss limits before you start trading. When hit, stop. No
                exceptions.
              </p>
            </div>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 8: NEXT STEPS */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="What's Next?"
          subtitle="Continue building your foundation"
        />

        <p className="text-muted-foreground mb-6 leading-relaxed">
          Now that you understand how to protect your capital, the next step is learning how to read
          order flow - understanding what the market is telling you through the order book.
        </p>

        {nextGuide && (
          <NextStepBlock
            href={`/${nextGuide.slug}`}
            title={nextGuide.title}
            description="Learn to read the order book and understand what order flow reveals about market sentiment and upcoming price movements."
            buttonText="Continue to Next Lesson"
          />
        )}
      </LessonRow>
    </LessonLayout>
  );
}
