/**
 * Lesson: Sports Betting 101
 *
 * Learn the fundamentals: moneylines, spreads, and totals.
 * Expanded version with etymology, history, and deep explanations.
 */

import { Trophy, ArrowUpDown, Hash, History, Calculator, AlertTriangle } from 'lucide-react';
import { getGuide, getNextGuide } from '@/data/guides';
import { LessonLayout, LessonRow, LessonHeader, SectionHeader } from '../LessonLayout';
import {
  ObjectiveBlock,
  ConstraintBlock,
  TipBlock,
  InsightBlock,
  WarningBlock,
  ConceptCard,
  NextStepBlock,
  ExampleBox,
  Term,
} from '../blocks';
import { GlossaryTooltip } from '@/components/atoms/GlossaryTooltip';
import { BetTypeVisualizer } from '../widgets/BetTypeVisualizer';

// Custom glossary terms for this lesson
const terms = {
  moneyline: {
    term: 'Moneyline',
    definition:
      'A bet on which team will win, regardless of score. The simplest bet type - just pick the winner.',
    category: 'Bet Types',
  },
  pointSpread: {
    term: 'Point Spread',
    definition:
      'A handicap given to the underdog. The favorite must win by more than the spread to "cover."',
    category: 'Bet Types',
  },
  total: {
    term: 'Total (Over/Under)',
    definition:
      'A bet on whether the combined score of both teams will be over or under a set number.',
    category: 'Bet Types',
  },
  favorite: {
    term: 'Favorite',
    definition:
      'The team expected to win. They give points on a spread or cost more on a moneyline.',
    category: 'Bet Types',
  },
  underdog: {
    term: 'Underdog',
    definition:
      'The team expected to lose. They receive points on a spread and pay more on a moneyline.',
    category: 'Bet Types',
  },
  cover: {
    term: 'Cover',
    definition:
      'When a team beats the point spread. The favorite must win by more than the spread; the underdog must lose by less.',
    category: 'Bet Types',
  },
  juice: {
    term: 'Juice (Vig)',
    definition:
      'The fee built into odds that gives the sportsbook its edge. Standard is -110 (bet $110 to win $100).',
    category: 'Fees',
  },
  hook: {
    term: 'Hook',
    definition: 'Half a point (.5) in a point spread. "Lakers -5.5" has a hook to eliminate ties.',
    category: 'Bet Types',
  },
};

export function SportsBetting101Lesson() {
  const guide = getGuide('sports-betting-101')!;
  const nextGuide = getNextGuide('sports-betting-101');

  return (
    <LessonLayout>
      {/* Header */}
      <LessonRow>
        <LessonHeader
          title={guide.title}
          description={guide.description}
          duration="15 min"
          category="Sports Betting"
        />
      </LessonRow>

      {/* Objective */}
      <LessonRow
        aside={
          <ConstraintBlock
            title="What We'll Cover"
            items={[
              'What moneylines are and why they exist',
              'How point spreads level the playing field',
              'When to use totals (over/under)',
              'The history and terminology of each bet type',
            ]}
          />
        }
      >
        <ObjectiveBlock>
          By the end of this lesson, you'll understand the three fundamental bet types in sports
          betting: moneylines, point spreads, and totals. More importantly, you'll understand
          <em>why</em> each exists and when to use them.
        </ObjectiveBlock>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 1: INTRODUCTION */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="Foundation First">
            These three bet types cover approximately 95% of all sports betting activity. Master
            these before exploring exotic bets like parlays or props.
          </TipBlock>
        }
      >
        <SectionHeader
          title="The Three Pillars of Sports Betting"
          subtitle="Everything builds on these fundamentals"
        />

        <p className="text-muted-foreground leading-relaxed">
          Before diving into the mechanics, let's understand <em>why</em> these three bet types
          exist. Each was invented to solve a specific problem in sports betting.
        </p>

        <div className="mt-6 grid gap-4 sm:grid-cols-3">
          <ConceptCard
            title="Moneyline"
            description="Who wins? The simplest question in sports."
            icon={<Trophy className="h-4 w-4" />}
          />
          <ConceptCard
            title="Point Spread"
            description="Created to make lopsided games interesting to bet."
            icon={<ArrowUpDown className="h-4 w-4" />}
          />
          <ConceptCard
            title="Total (O/U)"
            description="Bet on the game's nature without picking a winner."
            icon={<Hash className="h-4 w-4" />}
          />
        </div>

        <div className="bg-muted/50 mt-6 rounded-lg border p-4">
          <p className="text-muted-foreground text-sm">
            <strong>Historical note:</strong> These bet types evolved in different eras. Moneylines
            are the oldest (dating back centuries). Point spreads were invented in the 1940s by a
            math teacher. Totals emerged later as bettors wanted action without team preferences.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 2: MONEYLINES - DEEP DIVE */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="Why 'Moneyline'?">
            The term comes from the "line" of odds bookmakers wrote on chalkboards, showing how much
            "money" you needed to risk. American odds (-150, +130) evolved from this tradition.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="1. Moneylines"
          subtitle="The simplest bet: pick the winner"
        />

        <div className="mb-6 flex items-start gap-3">
          <div className="bg-primary/10 flex h-10 w-10 flex-shrink-0 items-center justify-center rounded-full">
            <History className="text-primary h-5 w-5" />
          </div>
          <div>
            <h4 className="font-semibold">Etymology & History</h4>
            <p className="text-muted-foreground mt-1 text-sm">
              The term "<GlossaryTooltip term={terms.moneyline}>moneyline</GlossaryTooltip>" is an
              American invention from the early 20th century. "Line" referred to the literal line of
              odds written on a bookmaker's board. "Money" distinguished it from point spread
              betting. In the UK and Europe, this same bet is simply called "to win" or "match
              result."
            </p>
          </div>
        </div>

        <p className="text-muted-foreground leading-relaxed">
          A <GlossaryTooltip term={terms.moneyline}>moneyline bet</GlossaryTooltip> is the most
          fundamental bet in sports: you pick which team wins the game. Nothing else matters — not
          the score, not the margin, not how the game unfolds. Just: who wins?
        </p>

        <div className="mt-6 space-y-4">
          <Term term="Favorite">
            The team expected to win. On Kalshi, the{' '}
            <GlossaryTooltip term={terms.favorite}>favorite</GlossaryTooltip> costs more (higher
            price) because they're more likely to win. At 65¢, you risk 65¢ to profit 35¢.
          </Term>
          <Term term="Underdog">
            The team expected to lose. The{' '}
            <GlossaryTooltip term={terms.underdog}>underdog</GlossaryTooltip> costs less (lower
            price) but pays more if they win. At 35¢, you risk 35¢ to profit 65¢.
          </Term>
        </div>

        <ExampleBox title="Lakers (65¢) vs Celtics (35¢)">
          <div className="space-y-4">
            <p className="text-muted-foreground text-sm">
              The prices tell you the market's probability estimate: 65% Lakers win, 35% Celtics
              win.
            </p>

            <div className="grid gap-4 sm:grid-cols-2">
              <div className="bg-muted/50 rounded p-3">
                <div className="mb-2 font-medium text-amber-400">
                  Betting on the Favorite (Lakers)
                </div>
                <ul className="text-muted-foreground space-y-1 text-sm">
                  <li>• Cost: 65¢ per contract</li>
                  <li>
                    • Lakers win → <span className="text-emerald-400">+35¢ profit</span>
                  </li>
                  <li>
                    • Lakers lose → <span className="text-rose-400">-65¢ loss</span>
                  </li>
                </ul>
              </div>
              <div className="bg-muted/50 rounded p-3">
                <div className="mb-2 font-medium text-violet-400">
                  Betting on the Underdog (Celtics)
                </div>
                <ul className="text-muted-foreground space-y-1 text-sm">
                  <li>• Cost: 35¢ per contract</li>
                  <li>
                    • Celtics win → <span className="text-emerald-400">+65¢ profit</span>
                  </li>
                  <li>
                    • Celtics lose → <span className="text-rose-400">-35¢ loss</span>
                  </li>
                </ul>
              </div>
            </div>

            <p className="text-muted-foreground border-t pt-3 text-xs">
              Notice: Favorites give you smaller profits for larger risk. Underdogs give larger
              profits for smaller risk — but they lose more often. The prices reflect this tradeoff.
            </p>
          </div>
        </ExampleBox>

        <BetTypeVisualizer type="moneyline" />

        <div className="bg-card mt-6 rounded-lg border p-4">
          <h4 className="mb-3 font-semibold">When to Use Moneylines</h4>
          <ul className="text-muted-foreground space-y-2 text-sm">
            <li className="flex items-start gap-2">
              <span className="mt-1 text-emerald-400">✓</span>
              <span>You're confident a team will win, regardless of margin</span>
            </li>
            <li className="flex items-start gap-2">
              <span className="mt-1 text-emerald-400">✓</span>
              <span>The underdog has upset potential (high reward)</span>
            </li>
            <li className="flex items-start gap-2">
              <span className="mt-1 text-emerald-400">✓</span>
              <span>You don't want to worry about point spreads</span>
            </li>
            <li className="flex items-start gap-2">
              <span className="mt-1 text-rose-400">✗</span>
              <span>Avoid heavy favorites where you risk a lot to win little</span>
            </li>
          </ul>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 3: POINT SPREADS - DEEP DIVE */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="The Spread's Inventor">
            Charles K. McNeil, a Chicago math teacher turned bookmaker, invented the point spread in
            the 1940s. His insight: handicapping made every game attractive, not just close ones.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="2. Point Spreads"
          subtitle="Handicapping to level the playing field"
        />

        <div className="mb-6 flex items-start gap-3">
          <div className="bg-primary/10 flex h-10 w-10 flex-shrink-0 items-center justify-center rounded-full">
            <History className="text-primary h-5 w-5" />
          </div>
          <div>
            <h4 className="font-semibold">Etymology & History</h4>
            <p className="text-muted-foreground mt-1 text-sm">
              Before point spreads, betting on obvious winners was boring — who wants to risk $5 to
              win $1? Charles K. McNeil realized that <em>handicapping</em> games would make every
              matchup interesting. The "spread" (gap) the favorite must overcome became the
              innovation that transformed American sports betting in the 1940s-50s.
            </p>
          </div>
        </div>

        <p className="text-muted-foreground leading-relaxed">
          A <GlossaryTooltip term={terms.pointSpread}>point spread</GlossaryTooltip> gives the
          underdog a head start. The favorite doesn't just need to win — they need to win by more
          than the spread. This makes lopsided matchups interesting to bet.
        </p>

        <div className="mt-6 space-y-4">
          <Term term="The Spread">
            The number of points handicapping the favorite. "Lakers -5.5" means Lakers must win by
            6+ points. "Celtics +5.5" means Celtics can lose by up to 5 points and still "cover."
          </Term>
          <Term term="Cover">
            When a team beats the spread.{' '}
            <GlossaryTooltip term={terms.cover}>Covering</GlossaryTooltip> means winning the bet.
            "Lakers covered" means they won by more than the spread.
          </Term>
          <Term term="Hook">
            The half-point (.5) in a spread like -5.5. The{' '}
            <GlossaryTooltip term={terms.hook}>hook</GlossaryTooltip> eliminates ties (pushes).
            Without it, a 5-point Lakers win would tie, and everyone gets their money back.
          </Term>
        </div>

        <ExampleBox title="Lakers -5.5 Points">
          <div className="space-y-4">
            <p className="text-sm">
              <strong>The question:</strong> "Will the Lakers win by 5.5 or more points?"
            </p>

            <div className="bg-muted/50 rounded-lg p-4">
              <h5 className="mb-3 text-sm font-semibold">How to Think About Spreads</h5>
              <p className="text-muted-foreground mb-3 text-sm">
                Imagine the game starts with the underdog already ahead by the spread amount:
              </p>
              <div className="bg-background rounded p-3 text-center font-mono">
                <span className="text-violet-400">Lakers 0</span>
                <span className="mx-4">vs</span>
                <span className="text-amber-400">Celtics 5.5</span>
              </div>
              <p className="text-muted-foreground mt-3 text-center text-xs">
                Lakers must outscore this "head start" to cover the spread
              </p>
            </div>

            <div className="grid gap-3 sm:grid-cols-2">
              <div className="rounded border border-emerald-500/30 bg-emerald-500/5 p-3">
                <div className="mb-2 font-medium text-emerald-400">YES Wins (Lakers Cover)</div>
                <ul className="text-muted-foreground space-y-1 text-sm">
                  <li>• Lakers win by 6 points</li>
                  <li>• Lakers win by 15 points</li>
                  <li>• Lakers win by 50 points</li>
                </ul>
              </div>
              <div className="rounded border border-rose-500/30 bg-rose-500/5 p-3">
                <div className="mb-2 font-medium text-rose-400">NO Wins (Lakers Don't Cover)</div>
                <ul className="text-muted-foreground space-y-1 text-sm">
                  <li>• Lakers win by 5 points</li>
                  <li>• Lakers win by 2 points</li>
                  <li>• Celtics win (any score)</li>
                </ul>
              </div>
            </div>

            <p className="text-muted-foreground border-t pt-3 text-xs">
              The .5 (hook) ensures one side always wins. A spread of -5 would mean a 5-point win is
              a "push" (tie) with money returned.
            </p>
          </div>
        </ExampleBox>

        <BetTypeVisualizer type="spread" />

        <div className="bg-card mt-6 rounded-lg border p-4">
          <h4 className="mb-3 font-semibold">When to Use Point Spreads</h4>
          <ul className="text-muted-foreground space-y-2 text-sm">
            <li className="flex items-start gap-2">
              <span className="mt-1 text-emerald-400">✓</span>
              <span>You think the favorite will win big (or the underdog will keep it close)</span>
            </li>
            <li className="flex items-start gap-2">
              <span className="mt-1 text-emerald-400">✓</span>
              <span>The moneyline is too expensive for the favorite</span>
            </li>
            <li className="flex items-start gap-2">
              <span className="mt-1 text-emerald-400">✓</span>
              <span>You have an opinion on margin, not just winner</span>
            </li>
            <li className="flex items-start gap-2">
              <span className="mt-1 text-rose-400">✗</span>
              <span>
                Avoid spreads in unpredictable games (injuries, rest, motivational factors)
              </span>
            </li>
          </ul>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 4: TOTALS - DEEP DIVE */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="Totals Traps">
            Late-game scenarios can swing totals: garbage time scoring, intentional fouls, overtime.
            A blowout can lead to bench players padding stats in the final minutes.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="3. Totals (Over/Under)"
          subtitle="Betting on the game's pace, not the winner"
        />

        <div className="mb-6 flex items-start gap-3">
          <div className="bg-primary/10 flex h-10 w-10 flex-shrink-0 items-center justify-center rounded-full">
            <History className="text-primary h-5 w-5" />
          </div>
          <div>
            <h4 className="font-semibold">Etymology & History</h4>
            <p className="text-muted-foreground mt-1 text-sm">
              <GlossaryTooltip term={terms.total}>Totals betting</GlossaryTooltip> emerged as
              bettors wanted action without caring who wins. "Over/under" describes the two possible
              outcomes. This bet type became popular for games with obvious winners — you can bet on
              the scoring environment without picking sides.
            </p>
          </div>
        </div>

        <p className="text-muted-foreground leading-relaxed">
          A <GlossaryTooltip term={terms.total}>total</GlossaryTooltip> bet ignores who wins. You're
          betting on whether the combined score of both teams will be over or under a set number.
          This is about the game's <em>nature</em>, not its winner.
        </p>

        <div className="mt-6 space-y-4">
          <Term term="The Total (O/U)">
            The line set by oddsmakers representing their expected combined score. A total of 227.5
            means they expect roughly 228 total points scored.
          </Term>
          <Term term="Over">
            Betting the combined score will exceed the total. For O/U 227.5, over wins if the final
            combined score is 228 or higher.
          </Term>
          <Term term="Under">
            Betting the combined score will fall short of the total. For O/U 227.5, under wins if
            the final combined score is 227 or lower.
          </Term>
        </div>

        <ExampleBox title="Lakers vs Celtics: Total 227.5">
          <div className="space-y-4">
            <p className="text-sm">
              <strong>The question:</strong> "Will the combined score be over 227.5 points?"
            </p>

            <div className="bg-muted/50 rounded-lg p-4">
              <h5 className="mb-3 text-sm font-semibold">What Affects Totals</h5>
              <div className="grid gap-3 text-sm sm:grid-cols-2">
                <div>
                  <div className="mb-1 font-medium text-emerald-400">Pushes Total Higher</div>
                  <ul className="text-muted-foreground space-y-1">
                    <li>• Fast pace teams</li>
                    <li>• Poor defense</li>
                    <li>• Overtime potential</li>
                    <li>• All-star game or exhibition</li>
                  </ul>
                </div>
                <div>
                  <div className="mb-1 font-medium text-rose-400">Pushes Total Lower</div>
                  <ul className="text-muted-foreground space-y-1">
                    <li>• Slow pace teams</li>
                    <li>• Elite defense</li>
                    <li>• Playoff intensity</li>
                    <li>• Resting starters</li>
                  </ul>
                </div>
              </div>
            </div>

            <div className="grid gap-3 sm:grid-cols-2">
              <div className="rounded border border-emerald-500/30 bg-emerald-500/5 p-3">
                <div className="mb-2 font-medium text-emerald-400">OVER Wins</div>
                <ul className="text-muted-foreground space-y-1 text-sm">
                  <li>• Lakers 118, Celtics 112 = 230 ✓</li>
                  <li>• Lakers 105, Celtics 128 = 233 ✓</li>
                  <li>• Lakers 114, Celtics 114 (OT) = 228+ ✓</li>
                </ul>
              </div>
              <div className="rounded border border-rose-500/30 bg-rose-500/5 p-3">
                <div className="mb-2 font-medium text-rose-400">UNDER Wins</div>
                <ul className="text-muted-foreground space-y-1 text-sm">
                  <li>• Lakers 108, Celtics 115 = 223 ✓</li>
                  <li>• Lakers 102, Celtics 98 = 200 ✓</li>
                  <li>• Lakers 120, Celtics 107 = 227 ✓</li>
                </ul>
              </div>
            </div>

            <p className="text-muted-foreground border-t pt-3 text-xs">
              Note: The .5 (hook) again ensures one side wins. With 227.5, exactly 228 total is
              impossible, so there's no push.
            </p>
          </div>
        </ExampleBox>

        <BetTypeVisualizer type="total" />

        <div className="bg-card mt-6 rounded-lg border p-4">
          <h4 className="mb-3 font-semibold">When to Use Totals</h4>
          <ul className="text-muted-foreground space-y-2 text-sm">
            <li className="flex items-start gap-2">
              <span className="mt-1 text-emerald-400">✓</span>
              <span>You have no preference for who wins</span>
            </li>
            <li className="flex items-start gap-2">
              <span className="mt-1 text-emerald-400">✓</span>
              <span>You have strong views on pace or defensive matchups</span>
            </li>
            <li className="flex items-start gap-2">
              <span className="mt-1 text-emerald-400">✓</span>
              <span>Weather conditions suggest high/low scoring (outdoor sports)</span>
            </li>
            <li className="flex items-start gap-2">
              <span className="mt-1 text-rose-400">✗</span>
              <span>Avoid when key players' statuses are uncertain (impacts pace)</span>
            </li>
          </ul>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 5: COMPARISON TABLE */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="Quick Comparison"
          subtitle="When to use each bet type"
        />

        <div className="overflow-x-auto">
          <table className="w-full text-sm">
            <thead>
              <tr className="border-b">
                <th className="w-1/5 py-3 text-left font-medium"></th>
                <th className="py-3 text-left font-medium">Moneyline</th>
                <th className="py-3 text-left font-medium">Point Spread</th>
                <th className="py-3 text-left font-medium">Total</th>
              </tr>
            </thead>
            <tbody className="divide-y">
              <tr>
                <td className="text-muted-foreground py-3 font-medium">Question</td>
                <td className="py-3">Who wins?</td>
                <td className="py-3">By how much?</td>
                <td className="py-3">High or low scoring?</td>
              </tr>
              <tr>
                <td className="text-muted-foreground py-3 font-medium">Invented</td>
                <td className="py-3">Ancient (centuries)</td>
                <td className="py-3">1940s (McNeil)</td>
                <td className="py-3">Mid-20th century</td>
              </tr>
              <tr>
                <td className="text-muted-foreground py-3 font-medium">Best for</td>
                <td className="py-3">Confidence in winner</td>
                <td className="py-3">View on margin</td>
                <td className="py-3">Pace/style opinion</td>
              </tr>
              <tr>
                <td className="text-muted-foreground py-3 font-medium">Risk level</td>
                <td className="py-3">Varies by favorite/dog</td>
                <td className="py-3">Usually 50/50 pricing</td>
                <td className="py-3">Usually 50/50 pricing</td>
              </tr>
              <tr>
                <td className="text-muted-foreground py-3 font-medium">On Kalshi</td>
                <td className="py-3 text-emerald-400">Two markets per game</td>
                <td className="py-3">One market</td>
                <td className="py-3">One market</td>
              </tr>
            </tbody>
          </table>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 6: KEY TERMS SUMMARY */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="Check the Glossary">
            For deeper definitions and etymology of any term, visit the{' '}
            <a
              href="/glossary"
              className="text-primary hover:underline"
            >
              Glossary
            </a>
            .
          </TipBlock>
        }
      >
        <SectionHeader
          title="Key Terms to Remember"
          subtitle="Master this vocabulary"
        />

        <div className="grid gap-4 sm:grid-cols-2">
          <ConceptCard
            title="Favorite vs Underdog"
            description="The expected winner (costs more) vs expected loser (pays more if they win)."
            icon={<Trophy className="h-4 w-4" />}
          />
          <ConceptCard
            title="Cover"
            description="Beat the spread. Favorites must win by more; underdogs can lose by less."
            icon={<ArrowUpDown className="h-4 w-4" />}
          />
          <ConceptCard
            title="Hook"
            description="The .5 in spreads/totals that eliminates ties and ensures a winner."
            icon={<Calculator className="h-4 w-4" />}
          />
          <ConceptCard
            title="Push"
            description="A tie (when there's no hook). Your stake is returned, no winner."
            icon={<AlertTriangle className="h-4 w-4" />}
          />
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 7: NEXT STEPS */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="What's Next?"
          subtitle="Understanding Kalshi's unique structure"
        />

        <p className="text-muted-foreground mb-6 leading-relaxed">
          Now that you understand the three bet types, the next step is understanding how Kalshi
          structures these differently from traditional sportsbooks. On Kalshi, moneyline bets have{' '}
          <strong>two separate markets per game</strong> — one for each team. This creates unique
          opportunities that we'll explore next.
        </p>

        {nextGuide && (
          <NextStepBlock
            href={`/${nextGuide.slug}`}
            title={nextGuide.title}
            description={`Learn how Kalshi's dual market structure for moneylines creates opportunities for market makers - and why it matters for your trading.`}
            buttonText="Continue to Next Lesson"
          />
        )}
      </LessonRow>
    </LessonLayout>
  );
}
