import { useState } from 'react';
import { LessonRow, SectionHeader } from '../LessonLayout';
import { TipBlock, InsightBlock, WarningBlock, ConstraintBlock } from '../blocks';

/**
 * User Interview Guide for Market Maker Software
 *
 * Questions to ask your client to uncover hidden needs.
 * Helps product builders run effective discovery calls.
 */

type Question = {
  question: string;
  why: string;
  followUps: string[];
  lookFor: string[];
  redFlags: string[];
};

type QuestionCategory = {
  category: string;
  description: string;
  questions: Question[];
};

function QuestionCard({ question, index }: { question: Question; index: number }) {
  const [isExpanded, setIsExpanded] = useState(false);

  return (
    <div className="overflow-hidden rounded-lg border border-slate-700 bg-slate-800/50">
      <button
        onClick={() => setIsExpanded(!isExpanded)}
        className="flex w-full items-start gap-3 p-4 text-left hover:bg-slate-700/30"
      >
        <span className="flex h-6 w-6 flex-shrink-0 items-center justify-center rounded-full bg-purple-600 text-sm text-white">
          {index + 1}
        </span>
        <div className="flex-1">
          <p className="font-medium text-white">&quot;{question.question}&quot;</p>
        </div>
        <span className="text-slate-400">{isExpanded ? '−' : '+'}</span>
      </button>

      {isExpanded && (
        <div className="space-y-4 px-4 pb-4">
          <div className="ml-9">
            <div className="rounded-lg border border-blue-700/50 bg-blue-900/20 p-3">
              <div className="mb-1 text-xs font-medium text-blue-400">Why ask this?</div>
              <p className="text-sm text-slate-300">{question.why}</p>
            </div>
          </div>

          <div className="ml-9">
            <div className="mb-2 text-xs font-medium text-slate-400">Follow-up questions:</div>
            <ul className="space-y-1">
              {question.followUps.map((q, i) => (
                <li
                  key={i}
                  className="text-sm italic text-slate-300"
                >
                  &quot;{q}&quot;
                </li>
              ))}
            </ul>
          </div>

          <div className="ml-9 grid gap-3 md:grid-cols-2">
            <div className="rounded-lg border border-green-700/50 bg-green-900/20 p-3">
              <div className="mb-1 text-xs font-medium text-green-400">
                Good signs (listen for):
              </div>
              <ul className="space-y-1 text-sm text-slate-300">
                {question.lookFor.map((item, i) => (
                  <li key={i}>• {item}</li>
                ))}
              </ul>
            </div>

            <div className="rounded-lg border border-red-700/50 bg-red-900/20 p-3">
              <div className="mb-1 text-xs font-medium text-red-400">Red flags:</div>
              <ul className="space-y-1 text-sm text-slate-300">
                {question.redFlags.map((item, i) => (
                  <li key={i}>• {item}</li>
                ))}
              </ul>
            </div>
          </div>
        </div>
      )}
    </div>
  );
}

const questionCategories: QuestionCategory[] = [
  {
    category: 'Current Workflow',
    description: 'Understand what they do today',
    questions: [
      {
        question: 'Walk me through your last trading session from start to finish.',
        why: 'Open-ended questions reveal the actual workflow, not what they think they do. Look for the sequence of actions, tools used, and decision points.',
        followUps: [
          'What did you do first when you sat down?',
          'How did you decide which markets to focus on?',
          'When did you take breaks, and why?',
        ],
        lookFor: [
          'Specific details about sequence of actions',
          'Tools and data sources mentioned',
          'Natural decision points in the workflow',
        ],
        redFlags: [
          'Vague or theoretical answers',
          'Describing ideal workflow vs. actual',
          "Can't remember specifics",
        ],
      },
      {
        question: "What's the first thing you check when you sit down to trade?",
        why: "This reveals their mental model of 'readiness to trade' and what information is most critical to them.",
        followUps: [
          'Why that specifically?',
          'What would make you not trade today?',
          'How long does this take?',
        ],
        lookFor: [
          'Consistent priority (same thing every time)',
          'Clear reasoning for why',
          'Quick checklist vs. deep analysis',
        ],
        redFlags: ['Different answer every time', 'No clear priority', 'Skips this step entirely'],
      },
      {
        question: 'What tools or software do you currently use?',
        why: 'Reveals the competitive landscape and what gaps exist in current solutions.',
        followUps: [
          'What do you like about each one?',
          "What's frustrating about them?",
          'If you could change one thing about [tool], what would it be?',
        ],
        lookFor: [
          'Specific complaints about current tools',
          "Workarounds they've built",
          "Features they pay for but don't use",
        ],
        redFlags: [
          'Happy with everything (unlikely, dig deeper)',
          'Uses too many tools (integration opportunity)',
          'Built their own tools (complex needs)',
        ],
      },
    ],
  },
  {
    category: 'Pain Points & Frustrations',
    description: 'Find the problems worth solving',
    questions: [
      {
        question: "What's the most frustrating part of your current setup?",
        why: 'Direct question about pain points. The strength of emotion indicates how valuable a solution would be.',
        followUps: [
          'How often does this happen?',
          'What do you do when it happens?',
          'How much time/money does this cost you?',
        ],
        lookFor: [
          'Strong emotional response (real pain)',
          'Specific examples with details',
          'Quantifiable impact',
        ],
        redFlags: [
          "Can't think of anything (not engaged or hiding something)",
          'Only minor annoyances (might not need your product)',
          'Problems outside your scope',
        ],
      },
      {
        question: 'Tell me about the last time something went wrong during trading.',
        why: 'Stories about failures reveal critical requirements and edge cases the UI must handle.',
        followUps: [
          'What happened exactly?',
          'How did you find out something was wrong?',
          'What did you do to fix it?',
          'How could it have been prevented?',
        ],
        lookFor: [
          'Specific incident with details',
          'Clear cause and effect',
          'Ideas for prevention',
        ],
        redFlags: [
          'Never had problems (not trading much, or not honest)',
          'Blames external factors only',
          'No lessons learned',
        ],
      },
      {
        question: 'What task takes way longer than it should?',
        why: "Efficiency opportunities. These are often features they'd pay for.",
        followUps: [
          'How long does it take now?',
          'How long should it take?',
          'What would you do with the time saved?',
        ],
        lookFor: [
          'Specific time estimates',
          'Repetitive tasks (automation opportunity)',
          'Multi-step processes (simplification opportunity)',
        ],
        redFlags: [
          'Everything takes too long (scope creep)',
          'Problems with learning curve, not tool',
          'Unrealistic expectations',
        ],
      },
    ],
  },
  {
    category: 'Decision Making',
    description: 'Understand how they think',
    questions: [
      {
        question: 'How do you decide whether to take a trade or not?',
        why: 'Reveals their decision framework and what information they need to make decisions.',
        followUps: [
          'What information do you look at?',
          'How long does this decision take?',
          'What makes you pass on a trade?',
        ],
        lookFor: [
          'Clear criteria (can be implemented as filters/alerts)',
          'Consistent process',
          'Speed requirements',
        ],
        redFlags: [
          'No clear criteria (might not know what they need)',
          'Overly complex process (hard to support)',
          'Gut feel only (hard to design for)',
        ],
      },
      {
        question: 'When do you decide to exit a position?',
        why: 'Exit decisions are often more stressful than entries. Reveals different information needs.',
        followUps: [
          'What triggers an exit?',
          'How quickly do you need to act?',
          'What information do you need in that moment?',
        ],
        lookFor: ['Clear exit rules', 'Time sensitivity', 'Specific data needs'],
        redFlags: [
          'No exit plan (risky trader)',
          'Emotional exits (hard to support with UI)',
          'Never exits (different use case)',
        ],
      },
      {
        question: "What's the hardest decision you have to make regularly?",
        why: 'Hard decisions are where the most value can be added. These are features worth investing in.',
        followUps: [
          'Why is it hard?',
          'What would make it easier?',
          'What information would help?',
        ],
        lookFor: ['Information gaps', 'Time pressure', 'Trade-offs that could be visualized'],
        redFlags: [
          'Nothing is hard (overconfident or inexperienced)',
          'Everything is hard (might not be target user)',
          'Problems outside software scope',
        ],
      },
    ],
  },
  {
    category: 'Information Needs',
    description: 'What data matters to them',
    questions: [
      {
        question: "What's the most important number on your screen right now?",
        why: 'Forces prioritization. This is what gets prime real estate in your UI.',
        followUps: [
          'Why that one?',
          "What's the second most important?",
          'What could you live without?',
        ],
        lookFor: [
          'Clear priority',
          'Reasoning behind priority',
          'Different answers in different modes',
        ],
        redFlags: [
          'Everything is equally important (no prioritization skill)',
          "Can't answer (doesn't think systematically)",
          "Metrics you can't provide",
        ],
      },
      {
        question: "What information do you wish you had that you don't have now?",
        why: 'Reveals feature opportunities and data gaps in current solutions.',
        followUps: [
          'Why would that be valuable?',
          'How would you use it?',
          'How much would you pay for it?',
        ],
        lookFor: [
          'Specific data requests',
          'Clear use case',
          'Willingness to pay (prioritization signal)',
        ],
        redFlags: [
          'Asks for impossible data',
          'Nice-to-have vs. need-to-have',
          "Already available but they don't know",
        ],
      },
      {
        question: 'What do you currently track in a spreadsheet or notebook?',
        why: "Things they track manually are features they'd pay to have automated.",
        followUps: [
          'Can I see an example?',
          'How often do you update it?',
          'What do you do with this data?',
        ],
        lookFor: [
          'Specific metrics tracked',
          'Regular updates (important to them)',
          'Analysis they do with the data',
        ],
        redFlags: [
          "Don't track anything (less sophisticated user)",
          'Overly complex tracking (hard to replicate)',
          "Sensitive data they won't share",
        ],
      },
    ],
  },
  {
    category: 'Speed & Performance',
    description: 'Understand latency requirements',
    questions: [
      {
        question: 'How fast do you need the software to respond?',
        why: 'Sets technical requirements. Some traders need milliseconds, others are fine with seconds.',
        followUps: [
          "What's the consequence of a slow response?",
          'Has slowness ever cost you money? How much?',
          "What's 'fast enough' for you?",
        ],
        lookFor: [
          'Specific time requirements',
          'Quantified cost of latency',
          'Reasonable expectations',
        ],
        redFlags: [
          'Demands impossible speed',
          'No sense of requirements',
          "Speed matters but can't say why",
        ],
      },
      {
        question: 'What actions do you need to do with keyboard shortcuts vs. mouse?',
        why: 'Reveals which actions are time-critical and need keyboard optimization.',
        followUps: [
          'What shortcuts do you use now?',
          'What do you wish had a shortcut?',
          'Do you ever use mouse for time-critical actions?',
        ],
        lookFor: [
          'Specific actions that need shortcuts',
          'Existing muscle memory',
          'Actions where speed matters',
        ],
        redFlags: [
          'Uses mouse for everything (less experienced)',
          'Expects every action to have shortcut (scope creep)',
          'No clear priority',
        ],
      },
    ],
  },
  {
    category: 'Validation & Prioritization',
    description: 'Check understanding and priorities',
    questions: [
      {
        question:
          'If I could only build one feature for you, what would have the most impact on your trading?',
        why: 'Forces absolute prioritization. This is your MVP feature.',
        followUps: [
          'Why that one?',
          'How much would that be worth to you?',
          "What's the second choice?",
        ],
        lookFor: ['Clear single answer', 'Strong reasoning', 'Aligns with earlier pain points'],
        redFlags: [
          "Can't choose (indecisive user)",
          'Changes answer frequently',
          'Picks something not mentioned earlier',
        ],
      },
      {
        question: 'What would make you switch from your current tools to something new?',
        why: 'Reveals switching costs and what threshold you need to clear.',
        followUps: [
          "What's keeping you on current tools?",
          'Have you tried switching before? What happened?',
          'What would make it not worth switching?',
        ],
        lookFor: ['Specific pain threshold', 'Clear deal-breakers', 'Realistic expectations'],
        redFlags: [
          'Would never switch (not your customer)',
          'Would switch for anything (low commitment)',
          'Unrealistic requirements',
        ],
      },
      {
        question: "Is there anything I should have asked but didn't?",
        why: 'Catches blind spots. Often reveals the most important insight of the interview.',
        followUps: [
          'Tell me more about that.',
          'Why is that important?',
          'How does that affect your trading?',
        ],
        lookFor: [
          'Surprise insights',
          "Topics they've been waiting to discuss",
          'Corrections to assumptions',
        ],
        redFlags: [
          'Nothing to add (short interview, dig more)',
          'Off-topic tangents',
          'Contradicts earlier answers',
        ],
      },
    ],
  },
];

function InterviewChecklist() {
  const [checks, setChecks] = useState<Record<string, boolean>>({});

  const items = [
    { id: 'prep', text: 'Review their current setup beforehand (if possible)' },
    { id: 'record', text: 'Get permission to record (or take detailed notes)' },
    { id: 'open', text: 'Start with open-ended questions before specific ones' },
    { id: 'silence', text: 'Let silences hang—they often lead to deeper answers' },
    { id: 'stories', text: 'Ask for specific stories, not hypotheticals' },
    { id: 'show', text: 'Ask them to show you, not just tell you' },
    { id: 'why', text: "Ask 'why' at least 3 times to get to root causes" },
    { id: 'summary', text: 'Summarize and confirm understanding before ending' },
    { id: 'followup', text: 'Schedule a follow-up to share what you learned' },
  ];

  return (
    <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-6">
      <h4 className="mb-4 text-lg font-semibold text-white">Interview Best Practices Checklist</h4>
      <div className="space-y-2">
        {items.map((item) => (
          <label
            key={item.id}
            className="flex cursor-pointer items-center gap-3"
          >
            <input
              type="checkbox"
              checked={checks[item.id] || false}
              onChange={(e) => setChecks({ ...checks, [item.id]: e.target.checked })}
              className="h-4 w-4 rounded"
            />
            <span className={`text-sm ${checks[item.id] ? 'text-green-400' : 'text-slate-300'}`}>
              {item.text}
            </span>
          </label>
        ))}
      </div>
    </div>
  );
}

function NoteTakingTemplate() {
  return (
    <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-6">
      <h4 className="mb-4 text-lg font-semibold text-white">Note-Taking Template</h4>
      <p className="mb-4 text-sm text-slate-400">
        Copy this structure for each interview. Fill in as they talk.
      </p>
      <div className="overflow-x-auto rounded-lg bg-slate-900 p-4 font-mono text-sm text-slate-300">
        <pre>{`## Interview: [Name] - [Date]

### Context
- Role:
- Experience level:
- Current tools:
- Trading frequency:

### Key Quotes
- "..."
- "..."

### Workflow Summary
1.
2.
3.

### Pain Points (ranked by intensity)
1. [CRITICAL]
2. [HIGH]
3. [MEDIUM]

### Feature Requests
- Must have:
- Nice to have:
- Mentioned but not priority:

### Information Hierarchy
1. Most important:
2. Second:
3. Third:

### Speed Requirements
- Latency tolerance:
- Keyboard vs mouse:
- Critical actions:

### Surprises / Insights
-
-

### Follow-up Questions
-
-

### Next Steps
- `}</pre>
      </div>
    </div>
  );
}

export function UserInterviewGuideLesson() {
  const [activeCategory, setActiveCategory] = useState(0);

  return (
    <div className="space-y-8">
      {/* Introduction */}
      <section>
        <SectionHeader
          title="User Interview Guide"
          subtitle="Questions that uncover what to build"
        />
        <LessonRow
          aside={
            <ConstraintBlock
              title="Question Categories"
              items={[
                'Current Workflow',
                'Pain Points',
                'Decision Making',
                'Information Needs',
                'Speed & Performance',
                'Validation',
              ]}
            />
          }
        >
          <p className="leading-relaxed text-slate-300">
            The best insights about what to build come from talking to users—but only if you ask the
            right questions. This guide provides a structured approach to interviewing market makers
            that will reveal their real needs, not just what they think they want.
          </p>
          <p className="mt-4 leading-relaxed text-slate-300">
            <strong className="text-white">The goal:</strong> Uncover pain points, understand
            workflows, and prioritize features based on real user needs rather than assumptions.
          </p>
        </LessonRow>
      </section>

      {/* Best Practices */}
      <section>
        <SectionHeader
          title="Before the Interview"
          subtitle="Set yourself up for success"
        />
        <LessonRow
          aside={
            <TipBlock title="Preparation">
              Do your homework first. Know the basics of market making so you can ask informed
              follow-up questions.
            </TipBlock>
          }
        >
          <InterviewChecklist />
        </LessonRow>
      </section>

      {/* Questions by Category */}
      <section>
        <SectionHeader
          title="Interview Questions"
          subtitle="Organized by topic area"
        />
        <LessonRow
          aside={
            <InsightBlock title="Listen, Don't Pitch">
              Your job is to understand, not to sell. Let them talk. Silence is okay— they'll fill
              it with insights.
            </InsightBlock>
          }
        >
          <p className="mb-4 text-sm text-slate-400">
            Click each question to see why it matters, follow-ups, and what to listen for.
          </p>

          <div className="mb-6 flex flex-wrap gap-2">
            {questionCategories.map((cat, i) => (
              <button
                key={cat.category}
                onClick={() => setActiveCategory(i)}
                className={`rounded-lg px-4 py-2 text-sm ${
                  activeCategory === i
                    ? 'bg-purple-600 text-white'
                    : 'bg-slate-700 text-slate-300 hover:bg-slate-600'
                }`}
              >
                {cat.category}
              </button>
            ))}
          </div>

          <div className="mb-4">
            <p className="text-sm text-slate-400">
              {questionCategories[activeCategory].description}
            </p>
          </div>

          <div className="space-y-3">
            {questionCategories[activeCategory].questions.map((q, i) => (
              <QuestionCard
                key={i}
                question={q}
                index={i}
              />
            ))}
          </div>
        </LessonRow>
      </section>

      {/* Note-Taking Template */}
      <section>
        <SectionHeader
          title="Note-Taking Template"
          subtitle="Capture insights systematically"
        />
        <LessonRow
          aside={
            <TipBlock title="Quote Directly">
              Write down their exact words, not your interpretation. Their language reveals their
              mental models.
            </TipBlock>
          }
        >
          <NoteTakingTemplate />
        </LessonRow>
      </section>

      {/* After the Interview */}
      <section>
        <SectionHeader
          title="After the Interview"
          subtitle="Turn insights into action"
        />
        <LessonRow
          aside={
            <InsightBlock title="Patterns > Individual Requests">
              One user's feature request might be personal preference. Three users' same request is
              a pattern worth building.
            </InsightBlock>
          }
        >
          <div className="space-y-4">
            <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-4">
              <h4 className="mb-2 font-semibold text-white">1. Debrief immediately</h4>
              <p className="text-sm text-slate-300">
                Within 30 minutes, write down key insights while fresh. What surprised you? What
                confirmed your assumptions? What new questions do you have?
              </p>
            </div>

            <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-4">
              <h4 className="mb-2 font-semibold text-white">2. Look for patterns</h4>
              <p className="text-sm text-slate-300">
                After 3-5 interviews, patterns emerge. What pain points do multiple users mention?
                What features are consistently requested? What problems are unique vs. universal?
              </p>
            </div>

            <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-4">
              <h4 className="mb-2 font-semibold text-white">3. Prioritize ruthlessly</h4>
              <p className="text-sm text-slate-300">
                Not everything users say is equally important. Prioritize by: frequency (how many
                mentioned it), intensity (how strongly they feel), and willingness to pay.
              </p>
            </div>

            <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-4">
              <h4 className="mb-2 font-semibold text-white">4. Validate before building</h4>
              <p className="text-sm text-slate-300">
                Show rough mockups or prototypes to users before investing in development.
                &quot;Would this solve your problem?&quot; is cheaper to ask than to discover after
                building.
              </p>
            </div>
          </div>
        </LessonRow>
      </section>

      {/* Common Mistakes */}
      <section>
        <SectionHeader
          title="Common Interview Mistakes"
          subtitle="What to avoid"
        />
        <LessonRow
          aside={
            <WarningBlock title="Biggest Mistake">
              Asking "Would you use X?" is useless. People say yes to hypotheticals. Ask about past
              behavior instead.
            </WarningBlock>
          }
        >
          <div className="grid gap-4 md:grid-cols-2">
            <div className="rounded-lg border border-red-700/50 bg-red-900/20 p-4">
              <h4 className="mb-2 font-semibold text-red-400">Leading questions</h4>
              <p className="mb-2 text-sm text-slate-300">
                <span className="text-red-400">✗</span> &quot;Would you like a faster cancel
                button?&quot;
              </p>
              <p className="text-sm text-slate-300">
                <span className="text-green-400">✓</span> &quot;Tell me about the last time you
                needed to cancel orders quickly.&quot;
              </p>
            </div>

            <div className="rounded-lg border border-red-700/50 bg-red-900/20 p-4">
              <h4 className="mb-2 font-semibold text-red-400">Hypothetical questions</h4>
              <p className="mb-2 text-sm text-slate-300">
                <span className="text-red-400">✗</span> &quot;Would you use feature X?&quot;
              </p>
              <p className="text-sm text-slate-300">
                <span className="text-green-400">✓</span> &quot;Tell me about the last time you
                needed to do [thing feature X would help with].&quot;
              </p>
            </div>

            <div className="rounded-lg border border-red-700/50 bg-red-900/20 p-4">
              <h4 className="mb-2 font-semibold text-red-400">Accepting first answer</h4>
              <p className="mb-2 text-sm text-slate-300">
                <span className="text-red-400">✗</span> Moving on after one answer.
              </p>
              <p className="text-sm text-slate-300">
                <span className="text-green-400">✓</span> &quot;Why is that?&quot; and &quot;Can you
                give me an example?&quot;
              </p>
            </div>

            <div className="rounded-lg border border-red-700/50 bg-red-900/20 p-4">
              <h4 className="mb-2 font-semibold text-red-400">Pitching your solution</h4>
              <p className="mb-2 text-sm text-slate-300">
                <span className="text-red-400">✗</span> &quot;We&apos;re building X which will solve
                Y...&quot;
              </p>
              <p className="text-sm text-slate-300">
                <span className="text-green-400">✓</span> Listen first. Pitch later (or never in the
                same session).
              </p>
            </div>
          </div>
        </LessonRow>
      </section>
    </div>
  );
}
