import { useState, useMemo } from 'react';
import { LessonRow, SectionHeader } from '../LessonLayout';
import { GlossaryTooltip } from '@/components/atoms/GlossaryTooltip';
import { Formula } from '@/components/atoms/Formula';
import { TipBlock, InsightBlock, WarningBlock } from '../blocks';

const terms = {
  variance: {
    term: 'Variance',
    definition:
      'The statistical measure of how spread out results are from the expected value. High variance means results swing widely in the short term.',
    category: 'Strategy' as const,
  },
  standardDeviation: {
    term: 'Standard Deviation',
    definition:
      'The square root of variance. About 68% of outcomes fall within 1 standard deviation of the mean, 95% within 2.',
    category: 'Strategy' as const,
  },
  sampleSize: {
    term: 'Sample Size',
    definition:
      'The number of observations or trades in a dataset. Larger samples give more reliable conclusions about true edge.',
    category: 'Strategy' as const,
  },
  drawdown: {
    term: 'Drawdown',
    definition:
      'The decline from a peak in your account value. Maximum drawdown is the largest peak-to-trough decline experienced.',
    category: 'Risk' as const,
  },
  expectedValue: {
    term: 'Expected Value',
    definition:
      'The average outcome if you repeated a bet infinitely many times. Positive EV means you profit long-term.',
    category: 'Strategy' as const,
  },
};

// Monte Carlo simulator for variance visualization
function VarianceSimulatorWidget() {
  const [winRate, setWinRate] = useState(55);
  const [numTraders, setNumTraders] = useState(100);
  const [numBets, setNumBets] = useState(100);
  const [results, setResults] = useState<number[][] | null>(null);

  const runSimulation = () => {
    const allResults: number[][] = [];

    for (let t = 0; t < numTraders; t++) {
      const traderResults: number[] = [0];
      let bankroll = 0;

      for (let b = 0; b < numBets; b++) {
        const won = Math.random() < winRate / 100;
        bankroll += won ? 50 : -50; // Even money bets
        traderResults.push(bankroll);
      }

      allResults.push(traderResults);
    }

    setResults(allResults);
  };

  const stats = useMemo(() => {
    if (!results) return null;

    const finalProfits = results.map((r) => r[r.length - 1]);
    const winners = finalProfits.filter((p) => p > 0).length;
    const losers = finalProfits.filter((p) => p < 0).length;
    const breakeven = finalProfits.filter((p) => p === 0).length;

    const maxProfit = Math.max(...finalProfits);
    const maxLoss = Math.min(...finalProfits);
    const avgProfit = finalProfits.reduce((a, b) => a + b, 0) / finalProfits.length;

    // Find worst drawdown across all traders
    let worstDrawdown = 0;
    for (const trader of results) {
      let peak = 0;
      for (const val of trader) {
        if (val > peak) peak = val;
        const dd = peak - val;
        if (dd > worstDrawdown) worstDrawdown = dd;
      }
    }

    // Find traders who had losing streaks despite +EV
    const losersWithPositiveEV = winRate > 50 ? losers : 0;

    return {
      winners,
      losers,
      breakeven,
      maxProfit,
      maxLoss,
      avgProfit,
      worstDrawdown,
      losersWithPositiveEV,
    };
  }, [results, winRate]);

  const theoreticalEV = (winRate / 100 - 0.5) * 100 * numBets; // 50¢ win/loss per bet

  return (
    <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-6">
      <h4 className="mb-4 text-lg font-semibold text-white">
        Variance Simulator: {numTraders} Traders, Same Strategy
      </h4>
      <p className="mb-6 text-sm text-slate-400">
        Watch {numTraders} traders all using the exact same {winRate}% win rate strategy. Same edge,
        wildly different outcomes.
      </p>

      <div className="mb-6 grid grid-cols-1 gap-4 md:grid-cols-3">
        <div>
          <label className="mb-1 block text-sm text-slate-400">Win Rate (%)</label>
          <input
            type="range"
            min="45"
            max="65"
            value={winRate}
            onChange={(e) => setWinRate(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center font-mono text-white">{winRate}%</div>
        </div>

        <div>
          <label className="mb-1 block text-sm text-slate-400">Number of Traders</label>
          <input
            type="range"
            min="10"
            max="500"
            step="10"
            value={numTraders}
            onChange={(e) => setNumTraders(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center font-mono text-white">{numTraders}</div>
        </div>

        <div>
          <label className="mb-1 block text-sm text-slate-400">Bets per Trader</label>
          <input
            type="range"
            min="20"
            max="500"
            step="10"
            value={numBets}
            onChange={(e) => setNumBets(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center font-mono text-white">{numBets}</div>
        </div>
      </div>

      <div className="mb-4 rounded bg-slate-900/50 p-3 text-center">
        <span className="text-slate-400">Theoretical EV per trader: </span>
        <span className={`font-bold ${theoreticalEV >= 0 ? 'text-green-400' : 'text-red-400'}`}>
          {theoreticalEV >= 0 ? '+' : ''}${theoreticalEV.toFixed(0)}
        </span>
        <span className="ml-2 text-sm text-slate-400">
          ({winRate > 50 ? '+' : ''}
          {((winRate - 50) * 2).toFixed(1)}% edge)
        </span>
        <div className="mt-2 text-xs text-slate-500">
          <Formula>{'EV = (P_{win} - 0.5) \\times \\text{bet size} \\times n'}</Formula>
        </div>
      </div>

      <button
        onClick={runSimulation}
        className="mb-4 w-full rounded-lg bg-purple-600 py-3 font-semibold text-white hover:bg-purple-700"
      >
        Run Simulation
      </button>

      {stats && (
        <div className="space-y-4">
          <div className="grid grid-cols-2 gap-3 md:grid-cols-4">
            <div className="rounded border border-green-800/50 bg-green-900/30 p-3 text-center">
              <div className="text-xs text-slate-400">Profitable</div>
              <div className="text-xl font-bold text-green-400">{stats.winners}</div>
              <div className="text-xs text-slate-500">
                {((stats.winners / numTraders) * 100).toFixed(0)}%
              </div>
            </div>
            <div className="rounded border border-red-800/50 bg-red-900/30 p-3 text-center">
              <div className="text-xs text-slate-400">Lost Money</div>
              <div className="text-xl font-bold text-red-400">{stats.losers}</div>
              <div className="text-xs text-slate-500">
                {((stats.losers / numTraders) * 100).toFixed(0)}%
              </div>
            </div>
            <div className="rounded bg-slate-900/50 p-3 text-center">
              <div className="text-xs text-slate-400">Best Result</div>
              <div className="text-xl font-bold text-green-400">+${stats.maxProfit}</div>
            </div>
            <div className="rounded bg-slate-900/50 p-3 text-center">
              <div className="text-xs text-slate-400">Worst Result</div>
              <div className="text-xl font-bold text-red-400">${stats.maxLoss}</div>
            </div>
          </div>

          <div className="grid grid-cols-2 gap-3">
            <div className="rounded bg-slate-900/50 p-3 text-center">
              <div className="text-xs text-slate-400">Average Result</div>
              <div
                className={`text-xl font-bold ${stats.avgProfit >= 0 ? 'text-green-400' : 'text-red-400'}`}
              >
                {stats.avgProfit >= 0 ? '+' : ''}${stats.avgProfit.toFixed(0)}
              </div>
            </div>
            <div className="rounded bg-slate-900/50 p-3 text-center">
              <div className="text-xs text-slate-400">Worst Drawdown Seen</div>
              <div className="text-xl font-bold text-amber-400">${stats.worstDrawdown}</div>
            </div>
          </div>

          {winRate > 50 && stats.losers > 0 && (
            <div className="rounded-lg border border-amber-700/50 bg-amber-900/30 p-4">
              <p className="mb-1 font-semibold text-amber-400">
                {stats.losers} traders lost money with a winning strategy!
              </p>
              <p className="text-sm text-slate-300">
                Even with a {winRate}% win rate (+EV),{' '}
                {((stats.losers / numTraders) * 100).toFixed(0)}% of traders ended up in the red
                over {numBets} bets. This is variance. Try increasing the number of bets to see how
                more volume reduces the chance of overall loss.
              </p>
            </div>
          )}

          {winRate > 50 && stats.losers === 0 && (
            <div className="rounded-lg border border-green-700/50 bg-green-900/30 p-4">
              <p className="mb-1 font-semibold text-green-400">All traders were profitable!</p>
              <p className="text-sm text-slate-300">
                With enough bets, positive EV eventually dominates variance. Try reducing the number
                of bets to see how variance affects smaller samples.
              </p>
            </div>
          )}
        </div>
      )}
    </div>
  );
}

// Streak probability calculator
function StreakCalculatorWidget() {
  const [winRate, setWinRate] = useState(55);
  const [streakLength, setStreakLength] = useState(5);

  const losingStreakProb = Math.pow((100 - winRate) / 100, streakLength);
  const winningStreakProb = Math.pow(winRate / 100, streakLength);

  // Probability of seeing X losing streak in N bets (approximation)
  const betsToExpectStreak = Math.ceil(1 / losingStreakProb);

  return (
    <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-6">
      <h4 className="mb-4 text-lg font-semibold text-white">Streak Probability Calculator</h4>
      <p className="mb-6 text-sm text-slate-400">
        How likely are losing (and winning) streaks at your win rate?
      </p>

      <div className="mb-6 grid grid-cols-1 gap-4 md:grid-cols-2">
        <div>
          <label className="mb-1 block text-sm text-slate-400">Your Win Rate</label>
          <input
            type="range"
            min="40"
            max="70"
            value={winRate}
            onChange={(e) => setWinRate(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center font-mono text-white">{winRate}%</div>
        </div>

        <div>
          <label className="mb-1 block text-sm text-slate-400">Streak Length</label>
          <input
            type="range"
            min="3"
            max="10"
            value={streakLength}
            onChange={(e) => setStreakLength(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center font-mono text-white">{streakLength} in a row</div>
        </div>
      </div>

      <div className="mb-4 rounded bg-slate-900/50 p-4 text-center">
        <Formula block>{'P(\\text{streak}) = p^n'}</Formula>
        <div className="mt-2 text-xs text-slate-500">
          where <Formula>{'p'}</Formula> = probability per bet, <Formula>{'n'}</Formula> = streak
          length
        </div>
      </div>

      <div className="grid grid-cols-2 gap-4">
        <div className="rounded-lg border border-red-800/50 bg-red-900/30 p-4">
          <div className="mb-1 text-sm text-slate-400">Losing {streakLength} in a row</div>
          <div className="text-2xl font-bold text-red-400">
            {(losingStreakProb * 100).toFixed(2)}%
          </div>
          <div className="mt-1 text-xs text-slate-500">
            <Formula>{`(${((100 - winRate) / 100).toFixed(2)})^{${streakLength}}`}</Formula> = ~1 in{' '}
            {Math.round(1 / losingStreakProb)} starts
          </div>
        </div>

        <div className="rounded-lg border border-green-800/50 bg-green-900/30 p-4">
          <div className="mb-1 text-sm text-slate-400">Winning {streakLength} in a row</div>
          <div className="text-2xl font-bold text-green-400">
            {(winningStreakProb * 100).toFixed(2)}%
          </div>
          <div className="mt-1 text-xs text-slate-500">
            <Formula>{`(${(winRate / 100).toFixed(2)})^{${streakLength}}`}</Formula> = ~1 in{' '}
            {Math.round(1 / winningStreakProb)} starts
          </div>
        </div>
      </div>

      <div className="mt-4 rounded bg-slate-900/50 p-4">
        <p className="text-sm text-slate-300">
          Over <span className="font-semibold text-white">{betsToExpectStreak} bets</span>, you
          should expect to experience at least one {streakLength}-bet losing streak.
          {losingStreakProb > 0.01 && (
            <span className="text-amber-400">
              {' '}
              With {winRate}% win rate, these streaks are common. Don&apos;t let them shake your
              confidence in a winning strategy.
            </span>
          )}
        </p>
      </div>
    </div>
  );
}

// Sample size required calculator
function SampleSizeWidget() {
  const [trueEdge, setTrueEdge] = useState(5);
  const [confidence, setConfidence] = useState(95);

  // Simplified calculation for sample size needed
  // Based on binomial confidence interval
  const zScore = confidence === 90 ? 1.645 : confidence === 95 ? 1.96 : 2.576;
  const p = 0.5 + trueEdge / 200; // Convert edge to probability
  const marginOfError = trueEdge / 200; // We want to detect an edge of this size

  // n = (z² × p × (1-p)) / e²
  const sampleSize = Math.ceil((zScore * zScore * p * (1 - p)) / (marginOfError * marginOfError));

  return (
    <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-6">
      <h4 className="mb-4 text-lg font-semibold text-white">How Many Bets to Prove Your Edge?</h4>
      <p className="mb-6 text-sm text-slate-400">
        How many bets do you need before you can be confident your edge is real?
      </p>

      <div className="mb-6 grid grid-cols-1 gap-4 md:grid-cols-2">
        <div>
          <label className="mb-1 block text-sm text-slate-400">
            Your True Edge (win rate above 50%)
          </label>
          <input
            type="range"
            min="1"
            max="15"
            value={trueEdge}
            onChange={(e) => setTrueEdge(Number(e.target.value))}
            className="w-full"
          />
          <div className="text-center font-mono text-white">
            {trueEdge}% edge ({50 + trueEdge}% win rate)
          </div>
        </div>

        <div>
          <label className="mb-1 block text-sm text-slate-400">Confidence Level</label>
          <div className="flex gap-2">
            {[90, 95, 99].map((c) => (
              <button
                key={c}
                onClick={() => setConfidence(c)}
                className={`flex-1 rounded py-2 ${
                  confidence === c
                    ? 'bg-blue-600 text-white'
                    : 'bg-slate-700 text-slate-300 hover:bg-slate-600'
                }`}
              >
                {c}%
              </button>
            ))}
          </div>
        </div>
      </div>

      <div className="rounded-lg bg-slate-900/50 p-6 text-center">
        <div className="mb-2 text-sm text-slate-400">Bets needed to confirm edge:</div>
        <div className="mb-2 text-4xl font-bold text-white">{sampleSize.toLocaleString()}</div>
        <div className="text-sm text-slate-400">
          At {confidence}% confidence with a {trueEdge}% true edge
        </div>
        <div className="mt-4 border-t border-slate-700 pt-4">
          <Formula block>{'n = \\frac{z^2 \\times p \\times (1-p)}{e^2}'}</Formula>
          <div className="mt-2 text-xs text-slate-500">
            where <Formula>{'z'}</Formula> = z-score, <Formula>{'p'}</Formula> = win probability,{' '}
            <Formula>{'e'}</Formula> = margin of error
          </div>
        </div>
      </div>

      <div className="mt-4 text-sm text-slate-300">
        {sampleSize > 1000 ? (
          <p className="text-amber-400">
            This is why most traders can&apos;t statistically prove their edge. With a small edge,
            variance drowns out signal for hundreds or thousands of bets. You need to trust process
            over results during this period.
          </p>
        ) : (
          <p className="text-green-400">
            A {trueEdge}% edge is large enough that you can confirm it relatively quickly. Most real
            edges in competitive markets are smaller than this.
          </p>
        )}
      </div>
    </div>
  );
}

export function VarianceLuckLesson() {
  return (
    <div className="space-y-8">
      {/* Introduction */}
      <section>
        <SectionHeader
          title="The Brutal Truth About Variance"
          subtitle="Why winning strategies lose money (sometimes for a long time)"
        />
        <LessonRow
          aside={
            <InsightBlock title="The Core Truth">
              Variance is the noise that obscures your edge. Understanding it is the difference
              between surviving long enough for your edge to show and blowing up during a normal
              drawdown.
            </InsightBlock>
          }
        >
          <p className="leading-relaxed text-slate-300">
            Here&apos;s a scenario that destroys unprepared traders: You have a genuine edge. Your
            strategy is mathematically profitable. You execute it perfectly. And you
            <strong className="text-red-400"> still lose money for weeks or months</strong>. Not
            because you did anything wrong, but because of{' '}
            <GlossaryTooltip term={terms.variance}>variance</GlossaryTooltip>.
          </p>
          <p className="mt-4 leading-relaxed text-slate-300">
            Understanding variance—the random noise that obscures your edge—is essential for
            survival. Without this understanding, you&apos;ll abandon winning strategies during
            inevitable downswings and convince yourself that losing strategies work during lucky
            runs.
          </p>
        </LessonRow>
      </section>

      {/* The Variance Simulator */}
      <section>
        <SectionHeader
          title="Same Edge, Different Outcomes"
          subtitle="Watching variance in action"
        />
        <LessonRow>
          <p className="mb-6 leading-relaxed text-slate-300">
            This simulator shows multiple traders using the <em>exact same winning strategy</em>.
            Same edge, same bet sizes, same everything. Watch how wildly their outcomes differ.
          </p>
          <VarianceSimulatorWidget />
          <p className="mt-6 leading-relaxed text-slate-300">
            <strong className="text-white">Key insight:</strong> Results tell you almost nothing
            about whether a strategy works in small samples. Two traders with identical skills can
            have completely opposite results over 100 bets purely due to luck.
          </p>
        </LessonRow>
      </section>

      {/* Losing Streaks are Normal */}
      <section>
        <SectionHeader
          title="Losing Streaks Are Mathematically Guaranteed"
          subtitle="What to expect from randomness"
        />
        <LessonRow
          aside={
            <WarningBlock title="Prepare Now">
              If you can't emotionally handle a 7-bet losing streak with real money, reduce your bet
              size until you can. The streak will come — it's mathematically certain.
            </WarningBlock>
          }
        >
          <p className="mb-6 leading-relaxed text-slate-300">
            Even with a winning strategy, losing streaks aren&apos;t just possible—they&apos;re
            inevitable. The question isn&apos;t <em>if</em> you&apos;ll hit a bad run, but{' '}
            <em>when</em> and <em>how bad</em>.
          </p>
          <StreakCalculatorWidget />
        </LessonRow>
      </section>

      {/* How Long to Prove Your Edge */}
      <section>
        <SectionHeader
          title="How Long Until Results Matter?"
          subtitle="The sample size problem"
        />
        <LessonRow>
          <p className="mb-6 leading-relaxed text-slate-300">
            This is the uncomfortable truth: you need <em>far more</em> data than you think before
            results are meaningful. Small edges are nearly invisible in small samples because{' '}
            <GlossaryTooltip term={terms.variance}>variance</GlossaryTooltip> is louder than signal.
          </p>
          <SampleSizeWidget />
          <p className="mt-6 leading-relaxed text-slate-300">
            Most traders quit profitable strategies or double down on losing ones because they draw
            conclusions from insufficient data. Your 30-bet &ldquo;hot streak&rdquo; or &ldquo;cold
            streak&rdquo; is statistically meaningless.
          </p>
        </LessonRow>
      </section>

      {/* Standard Deviation Intuition */}
      <section>
        <SectionHeader
          title="Understanding Standard Deviation"
          subtitle="The language of uncertainty"
        />
        <LessonRow>
          <p className="leading-relaxed text-slate-300">
            <GlossaryTooltip term={terms.standardDeviation}>Standard deviation</GlossaryTooltip>{' '}
            measures how spread out your results will be. Here&apos;s the practical meaning:
          </p>

          <div className="my-6 rounded-lg border border-slate-700 bg-slate-800/50 p-6">
            <h4 className="mb-4 text-lg font-semibold text-white">What to Expect</h4>
            <div className="space-y-4">
              <div className="flex items-center gap-4">
                <div className="w-20 text-right font-mono text-green-400">68%</div>
                <div className="h-4 flex-1 rounded-full bg-green-900/30" />
                <div className="text-sm text-slate-400">
                  of outcomes within <Formula>{'\\pm 1\\sigma'}</Formula>
                </div>
              </div>
              <div className="flex items-center gap-4">
                <div className="w-20 text-right font-mono text-blue-400">95%</div>
                <div className="h-4 flex-1 rounded-full bg-blue-900/30" />
                <div className="text-sm text-slate-400">
                  of outcomes within <Formula>{'\\pm 2\\sigma'}</Formula>
                </div>
              </div>
              <div className="flex items-center gap-4">
                <div className="w-20 text-right font-mono text-purple-400">99.7%</div>
                <div className="h-4 flex-1 rounded-full bg-purple-900/30" />
                <div className="text-sm text-slate-400">
                  of outcomes within <Formula>{'\\pm 3\\sigma'}</Formula>
                </div>
              </div>
            </div>
          </div>

          <p className="leading-relaxed text-slate-300">
            <strong className="text-white">Translation for traders:</strong> If your strategy has
            high variance, expect wild swings. Results <Formula>{'2\\sigma'}</Formula> from expected
            aren&apos;t &ldquo;unlucky&rdquo;—they happen 5% of the time. That&apos;s roughly once
            every 20 trading periods.
          </p>

          <div className="mt-4 rounded-lg bg-slate-900/50 p-4 text-sm">
            <p className="text-slate-400">Example: 55% win rate, 100 bets, $50 per bet</p>
            <p className="mt-2 text-white">
              Expected profit: <Formula>{'E[X] = \\$500'}</Formula>
            </p>
            <p className="text-white">
              Standard deviation: <Formula>{'\\sigma \\approx \\$500'}</Formula>
            </p>
            <p className="mt-2 text-slate-400">68% of the time: Profit between $0 and $1000</p>
            <p className="text-slate-400">95% of the time: Profit between -$500 and $1500</p>
            <p className="mt-2 text-amber-400">
              5% of the time: You lose money despite being profitable!
            </p>
          </div>
        </LessonRow>
      </section>

      {/* Drawdowns */}
      <section>
        <SectionHeader
          title="Preparing for Drawdowns"
          subtitle="The peak-to-trough experience"
        />
        <LessonRow>
          <p className="leading-relaxed text-slate-300">
            A <GlossaryTooltip term={terms.drawdown}>drawdown</GlossaryTooltip> is the decline from
            your account&apos;s peak to its lowest point. Every trader experiences drawdowns—the
            question is how deep and how long.
          </p>

          <div className="my-6 rounded-lg border border-slate-700 bg-slate-800/50 p-6">
            <h4 className="mb-4 text-lg font-semibold text-white">
              Expected Drawdowns by Win Rate
            </h4>
            <div className="space-y-3">
              <div>
                <div className="mb-1 flex justify-between text-sm">
                  <span className="text-slate-400">52% win rate (2% edge)</span>
                  <span className="text-red-400">Expect 30-40% max drawdown</span>
                </div>
                <div className="h-2 overflow-hidden rounded-full bg-slate-700">
                  <div
                    className="h-full rounded-full bg-red-500"
                    style={{ width: '35%' }}
                  />
                </div>
              </div>
              <div>
                <div className="mb-1 flex justify-between text-sm">
                  <span className="text-slate-400">55% win rate (5% edge)</span>
                  <span className="text-amber-400">Expect 20-30% max drawdown</span>
                </div>
                <div className="h-2 overflow-hidden rounded-full bg-slate-700">
                  <div
                    className="h-full rounded-full bg-amber-500"
                    style={{ width: '25%' }}
                  />
                </div>
              </div>
              <div>
                <div className="mb-1 flex justify-between text-sm">
                  <span className="text-slate-400">60% win rate (10% edge)</span>
                  <span className="text-yellow-400">Expect 15-20% max drawdown</span>
                </div>
                <div className="h-2 overflow-hidden rounded-full bg-slate-700">
                  <div
                    className="h-full rounded-full bg-yellow-500"
                    style={{ width: '17%' }}
                  />
                </div>
              </div>
              <div>
                <div className="mb-1 flex justify-between text-sm">
                  <span className="text-slate-400">65% win rate (15% edge)</span>
                  <span className="text-green-400">Expect 10-15% max drawdown</span>
                </div>
                <div className="h-2 overflow-hidden rounded-full bg-slate-700">
                  <div
                    className="h-full rounded-full bg-green-500"
                    style={{ width: '12%' }}
                  />
                </div>
              </div>
            </div>
          </div>

          <p className="leading-relaxed text-slate-300">
            <strong className="text-white">These drawdowns happen to winning strategies.</strong> If
            you have a 55% edge and you&apos;re not psychologically prepared to watch 25% of your
            bankroll disappear, you&apos;ll either quit or make emotional decisions that destroy
            your edge.
          </p>
        </LessonRow>
      </section>

      {/* Luck vs Skill */}
      <section>
        <SectionHeader
          title="Distinguishing Luck from Skill"
          subtitle="The hardest problem in trading"
        />
        <LessonRow
          aside={
            <TipBlock title="The Process Test">
              Instead of asking "Am I winning?", ask "Can I explain why I should win, independent of
              my results?" If you can't, you might be lucky, not skilled.
            </TipBlock>
          }
        >
          <p className="mb-6 leading-relaxed text-slate-300">
            After 50 bets, you&apos;re up 30%. Is your strategy good, or are you lucky? After 50
            bets, you&apos;re down 20%. Is your strategy bad, or are you unlucky?
          </p>

          <div className="mb-6 grid gap-4 md:grid-cols-2">
            <div className="rounded-lg border border-green-800/50 bg-green-900/20 p-4">
              <h4 className="mb-2 font-semibold text-green-400">Signs You Might Have Skill</h4>
              <ul className="space-y-2 text-sm text-slate-300">
                <li>• You can articulate WHY you have edge (not just results)</li>
                <li>• Your edge has logical basis (information, speed, analysis)</li>
                <li>• Results persist across different market conditions</li>
                <li>• Other smart traders recognize your edge</li>
                <li>• You win in ways you predicted for reasons you expected</li>
              </ul>
            </div>

            <div className="rounded-lg border border-red-800/50 bg-red-900/20 p-4">
              <h4 className="mb-2 font-semibold text-red-400">Signs You Might Be Lucky</h4>
              <ul className="space-y-2 text-sm text-slate-300">
                <li>• You can&apos;t explain your edge without citing results</li>
                <li>• Results came from a few big wins on unusual events</li>
                <li>• Strategy only worked in specific market conditions</li>
                <li>• You win in unexpected ways for unexpected reasons</li>
                <li>• You changed strategies mid-stream to chase what&apos;s working</li>
              </ul>
            </div>
          </div>

          <div className="rounded-lg border border-amber-700/50 bg-amber-900/20 p-4">
            <h4 className="mb-2 font-semibold text-amber-400">The Humble Truth</h4>
            <p className="text-sm text-slate-300">
              In the short run, you genuinely cannot know if you&apos;re skilled or lucky. This
              isn&apos;t false modesty—it&apos;s mathematical reality. The best you can do is focus
              on process (Is my reasoning sound? Am I finding legitimate edge?) rather than results.
              Over sufficient time, skill dominates luck. But &ldquo;sufficient time&rdquo; is
              measured in hundreds or thousands of bets.
            </p>
          </div>
        </LessonRow>
      </section>

      {/* Practical Strategies */}
      <section>
        <SectionHeader
          title="Surviving Variance"
          subtitle="Practical strategies for the long run"
        />
        <LessonRow
          aside={
            <InsightBlock title="Pre-Commit">
              Before starting a strategy, write down: "I will make 200 bets before judging results."
              This simple commitment prevents you from quitting during normal variance.
            </InsightBlock>
          }
        >
          <div className="space-y-6">
            <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-4">
              <h4 className="mb-2 font-semibold text-blue-400">1. Size for Survival</h4>
              <p className="text-sm text-slate-300">
                Position size so you can survive the worst drawdowns variance can deliver. If you
                size for average outcomes, you&apos;ll blow up during inevitable bad streaks. The
                Kelly Criterion (discussed in Risk Management) accounts for variance in optimal
                sizing.
              </p>
            </div>

            <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-4">
              <h4 className="mb-2 font-semibold text-blue-400">2. Track Process, Not Results</h4>
              <p className="text-sm text-slate-300">
                Keep a trading journal that focuses on decision quality, not outcomes. Did you
                identify edge correctly? Did you size appropriately? Did you execute your plan?
                These process metrics matter more than P&L in small samples.
              </p>
            </div>

            <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-4">
              <h4 className="mb-2 font-semibold text-blue-400">3. Pre-commit to Sample Size</h4>
              <p className="text-sm text-slate-300">
                Before trading a strategy, decide how many bets you&apos;ll make before evaluating
                results. Write this number down. Then stick to it regardless of interim results.
                This prevents abandoning strategies during normal variance.
              </p>
            </div>

            <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-4">
              <h4 className="mb-2 font-semibold text-blue-400">
                4. Diversify Across Opportunities
              </h4>
              <p className="text-sm text-slate-300">
                Variance decreases with more independent bets. If you can find edge across multiple
                markets or strategies, your aggregate variance shrinks. This is why professional
                traders seek many small edges rather than one big bet.
              </p>
            </div>

            <div className="rounded-lg border border-slate-700 bg-slate-800/50 p-4">
              <h4 className="mb-2 font-semibold text-blue-400">
                5. Expect to Be Wrong About Your Edge
              </h4>
              <p className="text-sm text-slate-300">
                Some strategies you believe are +EV will turn out to be -EV. This is normal. The key
                is recognizing it eventually (after enough sample) and adjusting. Build in periodic
                strategy reviews with enough data to be meaningful.
              </p>
            </div>
          </div>
        </LessonRow>
      </section>

      {/* The Mental Game */}
      <section>
        <SectionHeader
          title="The Mental Challenge"
          subtitle="Staying rational through randomness"
        />
        <LessonRow
          aside={
            <WarningBlock title="Size for the Worst">
              If your position size would cause emotional distress during a 30% drawdown, it's too
              big. Size for psychological survival, not optimal returns.
            </WarningBlock>
          }
        >
          <p className="leading-relaxed text-slate-300">
            Understanding variance intellectually is easy. Living through it emotionally is hard.
            When you&apos;re in a 20-bet losing streak with real money, knowing &ldquo;this is
            normal variance&rdquo; doesn&apos;t make it feel better.
          </p>

          <div className="my-6 rounded-lg border border-slate-700 bg-slate-800/50 p-6">
            <h4 className="mb-4 text-lg font-semibold text-white">
              Emotional Preparation Checklist
            </h4>
            <div className="space-y-3">
              <label className="flex items-start gap-3">
                <input
                  type="checkbox"
                  className="mt-1"
                />
                <span className="text-slate-300">
                  I accept that I will have 10+ bet losing streaks even with a winning strategy
                </span>
              </label>
              <label className="flex items-start gap-3">
                <input
                  type="checkbox"
                  className="mt-1"
                />
                <span className="text-slate-300">
                  I understand my max drawdown could exceed 25% of bankroll in normal conditions
                </span>
              </label>
              <label className="flex items-start gap-3">
                <input
                  type="checkbox"
                  className="mt-1"
                />
                <span className="text-slate-300">
                  I&apos;ve sized positions so a 30% drawdown won&apos;t affect my life
                </span>
              </label>
              <label className="flex items-start gap-3">
                <input
                  type="checkbox"
                  className="mt-1"
                />
                <span className="text-slate-300">
                  I won&apos;t evaluate my strategy until I have 200+ bets of data
                </span>
              </label>
              <label className="flex items-start gap-3">
                <input
                  type="checkbox"
                  className="mt-1"
                />
                <span className="text-slate-300">
                  I have a written plan for what I&apos;ll do during a bad streak (hint: nothing
                  different)
                </span>
              </label>
            </div>
          </div>

          <p className="leading-relaxed text-slate-300">
            If you can&apos;t check all these boxes honestly, you&apos;re not ready to trade with
            real money. Variance will break you before your edge has a chance to show.
          </p>
        </LessonRow>
      </section>

      {/* Key Takeaways */}
      <section>
        <SectionHeader
          title="Key Takeaways"
          subtitle="What to remember about variance"
        />
        <LessonRow>
          <div className="rounded-lg border border-purple-800/50 bg-gradient-to-r from-purple-900/30 to-blue-900/30 p-6">
            <ul className="space-y-3 text-slate-300">
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>
                  <GlossaryTooltip term={terms.variance}>Variance</GlossaryTooltip> causes winning
                  strategies to lose money and losing strategies to make money—temporarily.
                </span>
              </li>
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>
                  Losing streaks of 7+ bets are normal with a 55% win rate. Don&apos;t let them
                  shake your confidence in a sound strategy.
                </span>
              </li>
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>
                  You need hundreds of bets before results become statistically meaningful.
                  Conclusions from 50 bets are noise.
                </span>
              </li>
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>
                  Expect <GlossaryTooltip term={terms.drawdown}>drawdowns</GlossaryTooltip> of
                  20-30% even with a genuine edge. Size accordingly.
                </span>
              </li>
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>Focus on process (Is my reasoning sound?) not results (Did I win?).</span>
              </li>
              <li className="flex gap-2">
                <span className="text-green-400">✓</span>
                <span>
                  <strong className="text-white">
                    The goal isn&apos;t to avoid variance—it&apos;s to survive it long enough for
                    skill to dominate luck.
                  </strong>
                </span>
              </li>
            </ul>
          </div>
        </LessonRow>
      </section>
    </div>
  );
}
