/**
 * Lesson: YES/NO Markets Explained
 *
 * Understanding Kalshi's dual market structure for sports.
 * Expanded version with deep explanations of order books, liquidity, and why this matters.
 */

import {
  GitBranch,
  Scale,
  BookOpen,
  Users,
  ArrowLeftRight,
  Lightbulb,
  AlertTriangle,
} from 'lucide-react';
import { getGuide, getNextGuide } from '@/data/guides';
import { LessonLayout, LessonRow, LessonHeader, SectionHeader } from '../LessonLayout';
import {
  ObjectiveBlock,
  ConstraintBlock,
  TipBlock,
  InsightBlock,
  WarningBlock,
  ConceptBlock,
  ConceptCard,
  NextStepBlock,
  ExampleBox,
  Term,
} from '../blocks';
import { GlossaryTooltip } from '@/components/atoms/GlossaryTooltip';
import { DualMarketVisualizer } from '../widgets/DualMarketVisualizer';
import { OrderBookVisualizer } from '../widgets/OrderBookVisualizer';

// Custom glossary terms for this lesson
const terms = {
  orderBook: {
    term: 'Order Book',
    definition:
      'A real-time list of all buy and sell orders, organized by price. Shows what people are willing to pay (bids) and accept (asks).',
    category: 'Order Book',
  },
  bid: {
    term: 'Bid',
    definition:
      'The highest price a buyer is willing to pay. Multiple bids stack up at different prices.',
    category: 'Order Book',
  },
  ask: {
    term: 'Ask (Offer)',
    definition: 'The lowest price a seller is willing to accept. Also called the "offer" price.',
    category: 'Order Book',
  },
  spread: {
    term: 'Spread',
    definition:
      'The difference between the best bid and best ask. Tighter spreads mean more liquidity.',
    category: 'Order Book',
  },
  liquidity: {
    term: 'Liquidity',
    definition:
      'How easily you can trade without moving the price. High liquidity = many orders, tight spreads.',
    category: 'Markets',
  },
  queuePosition: {
    term: 'Queue Position',
    definition:
      'Your place in line when multiple orders exist at the same price. First in, first out (FIFO).',
    category: 'Order Book',
  },
  counterparty: {
    term: 'Counterparty',
    definition:
      'The other side of your trade. When you buy, someone sells to you - that seller is your counterparty.',
    category: 'Trading',
  },
  dualMarket: {
    term: 'Dual Market',
    definition:
      'Kalshi creates two separate markets for each game - one asking "Will Team A win?" and one asking "Will Team B win?"',
    category: 'Markets',
  },
};

export function YesNoMarketsLesson() {
  const guide = getGuide('yes-no-markets')!;
  const nextGuide = getNextGuide('yes-no-markets');

  return (
    <LessonLayout>
      {/* Header */}
      <LessonRow>
        <LessonHeader
          title={guide.title}
          description={guide.description}
          duration="12 min"
          category="Sports Betting"
        />
      </LessonRow>

      {/* Objective */}
      <LessonRow
        aside={
          <ConstraintBlock
            title="What We'll Cover"
            items={[
              'Why Kalshi creates two markets per game',
              'How YES/NO sides relate across markets',
              'What order books are and why they matter',
              'Why equivalent positions exist in different books',
            ]}
          />
        }
      >
        <ObjectiveBlock>
          By the end of this lesson, you'll understand Kalshi's unique dual market structure for
          moneyline bets, how order books work, and why the same outcome can be traded through
          different positions - a concept essential for understanding market making strategy.
        </ObjectiveBlock>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 1: WHY THIS IS CONFUSING */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="Key Insight">
            "Lakers YES" and "Celtics NO" both pay $1 if Lakers win. But they live in separate order
            books with potentially different prices and liquidity.
          </TipBlock>
        }
      >
        <SectionHeader
          title="Why Two Markets?"
          subtitle="The question that confuses everyone at first"
        />

        <p className="text-muted-foreground leading-relaxed">
          When you first look at Kalshi's sports markets, something seems redundant. For a simple
          Lakers vs Celtics game, you see <em>two</em> separate markets:
        </p>

        <div className="mt-6 grid gap-4 sm:grid-cols-2">
          <div className="bg-card rounded-lg border p-4">
            <h4 className="mb-2 flex items-center gap-2 font-semibold">
              <div className="flex h-6 w-6 items-center justify-center rounded bg-amber-500/20">
                <span className="text-xs font-bold text-amber-400">A</span>
              </div>
              "Will Lakers win?"
            </h4>
            <ul className="text-muted-foreground space-y-1 text-sm">
              <li>
                • <strong>YES</strong> = Lakers win
              </li>
              <li>
                • <strong>NO</strong> = Lakers lose (Celtics win)
              </li>
            </ul>
          </div>
          <div className="bg-card rounded-lg border p-4">
            <h4 className="mb-2 flex items-center gap-2 font-semibold">
              <div className="flex h-6 w-6 items-center justify-center rounded bg-violet-500/20">
                <span className="text-xs font-bold text-violet-400">B</span>
              </div>
              "Will Celtics win?"
            </h4>
            <ul className="text-muted-foreground space-y-1 text-sm">
              <li>
                • <strong>YES</strong> = Celtics win
              </li>
              <li>
                • <strong>NO</strong> = Celtics lose (Lakers win)
              </li>
            </ul>
          </div>
        </div>

        <div className="from-primary/10 via-primary/5 border-primary/30 mt-6 rounded-lg border bg-gradient-to-br to-transparent p-4">
          <p className="text-foreground">
            <strong>Wait - isn't "Lakers YES" the same as "Celtics NO"?</strong>
          </p>
          <p className="text-muted-foreground mt-2">
            Yes, economically they're identical. Both pay $1 if Lakers win. But they exist in
            <em>separate order books</em> - and that distinction is what creates trading
            opportunities.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 2: THE DUAL MARKET STRUCTURE */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="Design Choice">
            Kalshi could have created a single market with "Lakers" and "Celtics" as the two
            outcomes. Instead, they create two separate YES/NO markets. This is intentional - it
            provides flexibility for traders.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="The Dual Market Structure"
          subtitle="One game, two markets, four positions"
        />

        <p className="text-muted-foreground leading-relaxed">
          Kalshi's <GlossaryTooltip term={terms.dualMarket}>dual market structure</GlossaryTooltip>{' '}
          creates four possible positions for every two-team game:
        </p>

        <DualMarketVisualizer />

        <div className="mt-6 overflow-x-auto">
          <table className="w-full text-sm">
            <thead>
              <tr className="border-b">
                <th className="py-3 text-left font-medium">Position</th>
                <th className="py-3 text-left font-medium">Pays $1 if...</th>
                <th className="py-3 text-left font-medium">Economically equivalent to</th>
              </tr>
            </thead>
            <tbody className="divide-y">
              <tr className="bg-emerald-500/5">
                <td className="py-3 font-medium">Lakers YES</td>
                <td className="text-muted-foreground py-3">Lakers win</td>
                <td className="py-3">
                  <span className="font-medium text-emerald-400">= Celtics NO</span>
                </td>
              </tr>
              <tr className="bg-emerald-500/5">
                <td className="py-3 font-medium">Celtics NO</td>
                <td className="text-muted-foreground py-3">Lakers win (Celtics lose)</td>
                <td className="py-3">
                  <span className="font-medium text-emerald-400">= Lakers YES</span>
                </td>
              </tr>
              <tr className="bg-rose-500/5">
                <td className="py-3 font-medium">Lakers NO</td>
                <td className="text-muted-foreground py-3">Celtics win (Lakers lose)</td>
                <td className="py-3">
                  <span className="font-medium text-rose-400">= Celtics YES</span>
                </td>
              </tr>
              <tr className="bg-rose-500/5">
                <td className="py-3 font-medium">Celtics YES</td>
                <td className="text-muted-foreground py-3">Celtics win</td>
                <td className="py-3">
                  <span className="font-medium text-rose-400">= Lakers NO</span>
                </td>
              </tr>
            </tbody>
          </table>
        </div>

        <div className="bg-muted/50 mt-4 rounded-lg p-3">
          <p className="text-muted-foreground text-sm">
            <strong>Key insight:</strong> There are four positions but only two possible outcomes
            (Lakers win or Celtics win). This means pairs of positions are economically equivalent,
            but they trade in separate order books.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 3: UNDERSTANDING ORDER BOOKS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <ConceptBlock title="Order Book">
            Unlike a sportsbook where you see one price, an exchange shows you the full{' '}
            <GlossaryTooltip term={terms.orderBook}>order book</GlossaryTooltip> - all buy and sell
            orders from all traders, organized by price.
          </ConceptBlock>
        }
      >
        <SectionHeader
          title="What is an Order Book?"
          subtitle="The mechanism that sets prices on an exchange"
        />

        <p className="text-muted-foreground leading-relaxed">
          On a sportsbook, the house shows you one price: "Lakers -150" or "Lakers 65%." Take it or
          leave it. On Kalshi, you see the full{' '}
          <GlossaryTooltip term={terms.orderBook}>order book</GlossaryTooltip> - a list of every buy
          and sell order from every trader.
        </p>

        <div className="mt-6 space-y-4">
          <Term term="Bids (Buy Orders)">
            <GlossaryTooltip term={terms.bid}>Bids</GlossaryTooltip> are offers to buy. "I'll buy at
            55¢" is a bid. The highest bid is called the "best bid" - it's the most someone is
            willing to pay right now. If you want to sell immediately, you sell to the best bid.
          </Term>

          <Term term="Asks (Sell Orders)">
            <GlossaryTooltip term={terms.ask}>Asks</GlossaryTooltip> (also called "offers") are
            offers to sell. "I'll sell at 58¢" is an ask. The lowest ask is the "best ask" - it's
            the cheapest price available. If you want to buy immediately, you buy from the best ask.
          </Term>

          <Term term="The Spread">
            The <GlossaryTooltip term={terms.spread}>spread</GlossaryTooltip> is the gap between the
            best bid and best ask. If best bid is 55¢ and best ask is 58¢, the spread is 3¢. Tighter
            spreads indicate more{' '}
            <GlossaryTooltip term={terms.liquidity}>liquidity</GlossaryTooltip>.
          </Term>
        </div>

        <ExampleBox title="Reading an Order Book">
          <div className="grid gap-4 sm:grid-cols-2">
            <div>
              <div className="mb-3 text-sm font-medium text-emerald-400">Bids (Buyers)</div>
              <div className="space-y-1 font-mono text-sm">
                <div className="flex justify-between rounded border border-emerald-500/20 bg-emerald-500/10 p-2">
                  <span className="text-emerald-400">55¢</span>
                  <span className="text-muted-foreground">500 contracts</span>
                  <span className="text-xs text-emerald-400/70">← Best Bid</span>
                </div>
                <div className="flex justify-between rounded bg-emerald-500/5 p-2">
                  <span>54¢</span>
                  <span className="text-muted-foreground">300 contracts</span>
                </div>
                <div className="flex justify-between rounded bg-emerald-500/5 p-2">
                  <span>53¢</span>
                  <span className="text-muted-foreground">800 contracts</span>
                </div>
              </div>
              <p className="text-muted-foreground mt-2 text-xs">
                If you want to sell now, you get 55¢
              </p>
            </div>
            <div>
              <div className="mb-3 text-sm font-medium text-rose-400">Asks (Sellers)</div>
              <div className="space-y-1 font-mono text-sm">
                <div className="flex justify-between rounded border border-rose-500/20 bg-rose-500/10 p-2">
                  <span className="text-rose-400">58¢</span>
                  <span className="text-muted-foreground">400 contracts</span>
                  <span className="text-xs text-rose-400/70">← Best Ask</span>
                </div>
                <div className="flex justify-between rounded bg-rose-500/5 p-2">
                  <span>59¢</span>
                  <span className="text-muted-foreground">250 contracts</span>
                </div>
                <div className="flex justify-between rounded bg-rose-500/5 p-2">
                  <span>60¢</span>
                  <span className="text-muted-foreground">600 contracts</span>
                </div>
              </div>
              <p className="text-muted-foreground mt-2 text-xs">
                If you want to buy now, you pay 58¢
              </p>
            </div>
          </div>
          <div className="mt-4 border-t pt-4 text-center">
            <span className="text-sm font-medium">Spread: </span>
            <span className="text-primary font-bold">3¢</span>
            <span className="text-muted-foreground text-sm"> (58¢ - 55¢)</span>
          </div>
        </ExampleBox>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 4: WHY SEPARATE ORDER BOOKS MATTER */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <WarningBlock title="Don't Assume Equal Prices">
            Even though "Lakers YES" and "Celtics NO" are economically equivalent, their order books
            can have different prices, spreads, and liquidity.
          </WarningBlock>
        }
      >
        <SectionHeader
          title="Why Separate Books Matter"
          subtitle="Same outcome, different trading characteristics"
        />

        <p className="text-muted-foreground leading-relaxed">
          Here's the crucial insight: each market has its <em>own</em> order book. "Lakers YES"
          order book is completely separate from "Celtics NO" order book, even though both positions
          pay $1 if Lakers win.
        </p>

        <div className="bg-card mt-6 rounded-lg border p-5">
          <h4 className="mb-4 font-semibold">What Can Differ Between Equivalent Positions</h4>
          <div className="grid gap-4 sm:grid-cols-2">
            <div className="flex items-start gap-3">
              <div className="bg-primary/10 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
                <Scale className="text-primary h-4 w-4" />
              </div>
              <div>
                <div className="font-medium">Price</div>
                <p className="text-muted-foreground text-sm">
                  Lakers YES might be 65¢ while Celtics NO might be 64¢ - small differences can
                  exist.
                </p>
              </div>
            </div>
            <div className="flex items-start gap-3">
              <div className="bg-primary/10 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
                <BookOpen className="text-primary h-4 w-4" />
              </div>
              <div>
                <div className="font-medium">Spread</div>
                <p className="text-muted-foreground text-sm">
                  One market might have a 2¢ spread while the other has a 4¢ spread.
                </p>
              </div>
            </div>
            <div className="flex items-start gap-3">
              <div className="bg-primary/10 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
                <Users className="text-primary h-4 w-4" />
              </div>
              <div>
                <div className="font-medium">Liquidity</div>
                <p className="text-muted-foreground text-sm">
                  One market might have 10,000 contracts available, the other only 2,000.
                </p>
              </div>
            </div>
            <div className="flex items-start gap-3">
              <div className="bg-primary/10 flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full">
                <GitBranch className="text-primary h-4 w-4" />
              </div>
              <div>
                <div className="font-medium">Queue Position</div>
                <p className="text-muted-foreground text-sm">
                  Your <GlossaryTooltip term={terms.queuePosition}>queue position</GlossaryTooltip>{' '}
                  in one book may be better than in the other.
                </p>
              </div>
            </div>
          </div>
        </div>

        <OrderBookVisualizer />
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 5: QUEUE POSITION EXPLAINED */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <InsightBlock title="First Come, First Served">
            Order books use FIFO (first in, first out). If 10 people want to buy at 55¢, the first
            order placed gets filled first when someone sells at that price.
          </InsightBlock>
        }
      >
        <SectionHeader
          title="Understanding Queue Position"
          subtitle="Why your place in line matters"
        />

        <p className="text-muted-foreground leading-relaxed">
          <GlossaryTooltip term={terms.queuePosition}>Queue position</GlossaryTooltip> is your place
          in line when multiple orders exist at the same price. On exchanges, orders fill in the
          order they were placed - first in, first out (FIFO).
        </p>

        <ExampleBox title="Queue Position Example">
          <div className="space-y-4">
            <p className="text-sm">
              <strong>Scenario:</strong> Multiple traders want to buy at 55¢
            </p>

            <div className="bg-muted/50 rounded-lg p-4">
              <div className="mb-3 text-sm font-medium">Orders at 55¢ (in order placed)</div>
              <div className="space-y-2 font-mono text-sm">
                <div className="flex items-center gap-3 rounded bg-emerald-500/10 p-2">
                  <span className="bg-primary/20 text-primary flex h-6 w-6 items-center justify-center rounded-full text-xs">
                    1
                  </span>
                  <span>Trader A - 100 contracts</span>
                  <span className="ml-auto text-xs text-emerald-400">First in line</span>
                </div>
                <div className="bg-muted flex items-center gap-3 rounded p-2">
                  <span className="bg-muted-foreground/20 text-muted-foreground flex h-6 w-6 items-center justify-center rounded-full text-xs">
                    2
                  </span>
                  <span>Trader B - 200 contracts</span>
                </div>
                <div className="bg-muted flex items-center gap-3 rounded p-2">
                  <span className="bg-muted-foreground/20 text-muted-foreground flex h-6 w-6 items-center justify-center rounded-full text-xs">
                    3
                  </span>
                  <span>Trader C - 150 contracts</span>
                </div>
                <div className="flex items-center gap-3 rounded bg-violet-500/10 p-2">
                  <span className="flex h-6 w-6 items-center justify-center rounded-full bg-violet-500/20 text-xs text-violet-400">
                    4
                  </span>
                  <span>You - 50 contracts</span>
                  <span className="ml-auto text-xs text-violet-400">Last in line</span>
                </div>
              </div>
            </div>

            <p className="text-muted-foreground text-sm">
              If a seller sells 250 contracts at 55¢, Trader A gets filled (100), then Trader B gets
              partially filled (150 of their 200). You don't get filled at all - you're 4th in line
              and the sell order ran out.
            </p>
          </div>
        </ExampleBox>

        <div className="bg-primary/5 border-primary/20 mt-6 rounded-lg border p-4">
          <h4 className="mb-2 flex items-center gap-2 font-semibold">
            <Lightbulb className="text-primary h-4 w-4" />
            Why Queue Position Matters for Dual Markets
          </h4>
          <p className="text-muted-foreground text-sm">
            If "Lakers YES" has a long queue at 65¢ but "Celtics NO" has a short queue at 35¢, a
            market maker might prefer to post on Celtics NO - they'll get filled sooner when the
            economically equivalent flow comes in.
          </p>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 6: COUNTERPARTIES AND MATCHING */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <ConceptBlock title="Counterparty">
            Every trade has two sides. When you buy Lakers YES, someone is selling you that
            position. That seller is your{' '}
            <GlossaryTooltip term={terms.counterparty}>counterparty</GlossaryTooltip>.
          </ConceptBlock>
        }
      >
        <SectionHeader
          title="How Trades Match"
          subtitle="Finding your counterparty"
        />

        <p className="text-muted-foreground leading-relaxed">
          On an exchange, every trade needs a{' '}
          <GlossaryTooltip term={terms.counterparty}>counterparty</GlossaryTooltip>. When you buy,
          someone sells. When you sell, someone buys. The exchange matches these orders
          automatically.
        </p>

        <div className="bg-card mt-6 rounded-lg border p-5">
          <h4 className="mb-4 font-semibold">The Matching Logic</h4>
          <div className="space-y-4">
            <div className="flex items-center gap-4">
              <div className="flex-1 rounded bg-emerald-500/10 p-3 text-center">
                <div className="font-medium text-emerald-400">You buy Lakers YES</div>
                <div className="text-muted-foreground mt-1 text-xs">at 65¢</div>
              </div>
              <ArrowLeftRight className="text-muted-foreground h-5 w-5 flex-shrink-0" />
              <div className="flex-1 rounded bg-rose-500/10 p-3 text-center">
                <div className="font-medium text-rose-400">Someone sells Lakers YES</div>
                <div className="text-muted-foreground mt-1 text-xs">at 65¢</div>
              </div>
            </div>

            <p className="text-muted-foreground text-center text-sm">
              The seller's motivation doesn't matter - they might think Lakers will lose, or they
              might be a market maker closing a position. Either way, you're matched.
            </p>
          </div>
        </div>

        <div className="mt-6 grid gap-4 sm:grid-cols-2">
          <ConceptCard
            title="Market Orders"
            description="Trade immediately at the best available price. Guaranteed execution, but not guaranteed price."
            icon={<AlertTriangle className="h-4 w-4" />}
          />
          <ConceptCard
            title="Limit Orders"
            description="Set your own price and wait. Guaranteed price (or better), but not guaranteed execution."
            icon={<Scale className="h-4 w-4" />}
          />
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 7: WHY THIS ALL MATTERS */}
      {/* =================================================================== */}
      <LessonRow
        aside={
          <TipBlock title="Preview of Next Lesson">
            Understanding dual markets is the foundation for market making strategy. In the next
            lesson, you'll learn why professionals target specific positions.
          </TipBlock>
        }
      >
        <SectionHeader
          title="Why This Structure Matters"
          subtitle="Setting up the market making strategy"
        />

        <p className="text-muted-foreground mb-6 leading-relaxed">
          The dual market structure creates specific opportunities that market makers exploit.
          Here's the foundation you need to understand before the next lesson:
        </p>

        <div className="space-y-4">
          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full bg-amber-500/20">
              <span className="font-bold text-amber-400">1</span>
            </div>
            <div>
              <h4 className="font-medium">Recreational bettors have predictable behavior</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                When a fan likes the Lakers, they think "Lakers YES" - not "Celtics NO." This
                creates asymmetric flow patterns.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full bg-amber-500/20">
              <span className="font-bold text-amber-400">2</span>
            </div>
            <div>
              <h4 className="font-medium">Equivalent positions have different characteristics</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                "Lakers YES" and "Celtics NO" might have different prices, spreads, liquidity, and
                queue positions - even though they're economically the same.
              </p>
            </div>
          </div>

          <div className="bg-card flex items-start gap-3 rounded-lg border p-4">
            <div className="flex h-8 w-8 flex-shrink-0 items-center justify-center rounded-full bg-amber-500/20">
              <span className="font-bold text-amber-400">3</span>
            </div>
            <div>
              <h4 className="font-medium">Market makers can position strategically</h4>
              <p className="text-muted-foreground mt-1 text-sm">
                By understanding where retail flow goes, professionals can position in the opposite
                market's NO side to capture that flow with better queue positions.
              </p>
            </div>
          </div>
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 8: KEY TERMS RECAP */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="Key Terms to Remember"
          subtitle="Your order book vocabulary"
        />

        <div className="grid gap-4 sm:grid-cols-2 lg:grid-cols-3">
          <ConceptCard
            title="Order Book"
            description="The list of all buy (bid) and sell (ask) orders, organized by price."
            icon={<BookOpen className="h-4 w-4" />}
          />
          <ConceptCard
            title="Bid / Ask"
            description="Bids are buy offers. Asks are sell offers. Best bid < best ask."
            icon={<ArrowLeftRight className="h-4 w-4" />}
          />
          <ConceptCard
            title="Spread"
            description="The gap between best bid and best ask. Tighter = more liquid."
            icon={<Scale className="h-4 w-4" />}
          />
          <ConceptCard
            title="Queue Position"
            description="Your place in line at a price level. First in, first out."
            icon={<Users className="h-4 w-4" />}
          />
          <ConceptCard
            title="Liquidity"
            description="How easily you can trade without moving the price."
            icon={<GitBranch className="h-4 w-4" />}
          />
          <ConceptCard
            title="Counterparty"
            description="The other side of your trade. Every buy needs a sell."
            icon={<ArrowLeftRight className="h-4 w-4" />}
          />
        </div>
      </LessonRow>

      {/* =================================================================== */}
      {/* SECTION 9: NEXT STEPS */}
      {/* =================================================================== */}
      <LessonRow>
        <SectionHeader
          title="What's Next?"
          subtitle="Applying this knowledge to market making"
        />

        <p className="text-muted-foreground mb-6 leading-relaxed">
          Now that you understand dual markets and order books, you're ready for the key strategy
          lesson: how professional market makers use this structure to their advantage. You'll learn
          why they focus on the NO side, how they capture retail flow, and why queue position can be
          more important than price.
        </p>

        {nextGuide && (
          <NextStepBlock
            href={`/${nextGuide.slug}`}
            title={nextGuide.title}
            description="Learn the strategy that professional traders use: positioning on the NO side to capture predictable retail flow."
            buttonText="Continue to Strategy Lesson"
          />
        )}
      </LessonRow>
    </LessonLayout>
  );
}
