/**
 * Annotated Order Book Widget
 *
 * A teaching widget that mirrors the real OrderBook component
 * but with annotations explaining each part.
 */

import { useState } from 'react';
import { cn } from '@/lib/utils';
import { Info, Target, TrendingUp, TrendingDown } from 'lucide-react';

interface OrderLevel {
  price: number;
  size: number;
  cumulative: number;
  highlight?: 'size' | 'cumulative' | 'price';
}

// Sample data matching realistic market conditions
const sampleBids: OrderLevel[] = [
  { price: 53, size: 57, cumulative: 57 },
  { price: 52, size: 5283, cumulative: 5340 },
  { price: 51, size: 57343, cumulative: 62683 },
  { price: 50, size: 36030, cumulative: 98713 },
  { price: 49, size: 36444, cumulative: 135157 },
  { price: 48, size: 8966, cumulative: 144123 },
  { price: 47, size: 100, cumulative: 144223 },
];

const sampleAsks: OrderLevel[] = [
  { price: 54, size: 421469, cumulative: 421469 },
  { price: 55, size: 507841, cumulative: 929310 },
  { price: 56, size: 63290, cumulative: 992600 },
  { price: 57, size: 310, cumulative: 992910 },
  { price: 58, size: 251, cumulative: 993161 },
  { price: 60, size: 49, cumulative: 993210 },
  { price: 72, size: 22500, cumulative: 1015710 },
];

type AnnotationKey =
  | 'spread'
  | 'bids'
  | 'asks'
  | 'size'
  | 'cumulative'
  | 'depth'
  | 'price'
  | 'last'
  | null;

interface AnnotationInfo {
  title: string;
  description: string;
  color: string;
}

const annotations: Record<NonNullable<AnnotationKey>, AnnotationInfo> = {
  spread: {
    title: 'Spread',
    description:
      'The gap between best bid (53¢) and best ask (54¢). A 1¢ spread is tight, meaning good liquidity. Wider spreads = less liquid markets.',
    color: 'border-amber-500 bg-amber-500/10',
  },
  bids: {
    title: 'Bids (YES Side)',
    description:
      'These are buy orders for YES contracts. Green prices show what buyers are willing to pay. Higher prices = more bullish on the outcome.',
    color: 'border-emerald-500 bg-emerald-500/10',
  },
  asks: {
    title: 'Asks (NO Side)',
    description:
      'These are the other side - sellers of YES (or buyers of NO). Red prices show what sellers want. In prediction markets, buying NO at 54¢ is the same as selling YES at 46¢.',
    color: 'border-rose-500 bg-rose-500/10',
  },
  size: {
    title: 'Size (Quantity)',
    description:
      'How many contracts are available at this price level. Large sizes = strong support/resistance. Small sizes get filled quickly.',
    color: 'border-violet-500 bg-violet-500/10',
  },
  cumulative: {
    title: 'Total (Cumulative)',
    description:
      'Running total of all contracts at this price and better. Shows depth - how much volume would need to trade to reach that price level.',
    color: 'border-blue-500 bg-blue-500/10',
  },
  depth: {
    title: 'Depth Bars',
    description:
      'Visual representation of cumulative volume. Darker shading = more liquidity at that level. The bars show where "walls" of liquidity sit.',
    color: 'border-slate-500 bg-slate-500/10',
  },
  price: {
    title: 'Price Levels',
    description:
      'Prices in cents. YES bids go down from the spread (53→47), NO asks go up (54→72). The best prices are closest to the spread.',
    color: 'border-primary bg-primary/10',
  },
  last: {
    title: 'Last Trade',
    description:
      'The most recent execution price. Compare to the spread: if last (54¢) is at the ask, the last trader was an aggressive buyer.',
    color: 'border-cyan-500 bg-cyan-500/10',
  },
};

export function AnnotatedOrderBook() {
  const [activeAnnotation, setActiveAnnotation] = useState<AnnotationKey>(null);
  const spread = sampleAsks[0].price - sampleBids[0].price;
  const lastPrice = 54;

  const maxBidCumulative = sampleBids[sampleBids.length - 1].cumulative;
  const maxAskCumulative = sampleAsks[sampleAsks.length - 1].cumulative;

  return (
    <div className="space-y-4">
      {/* Controls */}
      <div className="flex flex-wrap gap-2 text-xs">
        <span className="text-muted-foreground py-1">Click to learn:</span>
        {(Object.keys(annotations) as NonNullable<AnnotationKey>[]).map((key) => (
          <button
            key={key}
            onClick={() => setActiveAnnotation(activeAnnotation === key ? null : key)}
            className={cn(
              'rounded border px-2 py-1 transition-colors',
              activeAnnotation === key ? annotations[key].color : 'border-border hover:bg-muted'
            )}
          >
            {annotations[key].title}
          </button>
        ))}
      </div>

      {/* Annotation description */}
      {activeAnnotation && (
        <div className={cn('rounded-lg border-l-4 p-3', annotations[activeAnnotation].color)}>
          <div className="flex items-start gap-2">
            <Info className="mt-0.5 h-4 w-4 flex-shrink-0" />
            <div>
              <div className="text-sm font-semibold">{annotations[activeAnnotation].title}</div>
              <p className="text-muted-foreground mt-1 text-sm">
                {annotations[activeAnnotation].description}
              </p>
            </div>
          </div>
        </div>
      )}

      {/* Order Book - mirrors actual component */}
      <div className="bg-card border-border/50 overflow-hidden rounded-lg border">
        {/* Header */}
        <div
          className={cn(
            'border-border/50 bg-muted/30 flex items-center justify-between border-b px-3 py-2 transition-colors',
            activeAnnotation === 'spread' && 'ring-2 ring-inset ring-amber-500'
          )}
          onClick={() => setActiveAnnotation(activeAnnotation === 'spread' ? null : 'spread')}
        >
          <h3 className="text-sm font-medium">Order Book</h3>
          <span
            className={cn(
              'text-muted-foreground rounded px-2 py-0.5 text-xs',
              activeAnnotation === 'spread' && 'bg-amber-500/20 text-amber-300'
            )}
          >
            Spread: <span className="font-mono">{spread}¢</span>
          </span>
        </div>

        {/* Column headers */}
        <div className="text-muted-foreground border-border/30 grid grid-cols-2 border-b px-2 py-1 text-[10px] uppercase tracking-wider">
          <div
            className={cn(
              '-mx-1 flex cursor-pointer justify-between rounded px-1 transition-colors',
              activeAnnotation === 'bids' && 'bg-emerald-500/20'
            )}
            onClick={() => setActiveAnnotation(activeAnnotation === 'bids' ? null : 'bids')}
          >
            <span className={cn(activeAnnotation === 'price' && 'text-primary')}>Price</span>
            <span className={cn(activeAnnotation === 'size' && 'text-violet-400')}>Size</span>
            <span className={cn(activeAnnotation === 'cumulative' && 'text-blue-400')}>Total</span>
          </div>
          <div
            className={cn(
              '-mx-1 flex cursor-pointer justify-between rounded px-1 transition-colors',
              activeAnnotation === 'asks' && 'bg-rose-500/20'
            )}
            onClick={() => setActiveAnnotation(activeAnnotation === 'asks' ? null : 'asks')}
          >
            <span className={cn(activeAnnotation === 'cumulative' && 'text-blue-400')}>Total</span>
            <span className={cn(activeAnnotation === 'size' && 'text-violet-400')}>Size</span>
            <span className={cn(activeAnnotation === 'price' && 'text-primary')}>Price</span>
          </div>
        </div>

        {/* Book content */}
        <div className="divide-border/30 grid grid-cols-2 divide-x">
          {/* Bids (left) */}
          <div
            className={cn(
              'flex flex-col transition-colors',
              activeAnnotation === 'bids' && 'bg-emerald-500/5'
            )}
          >
            {sampleBids.map((level, i) => {
              const depthPercent = (level.cumulative / maxBidCumulative) * 100;
              return (
                <div
                  key={`bid-${level.price}`}
                  className={cn(
                    'relative flex items-center px-2 py-0.5 font-mono text-sm',
                    i === 0 && activeAnnotation === 'spread' && 'bg-amber-500/10'
                  )}
                >
                  {/* Depth bar */}
                  <div
                    className={cn(
                      'absolute inset-y-0 right-0 bg-green-500/10',
                      activeAnnotation === 'depth' && 'bg-green-500/30'
                    )}
                    style={{ width: `${depthPercent}%` }}
                  />
                  <div className="relative flex w-full items-center justify-between gap-2">
                    <span
                      className={cn(
                        'w-8 text-green-500',
                        activeAnnotation === 'price' && 'text-primary font-bold'
                      )}
                    >
                      {level.price}
                    </span>
                    <span
                      className={cn(
                        'text-muted-foreground flex-1 text-right',
                        activeAnnotation === 'size' && 'font-bold text-violet-400'
                      )}
                    >
                      {level.size.toLocaleString()}
                    </span>
                    <span
                      className={cn(
                        'text-muted-foreground/60 w-16 text-right',
                        activeAnnotation === 'cumulative' && 'font-bold text-blue-400'
                      )}
                    >
                      {level.cumulative.toLocaleString()}
                    </span>
                  </div>
                </div>
              );
            })}
          </div>

          {/* Asks (right) */}
          <div
            className={cn(
              'flex flex-col transition-colors',
              activeAnnotation === 'asks' && 'bg-rose-500/5'
            )}
          >
            {sampleAsks.map((level, i) => {
              const depthPercent = (level.cumulative / maxAskCumulative) * 100;
              return (
                <div
                  key={`ask-${level.price}`}
                  className={cn(
                    'relative flex items-center px-2 py-0.5 font-mono text-sm',
                    i === 0 && activeAnnotation === 'spread' && 'bg-amber-500/10'
                  )}
                >
                  {/* Depth bar */}
                  <div
                    className={cn(
                      'absolute inset-y-0 left-0 bg-red-500/10',
                      activeAnnotation === 'depth' && 'bg-red-500/30'
                    )}
                    style={{ width: `${depthPercent}%` }}
                  />
                  <div className="relative flex w-full items-center justify-between gap-2">
                    <span
                      className={cn(
                        'text-muted-foreground/60 w-16',
                        activeAnnotation === 'cumulative' && 'font-bold text-blue-400'
                      )}
                    >
                      {level.cumulative.toLocaleString()}
                    </span>
                    <span
                      className={cn(
                        'text-muted-foreground flex-1',
                        activeAnnotation === 'size' && 'font-bold text-violet-400'
                      )}
                    >
                      {level.size.toLocaleString()}
                    </span>
                    <span
                      className={cn(
                        'w-8 text-right text-red-500',
                        activeAnnotation === 'price' && 'text-primary font-bold'
                      )}
                    >
                      {level.price}
                    </span>
                  </div>
                </div>
              );
            })}
          </div>
        </div>

        {/* Footer */}
        <div
          className={cn(
            'border-border/50 bg-muted/30 flex cursor-pointer items-center justify-center border-t px-3 py-2 transition-colors',
            activeAnnotation === 'last' && 'bg-cyan-500/10 ring-2 ring-inset ring-cyan-500'
          )}
          onClick={() => setActiveAnnotation(activeAnnotation === 'last' ? null : 'last')}
        >
          <span className="font-mono text-sm">
            Last:{' '}
            <span className={cn('font-bold', activeAnnotation === 'last' && 'text-cyan-300')}>
              {lastPrice}¢
            </span>
          </span>
        </div>
      </div>

      {/* Quick interpretation guide */}
      <div className="grid gap-3 text-sm sm:grid-cols-2">
        <div className="bg-card rounded-lg border p-3">
          <div className="mb-2 flex items-center gap-2 text-emerald-400">
            <TrendingUp className="h-4 w-4" />
            <span className="font-semibold">What Bids Tell You</span>
          </div>
          <ul className="text-muted-foreground space-y-1 text-xs">
            <li>
              • <strong>Thick bids (large sizes)</strong> = buyers willing to support price
            </li>
            <li>
              • <strong>Bids moving up</strong> = increasing bullish sentiment
            </li>
            <li>
              • <strong>Best bid at 53¢</strong> = immediate YES buyer at that price
            </li>
          </ul>
        </div>
        <div className="bg-card rounded-lg border p-3">
          <div className="mb-2 flex items-center gap-2 text-rose-400">
            <TrendingDown className="h-4 w-4" />
            <span className="font-semibold">What Asks Tell You</span>
          </div>
          <ul className="text-muted-foreground space-y-1 text-xs">
            <li>
              • <strong>Thick asks (large sizes)</strong> = sellers holding price down
            </li>
            <li>
              • <strong>421K at 54¢</strong> = massive wall of selling pressure
            </li>
            <li>
              • <strong>Best ask at 54¢</strong> = price to buy YES immediately
            </li>
          </ul>
        </div>
      </div>

      {/* Reading exercise */}
      <div className="from-primary/10 via-primary/5 border-primary/30 rounded-lg border bg-gradient-to-br to-transparent p-4">
        <div className="flex items-start gap-2">
          <Target className="text-primary mt-0.5 h-5 w-5 flex-shrink-0" />
          <div>
            <h4 className="mb-2 font-semibold">Reading This Book</h4>
            <p className="text-muted-foreground text-sm">
              This market shows strong selling pressure. Notice the 421K contracts at 54¢ (ask) vs
              only 57 at 53¢ (bid). The last trade at 54¢ was at the ask, meaning someone paid up to
              buy. But with such a massive ask wall, pushing price higher would require eating
              through half a million contracts. This asymmetry suggests bearish near-term pressure.
            </p>
          </div>
        </div>
      </div>
    </div>
  );
}
