/**
 * Bet Type Visualizer
 *
 * Interactive widget showing how different bet types work.
 */

import { useState } from 'react';
import { cn } from '@/lib/utils';

type BetType = 'moneyline' | 'spread' | 'total';

interface BetTypeVisualizerProps {
  type: BetType;
}

interface ScoreState {
  home: number;
  away: number;
}

const defaultScores: ScoreState = {
  home: 110,
  away: 105,
};

export function BetTypeVisualizer({ type }: BetTypeVisualizerProps) {
  const [scores, setScores] = useState<ScoreState>(defaultScores);
  const spread = 5.5; // Lakers favored by 5.5
  const total = 227.5;

  const adjustScore = (team: 'home' | 'away', delta: number) => {
    setScores((prev) => ({
      ...prev,
      [team]: Math.max(0, prev[team] + delta),
    }));
  };

  const homeWins = scores.home > scores.away;
  const awayWins = scores.away > scores.home;
  const homeMargin = scores.home - scores.away;
  const combinedTotal = scores.home + scores.away;
  const homeCoversSpread = homeMargin > spread;
  const isOver = combinedTotal > total;

  const renderOutcome = () => {
    const outcomes: Record<BetType, { yes: boolean; label: string; explanation: string }> = {
      moneyline: {
        yes: homeWins,
        label: homeWins ? 'Lakers WIN' : awayWins ? 'Celtics WIN' : 'TIE',
        explanation: homeWins
          ? 'Lakers YES pays out $1.00'
          : awayWins
            ? 'Lakers NO (Celtics YES) pays out $1.00'
            : 'Tie - contracts void',
      },
      spread: {
        yes: homeCoversSpread,
        label: homeCoversSpread ? `Lakers cover (-${spread})` : `Lakers don't cover`,
        explanation: homeCoversSpread
          ? `Lakers won by ${homeMargin}, which is MORE than ${spread}. YES wins.`
          : `Lakers won by ${homeMargin}, which is LESS than ${spread}. NO wins.`,
      },
      total: {
        yes: isOver,
        label: isOver ? `OVER ${total}` : `UNDER ${total}`,
        explanation: `Combined score: ${combinedTotal}. ${isOver ? 'Over' : 'Under'} ${total} wins.`,
      },
    };

    const outcome = outcomes[type];
    return (
      <div
        className={cn(
          'rounded-lg p-3 text-center',
          outcome.yes ? 'bg-emerald-500/10 text-emerald-400' : 'bg-rose-500/10 text-rose-400'
        )}
      >
        <div className="font-semibold">{outcome.label}</div>
        <div className="mt-1 text-xs opacity-80">{outcome.explanation}</div>
      </div>
    );
  };

  const getMarketQuestion = () => {
    const questions: Record<BetType, string> = {
      moneyline: 'Will Lakers win?',
      spread: `Will Lakers win by ${spread}+ points?`,
      total: `Will total points be over ${total}?`,
    };
    return questions[type];
  };

  return (
    <div className="bg-card mt-4 rounded-lg border p-4">
      <div className="text-muted-foreground mb-3 text-xs font-semibold uppercase tracking-wider">
        Interactive: {getMarketQuestion()}
      </div>

      {/* Scoreboard */}
      <div className="mb-4 grid grid-cols-3 gap-4">
        {/* Away Team */}
        <div className="text-center">
          <div className="text-muted-foreground mb-1 text-xs">Celtics (Away)</div>
          <div className="flex items-center justify-center gap-2">
            <button
              onClick={() => adjustScore('away', -5)}
              className="bg-muted hover:bg-muted/80 h-6 w-6 rounded text-xs"
            >
              -
            </button>
            <span className="w-12 text-2xl font-bold tabular-nums">{scores.away}</span>
            <button
              onClick={() => adjustScore('away', 5)}
              className="bg-muted hover:bg-muted/80 h-6 w-6 rounded text-xs"
            >
              +
            </button>
          </div>
        </div>

        {/* VS */}
        <div className="flex items-center justify-center">
          <span className="text-muted-foreground text-sm">vs</span>
        </div>

        {/* Home Team */}
        <div className="text-center">
          <div className="text-muted-foreground mb-1 text-xs">Lakers (Home)</div>
          <div className="flex items-center justify-center gap-2">
            <button
              onClick={() => adjustScore('home', -5)}
              className="bg-muted hover:bg-muted/80 h-6 w-6 rounded text-xs"
            >
              -
            </button>
            <span className="w-12 text-2xl font-bold tabular-nums">{scores.home}</span>
            <button
              onClick={() => adjustScore('home', 5)}
              className="bg-muted hover:bg-muted/80 h-6 w-6 rounded text-xs"
            >
              +
            </button>
          </div>
        </div>
      </div>

      {/* Stats */}
      <div className="text-muted-foreground mb-4 grid grid-cols-3 gap-4 text-center text-xs">
        <div>
          <div className="text-foreground font-medium">{combinedTotal}</div>
          <div>Total Points</div>
        </div>
        <div>
          <div className="text-foreground font-medium">
            {homeMargin > 0 ? '+' : ''}
            {homeMargin}
          </div>
          <div>Lakers Margin</div>
        </div>
        <div>
          <div className="text-foreground font-medium">
            {homeWins ? 'Lakers' : awayWins ? 'Celtics' : 'Tie'}
          </div>
          <div>Winner</div>
        </div>
      </div>

      {/* Outcome */}
      {renderOutcome()}

      <div className="text-muted-foreground mt-3 text-center text-xs">
        Adjust the scores to see how the outcome changes
      </div>
    </div>
  );
}
