/**
 * Market Flow Visualizer
 *
 * Animated visualization showing how orders flow and match across markets.
 */

import { useState, useEffect } from 'react';
import { cn } from '@/lib/utils';
import { ArrowRight, Play, RotateCcw, User, Building2 } from 'lucide-react';

type AnimationStep = 0 | 1 | 2 | 3 | 4;

export function MarketFlowVisualizer() {
  const [step, setStep] = useState<AnimationStep>(0);
  const [isPlaying, setIsPlaying] = useState(false);

  useEffect(() => {
    if (!isPlaying) return;

    const timer = setInterval(() => {
      setStep((prev) => {
        if (prev >= 4) {
          setIsPlaying(false);
          return 4;
        }
        return (prev + 1) as AnimationStep;
      });
    }, 1500);

    return () => clearInterval(timer);
  }, [isPlaying]);

  const startAnimation = () => {
    setStep(0);
    setIsPlaying(true);
  };

  const resetAnimation = () => {
    setStep(0);
    setIsPlaying(false);
  };

  const stepDescriptions: Record<AnimationStep, string> = {
    0: 'Market maker posts order on Celtics NO book',
    1: 'Recreational bettor wants to bet on Lakers',
    2: 'Bettor submits Lakers YES order',
    3: 'Kalshi matches the orders across markets',
    4: 'Both traders now have opposite positions',
  };

  return (
    <div className="bg-card mt-4 rounded-lg border p-4">
      <div className="mb-4 flex items-center justify-between">
        <div className="text-muted-foreground text-xs font-semibold uppercase tracking-wider">
          Order Flow Animation
        </div>
        <div className="flex items-center gap-2">
          <button
            onClick={resetAnimation}
            className="hover:bg-muted rounded p-1.5 transition-colors"
            title="Reset"
          >
            <RotateCcw className="h-4 w-4" />
          </button>
          <button
            onClick={startAnimation}
            disabled={isPlaying}
            className={cn(
              'flex items-center gap-1.5 rounded px-3 py-1.5 text-sm font-medium transition-colors',
              isPlaying
                ? 'bg-muted text-muted-foreground cursor-not-allowed'
                : 'bg-primary text-primary-foreground hover:bg-primary/90'
            )}
          >
            <Play className="h-3 w-3" />
            {isPlaying ? 'Playing...' : 'Play'}
          </button>
        </div>
      </div>

      {/* Visualization */}
      <div className="bg-muted/30 relative min-h-[200px] rounded-lg p-4">
        {/* Market Maker */}
        <div
          className={cn(
            'absolute left-4 top-1/2 -translate-y-1/2 transition-all duration-500',
            step >= 0 ? 'opacity-100' : 'opacity-0'
          )}
        >
          <div className="flex flex-col items-center gap-2">
            <div
              className={cn(
                'flex h-12 w-12 items-center justify-center rounded-full transition-colors',
                step >= 3 ? 'bg-emerald-500/20' : 'bg-muted'
              )}
            >
              <Building2 className="h-6 w-6 text-emerald-400" />
            </div>
            <span className="text-muted-foreground text-xs">Market Maker</span>
          </div>
        </div>

        {/* Celtics NO Order Book */}
        <div
          className={cn(
            'absolute left-24 top-8 transition-all duration-500',
            step >= 0 ? 'scale-100 opacity-100' : 'scale-95 opacity-0'
          )}
        >
          <div
            className={cn(
              'rounded border p-2 text-xs transition-colors',
              step >= 3 ? 'border-emerald-500/50 bg-emerald-500/10' : 'bg-card'
            )}
          >
            <div className="mb-1 font-medium">Celtics NO Book</div>
            <div className={cn('transition-colors', step >= 0 && step < 3 && 'text-primary')}>
              Bid: $0.35 × 100
              {step >= 0 && step < 3 && <span className="ml-1">(MM)</span>}
            </div>
          </div>
        </div>

        {/* Arrow: MM to Book */}
        <div
          className={cn(
            'absolute left-16 top-[60px] transition-opacity duration-300',
            step === 0 ? 'opacity-100' : 'opacity-30'
          )}
        >
          <ArrowRight className="h-4 w-4 text-emerald-400" />
        </div>

        {/* Recreational Bettor */}
        <div
          className={cn(
            'absolute right-4 top-1/2 -translate-y-1/2 transition-all duration-500',
            step >= 1 ? 'opacity-100' : 'opacity-0'
          )}
        >
          <div className="flex flex-col items-center gap-2">
            <div
              className={cn(
                'flex h-12 w-12 items-center justify-center rounded-full transition-colors',
                step >= 3 ? 'bg-rose-500/20' : 'bg-muted'
              )}
            >
              <User className="h-6 w-6 text-rose-400" />
            </div>
            <span className="text-muted-foreground text-xs">Bettor</span>
          </div>
        </div>

        {/* Lakers YES Order Book */}
        <div
          className={cn(
            'absolute bottom-8 right-24 transition-all duration-500',
            step >= 2 ? 'scale-100 opacity-100' : 'scale-95 opacity-0'
          )}
        >
          <div
            className={cn(
              'rounded border p-2 text-xs transition-colors',
              step >= 3 ? 'border-rose-500/50 bg-rose-500/10' : 'bg-card'
            )}
          >
            <div className="mb-1 font-medium">Lakers YES Book</div>
            <div className={cn('transition-colors', step >= 2 && step < 3 && 'text-rose-400')}>
              Buy: $0.65 × 100
              {step >= 2 && step < 3 && <span className="ml-1">(bettor)</span>}
            </div>
          </div>
        </div>

        {/* Arrow: Bettor to Book */}
        <div
          className={cn(
            'absolute bottom-[60px] right-16 transition-opacity duration-300',
            step === 2 ? 'opacity-100' : 'opacity-30'
          )}
        >
          <ArrowRight className="h-4 w-4 rotate-180 text-rose-400" />
        </div>

        {/* Match indicator */}
        {step >= 3 && (
          <div className="absolute inset-0 flex items-center justify-center">
            <div
              className={cn(
                'bg-primary/20 border-primary/50 rounded-lg border px-4 py-2 text-sm font-medium',
                'animate-in fade-in zoom-in-95 duration-300'
              )}
            >
              <span className="text-primary">MATCHED!</span>
            </div>
          </div>
        )}

        {/* Connecting line */}
        {step >= 3 && (
          <svg className="pointer-events-none absolute inset-0 h-full w-full">
            <line
              x1="80"
              y1="60"
              x2="calc(100% - 80px)"
              y2="calc(100% - 60px)"
              stroke="currentColor"
              strokeWidth="2"
              strokeDasharray="4 4"
              className="text-primary/50 animate-in fade-in duration-500"
            />
          </svg>
        )}
      </div>

      {/* Step description */}
      <div className="bg-muted/50 mt-4 rounded p-3">
        <div className="flex items-center gap-2">
          <span className="bg-primary/20 text-primary flex h-6 w-6 items-center justify-center rounded-full text-xs font-bold">
            {step + 1}
          </span>
          <span className="text-sm">{stepDescriptions[step]}</span>
        </div>
      </div>

      {/* Step indicators */}
      <div className="mt-3 flex items-center justify-center gap-2">
        {[0, 1, 2, 3, 4].map((s) => (
          <button
            key={s}
            onClick={() => {
              setStep(s as AnimationStep);
              setIsPlaying(false);
            }}
            className={cn(
              'h-2 w-2 rounded-full transition-colors',
              step >= s ? 'bg-primary' : 'bg-muted'
            )}
          />
        ))}
      </div>
    </div>
  );
}
