/**
 * Trading Glossary
 *
 * Comprehensive glossary of trading and sports betting terms
 * used throughout the guides.
 */

export interface GlossaryEntry {
  term: string;
  definition: string;
  category: string;
  etymology?: string;
  seeAlso?: string[];
  usedIn?: string[]; // slugs of guides that use this term
}

export const tradingGlossary: Record<string, GlossaryEntry> = {
  // ==========================================================================
  // CONTRACTS & MARKETS
  // ==========================================================================
  contract: {
    term: 'Contract',
    definition:
      "The basic unit you trade on Kalshi. Each contract pays $1.00 if the stated outcome happens, $0.00 if it doesn't. Contracts are binary - only two possible results.",
    category: 'Contracts',
    etymology:
      'From Latin "contractus" (agreement). In trading, a contract represents a standardized agreement between buyer and seller about terms of exchange.',
    usedIn: ['what-is-kalshi', 'yes-no-markets'],
  },
  yes: {
    term: 'YES Contract',
    definition:
      "A contract that pays $1.00 if the stated event DOES happen. If you buy YES at 60¢ and the event happens, you profit 40¢. If it doesn't happen, you lose your 60¢.",
    category: 'Contracts',
    seeAlso: ['no', 'contract'],
    usedIn: ['what-is-kalshi', 'yes-no-markets'],
  },
  no: {
    term: 'NO Contract',
    definition:
      'A contract that pays $1.00 if the stated event does NOT happen. Buying NO is mathematically equivalent to selling YES. If YES trades at 60¢, NO effectively trades at 40¢.',
    category: 'Contracts',
    seeAlso: ['yes', 'contract'],
    usedIn: ['what-is-kalshi', 'yes-no-markets'],
  },
  predictionMarket: {
    term: 'Prediction Market',
    definition:
      'A market where people trade contracts based on the outcomes of future events. Prices reflect the collective probability estimate of participants. Kalshi is a regulated prediction market.',
    category: 'Markets',
    etymology:
      'Modern prediction markets evolved from "information markets" in the 1980s. The Iowa Electronic Markets (1988) pioneered using markets to forecast election outcomes.',
    usedIn: ['what-is-kalshi'],
  },
  exchange: {
    term: 'Exchange',
    definition:
      'A marketplace where buyers and sellers trade directly with each other through an order book. Unlike a sportsbook where you bet against the house, on an exchange you trade against other users. Kalshi is an exchange.',
    category: 'Markets',
    etymology:
      'From Old French "eschange" (trade). Stock exchanges date to the 1600s (Amsterdam). Prediction market exchanges are a 21st-century development.',
    seeAlso: ['orderBook', 'sportsbook'],
    usedIn: ['what-is-kalshi'],
  },
  sportsbook: {
    term: 'Sportsbook',
    definition:
      'A company that takes bets on sports, setting odds and acting as your counterparty. You bet against the house, not other bettors. DraftKings, FanDuel, and BetMGM are sportsbooks. Kalshi is NOT a sportsbook.',
    category: 'Markets',
    etymology:
      'From "sports" + "book" (as in a bookmaker\'s record of bets). Term dates to early 20th century American gambling.',
    seeAlso: ['exchange', 'bookmaker'],
    usedIn: ['what-is-kalshi'],
  },

  // ==========================================================================
  // ORDER BOOK & TRADING
  // ==========================================================================
  orderBook: {
    term: 'Order Book',
    definition:
      'A real-time list of all buy and sell orders for a market, organized by price. Shows the bids (what buyers will pay) and asks (what sellers want). The order book reveals supply and demand at each price level.',
    category: 'Order Book',
    etymology:
      'Originally a physical ledger where exchanges recorded orders. Electronic order books emerged in the 1970s-80s with computerized trading.',
    seeAlso: ['bid', 'ask', 'depth'],
    usedIn: ['what-is-kalshi', 'yes-no-markets', 'market-making-strategy'],
  },
  bid: {
    term: 'Bid',
    definition:
      'The highest price a buyer is currently willing to pay. If the best bid is 55¢, that means someone will buy immediately at 55¢. Multiple bids can exist at different prices, forming the "bid side" of the order book.',
    category: 'Order Book',
    etymology:
      'From Old English "biddan" (to offer, ask). Used in auctions since medieval times to denote a buyer\'s offer.',
    seeAlso: ['ask', 'spread', 'orderBook'],
    usedIn: ['what-is-kalshi', 'yes-no-markets'],
  },
  ask: {
    term: 'Ask (Offer)',
    definition:
      'The lowest price a seller is currently willing to accept. If the best ask is 58¢, that means someone will sell immediately at 58¢. Also called the "offer" price.',
    category: 'Order Book',
    etymology:
      'From Old English "ascian" (to request). The term "offer" comes from Latin "offerre" (to present).',
    seeAlso: ['bid', 'spread', 'orderBook'],
    usedIn: ['what-is-kalshi', 'yes-no-markets'],
  },
  spread: {
    term: 'Spread',
    definition:
      'The difference between the best bid and best ask price. A 55¢ bid / 58¢ ask has a 3¢ spread. Tighter spreads indicate more liquidity. Market makers profit by capturing the spread.',
    category: 'Order Book',
    etymology:
      'From "spread" as in the gap or distance. Financial usage dates to the 19th century securities markets.',
    seeAlso: ['bid', 'ask', 'liquidity'],
    usedIn: ['what-is-kalshi', 'market-making-strategy'],
  },
  depth: {
    term: 'Market Depth',
    definition:
      'The volume of orders at each price level in the order book. Shows how much can be traded before moving the price. A market with 10,000 contracts on the bid has more depth than one with 100.',
    category: 'Order Book',
    seeAlso: ['orderBook', 'liquidity'],
    usedIn: ['yes-no-markets'],
  },
  queuePosition: {
    term: 'Queue Position',
    definition:
      'Your place in line when multiple orders exist at the same price. Orders fill in the order they were placed (FIFO - first in, first out). A good queue position means getting filled before others at the same price.',
    category: 'Order Book',
    etymology:
      'From French "queue" (tail, line). In trading, the queue is the line of orders waiting to be executed.',
    usedIn: ['market-making-strategy'],
  },
  limitOrder: {
    term: 'Limit Order',
    definition:
      "An order to buy or sell at a specific price or better. A limit buy at 55¢ will only fill at 55¢ or lower. Limit orders may not fill immediately if the market price doesn't reach your limit.",
    category: 'Orders',
    seeAlso: ['marketOrder'],
    usedIn: ['what-is-kalshi', 'market-making-strategy'],
  },
  marketOrder: {
    term: 'Market Order',
    definition:
      'An order that fills immediately at the best available price. Guarantees execution but not price. In a thin market, large market orders can cause significant slippage.',
    category: 'Orders',
    seeAlso: ['limitOrder', 'slippage'],
    usedIn: ['what-is-kalshi'],
  },
  fill: {
    term: 'Fill',
    definition:
      'When your order executes. "Filled at 55¢" means your order completed at that price. Partial fills occur when only some of your order executes.',
    category: 'Orders',
    usedIn: ['market-making-strategy'],
  },
  liquidity: {
    term: 'Liquidity',
    definition:
      'How easily you can buy or sell without significantly moving the price. High liquidity = many orders, tight spreads, large depth. Low liquidity = few orders, wide spreads, easy to move the price.',
    category: 'Markets',
    etymology:
      'From Latin "liquidus" (fluid). A liquid asset can be quickly converted, like water flowing easily.',
    seeAlso: ['spread', 'depth'],
    usedIn: ['what-is-kalshi', 'yes-no-markets', 'market-making-strategy'],
  },

  // ==========================================================================
  // SPORTS BETTING
  // ==========================================================================
  moneyline: {
    term: 'Moneyline',
    definition:
      'A bet on which team will win the game, regardless of score. No point spread involved. The term comes from the "line" (odds) that bookmakers set, determining how much "money" you need to risk.',
    category: 'Bet Types',
    etymology:
      'American term from the early 20th century. The "line" was literally a line of odds written on a chalkboard. "Money" distinguished it from point spread betting.',
    seeAlso: ['pointSpread', 'total'],
    usedIn: ['sports-betting-101', 'yes-no-markets', 'market-making-strategy'],
  },
  pointSpread: {
    term: 'Point Spread (Spread)',
    definition:
      'A handicap given to the underdog to level the playing field. "Lakers -5.5" means Lakers must win by 6+ points to "cover." Created to make betting interesting even when one team is heavily favored.',
    category: 'Bet Types',
    etymology:
      'Invented by Charles K. McNeil, a Chicago math teacher turned bookmaker, in the 1940s. He realized handicapping would balance action and create more interesting bets.',
    seeAlso: ['moneyline', 'cover'],
    usedIn: ['sports-betting-101'],
  },
  total: {
    term: 'Total (Over/Under)',
    definition:
      'A bet on whether the combined score of both teams will be over or under a set number. "O/U 227.5" means you bet whether the total will be 228+ (over) or 227 or less (under).',
    category: 'Bet Types',
    etymology:
      'Emerged as bettors wanted action on games with obvious winners. Rather than betting on who wins, you bet on the nature of the game (high-scoring vs defensive).',
    seeAlso: ['moneyline', 'pointSpread'],
    usedIn: ['sports-betting-101'],
  },
  cover: {
    term: 'Cover (the Spread)',
    definition:
      'When a team beats the point spread. If Lakers are -5.5 and win by 7, they "covered." If they win by 4, they didn\'t cover (bettors who took Lakers lose).',
    category: 'Bet Types',
    seeAlso: ['pointSpread'],
    usedIn: ['sports-betting-101'],
  },
  favorite: {
    term: 'Favorite',
    definition:
      'The team expected to win. On a moneyline, favorites cost more (you risk more to win less). On a spread, favorites "give" points.',
    category: 'Bet Types',
    seeAlso: ['underdog'],
    usedIn: ['sports-betting-101'],
  },
  underdog: {
    term: 'Underdog',
    definition:
      'The team expected to lose. On a moneyline, underdogs pay more (you risk less to win more). On a spread, underdogs "get" points.',
    category: 'Bet Types',
    etymology:
      'From dog fighting - the losing dog was often underneath. In betting, the expected loser.',
    seeAlso: ['favorite'],
    usedIn: ['sports-betting-101'],
  },
  juice: {
    term: 'Juice (Vig/Vigorish)',
    definition:
      'The fee built into odds that gives the house its edge. In traditional sportsbooks, standard juice is -110 on both sides (bet $110 to win $100). On Kalshi, fees are explicit per-trade commissions.',
    category: 'Fees',
    etymology:
      'From Yiddish "vig" or Russian "vyigrysh" (winnings). The house\'s "cut" of every bet.',
    seeAlso: ['edge'],
    usedIn: ['what-is-kalshi'],
  },

  // ==========================================================================
  // MARKET MAKING & STRATEGY
  // ==========================================================================
  marketMaker: {
    term: 'Market Maker',
    definition:
      "A trader who provides liquidity by posting both buy and sell orders, profiting from the spread between them. Market makers don't bet on outcomes - they profit by facilitating trades for others.",
    category: 'Strategy',
    etymology:
      'Term emerged with electronic trading in the 1980s-90s. Originally called "specialists" on NYSE.',
    seeAlso: ['marketTaker', 'spread', 'liquidity'],
    usedIn: ['market-making-strategy'],
  },
  marketTaker: {
    term: 'Market Taker',
    definition:
      'A trader who removes liquidity by filling existing orders. Takers want immediate execution and "take" the prices market makers offer. Recreational bettors are typically takers.',
    category: 'Strategy',
    seeAlso: ['marketMaker'],
    usedIn: ['market-making-strategy'],
  },
  retailFlow: {
    term: 'Retail Flow',
    definition:
      'Order flow from recreational (non-professional) traders. Retail bettors are predictable - they typically bet YES on teams they like. Market makers position to capture this flow.',
    category: 'Strategy',
    usedIn: ['market-making-strategy'],
  },
  counterparty: {
    term: 'Counterparty',
    definition:
      'The other side of your trade. When you buy, someone sells to you - that seller is your counterparty. On exchanges, counterparties are matched automatically.',
    category: 'Trading',
    usedIn: ['yes-no-markets', 'market-making-strategy'],
  },
  edge: {
    term: 'Edge',
    definition:
      'Your advantage over the market. If you believe true probability is 55% but can buy at a price implying 50%, you have a 5% edge. Positive edge = profitable over time.',
    category: 'Strategy',
    usedIn: ['market-making-strategy'],
  },
  ev: {
    term: 'EV (Expected Value)',
    definition:
      'The average outcome of a bet if repeated many times. Calculated as (probability × win amount) - ((1 - probability) × loss amount). Positive EV (+EV) bets are profitable long-term.',
    category: 'Strategy',
    usedIn: ['market-making-strategy'],
  },

  // ==========================================================================
  // PORTFOLIO & PERFORMANCE
  // ==========================================================================
  position: {
    term: 'Position',
    definition:
      'Your current holdings in a market. "Long Lakers YES" means you own Lakers YES contracts. Positions can be long (you own) or short (you owe).',
    category: 'Portfolio',
    usedIn: ['what-is-kalshi'],
  },
  pnl: {
    term: 'P&L (Profit & Loss)',
    definition:
      'Your profit or loss. Realized P&L is from closed positions. Unrealized P&L is the paper gain/loss on open positions based on current prices.',
    category: 'Portfolio',
    usedIn: ['what-is-kalshi'],
  },
  bankroll: {
    term: 'Bankroll',
    definition:
      'The total amount of money you have set aside for trading/betting. Proper bankroll management (never risking too much on one bet) is crucial for long-term survival.',
    category: 'Portfolio',
    usedIn: ['what-is-kalshi'],
  },
  roi: {
    term: 'ROI (Return on Investment)',
    definition:
      'Profit divided by amount risked, expressed as a percentage. 10% ROI means $10 profit per $100 bet over time. Sustainable ROI in sports betting is typically 2-5%.',
    category: 'Portfolio',
  },
  slippage: {
    term: 'Slippage',
    definition:
      'The difference between expected and actual fill price. Occurs when the market moves before your order fills, or when your order is large enough to "walk" the book through multiple price levels.',
    category: 'Orders',
    seeAlso: ['marketOrder', 'liquidity'],
  },
};

// Helper functions
export function getTermsByCategory(category: string): GlossaryEntry[] {
  return Object.values(tradingGlossary).filter((t) => t.category === category);
}

export function getAllCategories(): string[] {
  const categories = new Set<string>();
  Object.values(tradingGlossary).forEach((t) => categories.add(t.category));
  return Array.from(categories).sort();
}

export function getTermsUsedInGuide(slug: string): GlossaryEntry[] {
  return Object.values(tradingGlossary).filter((t) => t.usedIn?.includes(slug));
}

export function getTerm(key: string): GlossaryEntry | undefined {
  return tradingGlossary[key];
}
