/**
 * Guide Registry
 *
 * All guides are defined here for discovery and navigation.
 */

import type { Guide, GuideCategory } from './types';

export * from './types';

const guides: Record<string, Guide> = {
  // === FUNDAMENTALS (1-5) ===
  'what-is-kalshi': {
    slug: 'what-is-kalshi',
    title: 'What is Kalshi?',
    description: 'Introduction to prediction markets and how Kalshi works',
    category: 'fundamentals',
    duration: '5 min',
    order: 1,
    nextGuide: 'moneyline',
  },
  moneyline: {
    slug: 'moneyline',
    title: 'Understanding Moneylines',
    description: 'What moneylines really mean, where the term comes from, and how to read them',
    category: 'fundamentals',
    duration: '20 min',
    order: 2,
    prerequisites: ['what-is-kalshi'],
    nextGuide: 'sports-betting-101',
  },
  'sports-betting-101': {
    slug: 'sports-betting-101',
    title: 'Sports Betting 101',
    description: 'Learn the fundamentals: moneylines, spreads, and totals',
    category: 'sports-betting',
    duration: '10 min',
    order: 3,
    prerequisites: ['moneyline'],
    nextGuide: 'reading-odds-screen',
  },
  'reading-odds-screen': {
    slug: 'reading-odds-screen',
    title: 'Reading a Sportsbook Odds Screen',
    description:
      'Learn to decode the numbers, columns, and formatting used by professional sports bettors',
    category: 'sports-betting',
    duration: '20 min',
    order: 3.5,
    prerequisites: ['sports-betting-101'],
    nextGuide: 'yes-no-markets',
  },
  'yes-no-markets': {
    slug: 'yes-no-markets',
    title: 'YES/NO Markets Explained',
    description: "Understanding Kalshi's dual market structure for sports",
    category: 'sports-betting',
    duration: '8 min',
    order: 4,
    prerequisites: ['sports-betting-101'],
    nextGuide: 'buy-sell-yes-no',
  },
  'buy-sell-yes-no': {
    slug: 'buy-sell-yes-no',
    title: 'Understanding Buy/Sell and YES/NO',
    description: 'The two independent dimensions of every Kalshi order and how they work together',
    category: 'fundamentals',
    duration: '15 min',
    order: 5,
    prerequisites: ['yes-no-markets'],
    nextGuide: 'probability-expected-value',
  },
  'probability-expected-value': {
    slug: 'probability-expected-value',
    title: 'Probability & Expected Value',
    description: 'The mathematical foundation of profitable trading',
    category: 'fundamentals',
    duration: '12 min',
    order: 6,
    prerequisites: ['buy-sell-yes-no'],
    nextGuide: 'market-making-strategy',
  },

  // === STRATEGIES (7-11) ===
  'market-making-strategy': {
    slug: 'market-making-strategy',
    title: 'Market Making Strategy',
    description: 'How professional traders provide liquidity on the NO side',
    category: 'strategies',
    duration: '15 min',
    order: 7,
    prerequisites: ['probability-expected-value'],
    nextGuide: 'risk-management',
  },
  'risk-management': {
    slug: 'risk-management',
    title: 'Risk Management & Bankroll',
    description: 'Protect your capital with position sizing and risk control',
    category: 'strategies',
    duration: '12 min',
    order: 8,
    prerequisites: ['market-making-strategy'],
    nextGuide: 'variance-luck',
  },
  'variance-luck': {
    slug: 'variance-luck',
    title: 'Variance, Luck & Sample Size',
    description: 'Understanding why winning strategies lose money (sometimes)',
    category: 'strategies',
    duration: '10 min',
    order: 9,
    prerequisites: ['risk-management'],
    nextGuide: 'reading-order-flow',
  },
  'reading-order-flow': {
    slug: 'reading-order-flow',
    title: 'Reading Order Flow',
    description: 'Understanding what the order book tells you about market sentiment',
    category: 'strategies',
    duration: '10 min',
    order: 10,
    prerequisites: ['variance-luck'],
    nextGuide: 'fees-and-costs',
  },
  'fees-and-costs': {
    slug: 'fees-and-costs',
    title: 'Kalshi Fees & Costs',
    description: 'Understanding the true cost of trading and minimizing fees',
    category: 'fundamentals',
    duration: '8 min',
    order: 11,
    prerequisites: ['reading-order-flow'],
  },

  // === PRODUCT & UX (for software builders) ===
  'building-for-market-makers': {
    slug: 'building-for-market-makers',
    title: 'Building for Market Makers',
    description: 'Understanding market maker users to build the right trading software',
    category: 'product',
    duration: '20 min',
    order: 50,
    nextGuide: 'mm-trade-identification',
  },
  'mm-trade-identification': {
    slug: 'mm-trade-identification',
    title: 'How MMs Identify Trades',
    description: 'Learn the decision framework market makers use to evaluate opportunities',
    category: 'product',
    duration: '15 min',
    order: 51,
    prerequisites: ['building-for-market-makers'],
    nextGuide: 'mm-practice-quiz',
  },
  'mm-practice-quiz': {
    slug: 'mm-practice-quiz',
    title: 'MM Trade Practice Quiz',
    description: 'Practice identifying market making opportunities with realistic scenarios',
    category: 'product',
    duration: '10 min',
    order: 52,
    prerequisites: ['mm-trade-identification'],
    nextGuide: 'mm-scenarios',
  },
  'mm-scenarios': {
    slug: 'mm-scenarios',
    title: 'Day-in-the-Life Scenarios',
    description: 'Walk through realistic market maker scenarios to understand UX needs',
    category: 'product',
    duration: '12 min',
    order: 53,
    prerequisites: ['mm-practice-quiz'],
    nextGuide: 'user-interview-guide',
  },
  'user-interview-guide': {
    slug: 'user-interview-guide',
    title: 'User Interview Guide',
    description: 'Questions and frameworks for discovering what traders really need',
    category: 'product',
    duration: '10 min',
    order: 54,
    prerequisites: ['mm-scenarios'],
  },

  // === DEVELOPER ===
  'kalshi-api-entities': {
    slug: 'kalshi-api-entities',
    title: 'Kalshi API Entity Model',
    description: 'Understanding every entity in the Kalshi API and how they relate',
    category: 'developer',
    duration: '25 min',
    order: 60,
    nextGuide: 'kalshi-api-explorer',
  },
  'kalshi-api-explorer': {
    slug: 'kalshi-api-explorer',
    title: 'Kalshi API Explorer',
    description: 'Interactive reference for all Kalshi REST and WebSocket API endpoints',
    category: 'developer',
    duration: 'Reference',
    order: 61,
    prerequisites: ['kalshi-api-entities'],
  },

  // === REFERENCE ===
  glossary: {
    slug: 'glossary',
    title: 'Glossary',
    description: 'Complete reference of trading and betting terminology',
    category: 'fundamentals',
    duration: 'Reference',
    order: 100, // Always last
  },
};

/**
 * Get all guides as an array, sorted by order
 */
export function getAllGuides(): Guide[] {
  return Object.values(guides).sort((a, b) => a.order - b.order);
}

/**
 * Get a single guide by slug
 */
export function getGuide(slug: string): Guide | undefined {
  return guides[slug];
}

/**
 * Get guides filtered by category
 */
export function getGuidesByCategory(category: GuideCategory): Guide[] {
  return getAllGuides().filter((g) => g.category === category);
}

/**
 * Get the next guide in sequence
 */
export function getNextGuide(currentSlug: string): Guide | undefined {
  const current = guides[currentSlug];
  if (!current?.nextGuide) return undefined;
  return guides[current.nextGuide];
}

/**
 * Get prerequisite guides
 */
export function getPrerequisites(slug: string): Guide[] {
  const guide = guides[slug];
  if (!guide?.prerequisites) return [];
  return guide.prerequisites.map((s) => guides[s]).filter(Boolean) as Guide[];
}
