/**
 * Guides Layout
 *
 * Layout wrapper for the guides section with sidebar navigation.
 * This is a standalone layout that doesn't require authentication.
 */

import { ReactNode } from 'react';
import { Link, useParams } from 'react-router-dom';
import { BookOpen, ChevronRight } from 'lucide-react';
import { getAllGuides, categoryLabels, type Guide, type GuideCategory } from '@/data/guides';
import { cn } from '@/lib/utils';

interface GuidesLayoutProps {
  children: ReactNode;
}

export function GuidesLayout({ children }: GuidesLayoutProps) {
  const { slug } = useParams<{ slug: string }>();
  const guides = getAllGuides();

  // Group guides by category
  const groupedGuides = guides.reduce(
    (acc, guide) => {
      if (!acc[guide.category]) {
        acc[guide.category] = [];
      }
      acc[guide.category].push(guide);
      return acc;
    },
    {} as Record<GuideCategory, Guide[]>
  );

  return (
    <div className="bg-background flex min-h-screen">
      {/* Sidebar */}
      <aside className="border-border bg-card flex w-72 flex-shrink-0 flex-col border-r">
        {/* Header */}
        <div className="border-border flex h-14 items-center justify-between border-b px-4">
          <Link
            to="/"
            className="flex items-center gap-2 font-semibold"
          >
            <BookOpen className="text-primary h-5 w-5" />
            <span>Trading Guides</span>
          </Link>
        </div>

        {/* Navigation */}
        <nav className="flex-1 overflow-y-auto p-3">
          {Object.entries(groupedGuides).map(([category, categoryGuides]) => (
            <div
              key={category}
              className="mb-6"
            >
              <h3 className="text-muted-foreground mb-2 px-2 text-xs font-semibold uppercase tracking-wider">
                {categoryLabels[category as GuideCategory]}
              </h3>
              <div className="space-y-1">
                {categoryGuides.map((guide) => {
                  const isActive = slug === guide.slug;
                  return (
                    <Link
                      key={guide.slug}
                      to={`/${guide.slug}`}
                      className={cn(
                        'flex items-center justify-between rounded-md px-2 py-2 text-sm transition-colors',
                        isActive
                          ? 'bg-primary/10 text-foreground'
                          : 'text-muted-foreground hover:bg-muted hover:text-foreground'
                      )}
                    >
                      <span className="truncate">{guide.title}</span>
                      <ChevronRight
                        className={cn(
                          'h-4 w-4 flex-shrink-0',
                          isActive ? 'opacity-100' : 'opacity-0'
                        )}
                      />
                    </Link>
                  );
                })}
              </div>
            </div>
          ))}
        </nav>

        {/* Footer */}
        <div className="border-border border-t p-4">
          <div className="text-muted-foreground text-xs">{guides.length} guides available</div>
        </div>
      </aside>

      {/* Main Content */}
      <main className="flex-1 overflow-y-auto">
        <div className="mx-auto max-w-4xl p-8">{children}</div>
      </main>
    </div>
  );
}

/**
 * Guides Index Page
 *
 * Landing page showing all available guides.
 */
export function GuidesIndexPage() {
  const guides = getAllGuides();

  // Group guides by category
  const groupedGuides = guides.reduce(
    (acc, guide) => {
      if (!acc[guide.category]) {
        acc[guide.category] = [];
      }
      acc[guide.category].push(guide);
      return acc;
    },
    {} as Record<GuideCategory, Guide[]>
  );

  return (
    <div className="space-y-8">
      <div>
        <h1 className="text-3xl font-bold">Trading Guides</h1>
        <p className="text-muted-foreground mt-2">
          Learn everything you need to know about prediction markets, sports betting, and market
          making strategies.
        </p>
      </div>

      {/* Quick Start */}
      <div className="bg-card rounded-lg border p-6">
        <h2 className="mb-2 text-lg font-semibold">New to Kalshi?</h2>
        <p className="text-muted-foreground mb-4 text-sm">
          Start with the fundamentals and work your way up to advanced strategies.
        </p>
        <Link
          to="/what-is-kalshi"
          className="bg-primary text-primary-foreground hover:bg-primary/90 inline-flex items-center justify-center rounded-md px-4 py-2 text-sm font-medium transition-colors"
        >
          Start Learning
          <ChevronRight className="ml-1 h-4 w-4" />
        </Link>
      </div>

      {/* All Guides by Category */}
      {Object.entries(groupedGuides).map(([category, categoryGuides]) => (
        <section key={category}>
          <h2 className="mb-4 text-lg font-semibold">
            {categoryLabels[category as GuideCategory]}
          </h2>
          <div className="grid gap-4 md:grid-cols-2">
            {categoryGuides.map((guide) => (
              <Link
                key={guide.slug}
                to={`/${guide.slug}`}
                className="bg-card hover:border-primary/50 block rounded-lg border p-4 transition-colors"
              >
                <div className="flex items-start justify-between">
                  <div>
                    <h3 className="font-medium">{guide.title}</h3>
                    <p className="text-muted-foreground mt-1 text-sm">{guide.description}</p>
                  </div>
                  <span className="text-muted-foreground ml-4 flex-shrink-0 text-xs">
                    {guide.duration}
                  </span>
                </div>
              </Link>
            ))}
          </div>
        </section>
      ))}
    </div>
  );
}
