/**
 * Configuration Management
 *
 * Loads and validates server configuration from environment variables.
 */
/**
 * Loads and validates server configuration from environment variables.
 *
 * @returns ServerConfig with validated values
 * @throws Error if required config is missing or invalid
 */
export function loadConfig() {
    const port = parseInt(process.env.PORT || '8787', 10);
    if (isNaN(port) || port < 1 || port > 65535) {
        throw new Error('Invalid PORT: must be a number between 1 and 65535');
    }
    const nodeEnv = (process.env.NODE_ENV || 'development');
    if (nodeEnv !== 'development' && nodeEnv !== 'production') {
        throw new Error('Invalid NODE_ENV: must be "development" or "production"');
    }
    // CORS origin: '*' or specific origin in dev, specific origin(s) in prod
    let corsOrigin;
    const corsOriginEnv = process.env.CORS_ORIGIN;
    if (corsOriginEnv === '*' || corsOriginEnv === undefined) {
        corsOrigin = nodeEnv === 'development' ? true : '*';
    }
    else {
        corsOrigin = corsOriginEnv;
    }
    const httpTimeoutMs = parseInt(process.env.HTTP_TIMEOUT_MS || '30000', 10);
    if (isNaN(httpTimeoutMs) || httpTimeoutMs < 1000) {
        throw new Error('Invalid HTTP_TIMEOUT_MS: must be at least 1000ms');
    }
    const wsReconnectDelayMs = parseInt(process.env.WS_RECONNECT_DELAY_MS || '1000', 10);
    if (isNaN(wsReconnectDelayMs) || wsReconnectDelayMs < 100) {
        throw new Error('Invalid WS_RECONNECT_DELAY_MS: must be at least 100ms');
    }
    const wsMaxReconnectAttempts = parseInt(process.env.WS_MAX_RECONNECT_ATTEMPTS || '5', 10);
    if (isNaN(wsMaxReconnectAttempts) || wsMaxReconnectAttempts < 0) {
        throw new Error('Invalid WS_MAX_RECONNECT_ATTEMPTS: must be >= 0');
    }
    const requestSizeLimit = parseInt(process.env.REQUEST_SIZE_LIMIT || '10485760', 10);
    if (isNaN(requestSizeLimit) || requestSizeLimit < 1024) {
        throw new Error('Invalid REQUEST_SIZE_LIMIT: must be at least 1024 bytes');
    }
    const kalshiBaseUrl = process.env.KALSHI_BASE_URL || 'https://api.elections.kalshi.com';
    if (!kalshiBaseUrl.startsWith('https://')) {
        throw new Error('Invalid KALSHI_BASE_URL: must start with https://');
    }
    const smartRelayEnabled = process.env.SMART_RELAY_ENABLED === 'true';
    const smartRelayRefreshMs = parseInt(process.env.SMART_RELAY_REFRESH_MS || '300000', 10);
    if (isNaN(smartRelayRefreshMs) || smartRelayRefreshMs < 10000) {
        throw new Error('Invalid SMART_RELAY_REFRESH_MS: must be at least 10000ms');
    }
    const kalshiWsUrl = process.env.KALSHI_WS_URL ||
        kalshiBaseUrl.replace('https://', 'wss://').replace('http://', 'ws://') + '/trade-api/ws/v2';
    const polymarketWsUrl = process.env.POLYMARKET_WS_URL || 'wss://ws-subscriptions-clob.polymarket.com/ws/market';
    const adminSecret = process.env.ADMIN_SECRET || undefined;
    const streamToken = process.env.STREAM_TOKEN || undefined;
    const cacheTtlMs = parseInt(process.env.CACHE_TTL_MS || '86400000', 10); // 24h default
    if (isNaN(cacheTtlMs) || cacheTtlMs < 60000) {
        throw new Error('Invalid CACHE_TTL_MS: must be at least 60000ms (1 minute)');
    }
    const relayCacheEnabled = process.env.RELAY_CACHE_ENABLED !== 'false';
    return {
        port,
        nodeEnv,
        corsOrigin,
        httpTimeoutMs,
        wsReconnectDelayMs,
        wsMaxReconnectAttempts,
        requestSizeLimit,
        kalshiBaseUrl,
        smartRelayEnabled,
        smartRelayRefreshMs,
        kalshiWsUrl,
        polymarketWsUrl,
        adminSecret,
        streamToken,
        cacheTtlMs,
        relayCacheEnabled,
    };
}
//# sourceMappingURL=config.js.map