/**
 * Admin Auth Middleware
 *
 * Gates admin endpoints behind a bearer token (ADMIN_SECRET env var).
 * Returns 401 if no token provided, 403 if token doesn't match.
 */
export function createAdminAuth(adminSecret) {
    return (req, res, next) => {
        if (!adminSecret) {
            res.status(503).json({ error: 'Admin endpoints not configured (ADMIN_SECRET not set)' });
            return;
        }
        const authHeader = req.headers.authorization;
        if (!authHeader) {
            res.status(401).json({ error: 'Authorization header required' });
            return;
        }
        const [scheme, token] = authHeader.split(' ');
        if (scheme?.toLowerCase() !== 'bearer' || !token) {
            res.status(401).json({ error: 'Bearer token required' });
            return;
        }
        // Constant-time comparison to prevent timing attacks
        if (!timingSafeEqual(token, adminSecret)) {
            res.status(403).json({ error: 'Invalid admin token' });
            return;
        }
        next();
    };
}
function timingSafeEqual(a, b) {
    if (a.length !== b.length)
        return false;
    let result = 0;
    for (let i = 0; i < a.length; i++) {
        result |= a.charCodeAt(i) ^ b.charCodeAt(i);
    }
    return result === 0;
}
//# sourceMappingURL=adminAuth.js.map