/**
 * Error Handling Middleware
 *
 * Global error handler for Express.
 */
import { RelayError } from '@galactus/shared';
/**
 * Express error handling middleware.
 *
 * Catches all errors and returns standardized error responses.
 * Never exposes stack traces in production.
 *
 * @param logger - Logger instance
 * @returns Express error handler middleware
 */
export function createErrorHandler(logger) {
    return (err, req, res, _next) => {
        // Log error
        logger.error('Request error', {
            method: req.method,
            path: req.path,
            error: err.message,
            stack: process.env.NODE_ENV === 'development' ? err.stack : undefined,
        });
        // Handle known error types
        if (err instanceof RelayError) {
            return res.status(err.statusCode).json({
                error: 'Relay error',
                message: err.message,
                code: err.code,
                requestId: err.requestId,
            });
        }
        // Handle unknown errors
        const statusCode = err.statusCode || 500;
        const response = {
            error: 'Internal server error',
            message: process.env.NODE_ENV === 'development' ? err.message : 'An error occurred',
        };
        if (process.env.NODE_ENV === 'development') {
            response.code = err.name;
        }
        res.status(statusCode).json(response);
    };
}
//# sourceMappingURL=errorHandler.js.map