/**
 * ApiKeyStore — Manages a pool of Kalshi API keys for rotation.
 *
 * Keys can be loaded from KALSHI_API_KEYS env var or added at runtime via admin endpoints.
 * Round-robin rotation across healthy keys. Auto-disables keys that get 401'd.
 *
 * Env format: KALSHI_API_KEYS=accessKey1:pemBase64_1,accessKey2:pemBase64_2,...
 * The PEM is base64-encoded (standard base64 of the full PEM string) so commas in PEM don't conflict.
 */
import type { Logger } from '../logger.js';
export interface ApiKey {
    id: string;
    accessKey: string;
    /** Raw PEM string (decoded from base64) */
    privateKeyPem: string;
    enabled: boolean;
    disabledReason?: string;
    disabledAt?: string;
    addedAt: string;
    /** Total requests made with this key */
    requestCount: number;
    /** Total errors (non-429, non-401) */
    errorCount: number;
    /** If rate-limited, don't use until this timestamp */
    rateLimitedUntilMs: number;
}
export interface ApiKeyPublic {
    id: string;
    accessKey: string;
    enabled: boolean;
    disabledReason?: string;
    disabledAt?: string;
    addedAt: string;
    requestCount: number;
    errorCount: number;
    rateLimited: boolean;
}
export declare class ApiKeyStore {
    private keys;
    private rotationIndex;
    private logger;
    constructor(logger: Logger);
    /** Load keys from KALSHI_API_KEYS env var. Does not clear existing keys. */
    loadFromEnv(): void;
    /** Add a key. Returns the key ID. */
    addKey(accessKey: string, privateKeyPem: string): string;
    /** Remove a key by ID. */
    removeKey(id: string): boolean;
    /** Disable a key (e.g., on 401). */
    disableKey(id: string, reason: string): void;
    /** Re-enable a key. */
    enableKey(id: string): boolean;
    /** Mark a key as rate-limited until a future timestamp. */
    rateLimitKey(id: string, retryAfterMs: number): void;
    /** Record a request against a key. */
    recordRequest(id: string): void;
    /** Record an error against a key. */
    recordError(id: string): void;
    /**
     * Get the next available key via round-robin.
     * Skips disabled and currently-rate-limited keys.
     * Returns null if no keys are available.
     */
    getNextKey(): ApiKey | null;
    /** Check if any keys are configured. */
    hasKeys(): boolean;
    /** Check if any keys are currently available (enabled + not rate-limited). */
    hasAvailableKeys(): boolean;
    /** List all keys (public info only, no PEMs). */
    listKeys(): ApiKeyPublic[];
    /** Get count of all / enabled / available keys. */
    getStats(): {
        total: number;
        enabled: number;
        available: number;
    };
}
//# sourceMappingURL=ApiKeyStore.d.ts.map