/**
 * ApiKeyStore — Manages a pool of Kalshi API keys for rotation.
 *
 * Keys can be loaded from KALSHI_API_KEYS env var or added at runtime via admin endpoints.
 * Round-robin rotation across healthy keys. Auto-disables keys that get 401'd.
 *
 * Env format: KALSHI_API_KEYS=accessKey1:pemBase64_1,accessKey2:pemBase64_2,...
 * The PEM is base64-encoded (standard base64 of the full PEM string) so commas in PEM don't conflict.
 */
let nextKeyId = 1;
export class ApiKeyStore {
    keys = new Map();
    rotationIndex = 0;
    logger;
    constructor(logger) {
        this.logger = logger;
    }
    /** Load keys from KALSHI_API_KEYS env var. Does not clear existing keys. */
    loadFromEnv() {
        const raw = process.env.KALSHI_API_KEYS;
        if (!raw || !raw.trim())
            return;
        const pairs = raw.split(',').filter((s) => s.trim());
        for (const pair of pairs) {
            const colonIdx = pair.indexOf(':');
            if (colonIdx <= 0) {
                this.logger.warn('ApiKeyStore: skipping malformed key entry (no colon separator)');
                continue;
            }
            const accessKey = pair.substring(0, colonIdx).trim();
            const pemBase64 = pair.substring(colonIdx + 1).trim();
            if (!accessKey || !pemBase64) {
                this.logger.warn('ApiKeyStore: skipping empty key entry');
                continue;
            }
            let privateKeyPem;
            try {
                privateKeyPem = Buffer.from(pemBase64, 'base64').toString('utf-8');
            }
            catch {
                this.logger.warn('ApiKeyStore: skipping key with invalid base64 PEM', {
                    accessKey: accessKey.substring(0, 8) + '...',
                });
                continue;
            }
            this.addKey(accessKey, privateKeyPem);
        }
        this.logger.info('ApiKeyStore: loaded keys from env', { count: this.keys.size });
    }
    /** Add a key. Returns the key ID. */
    addKey(accessKey, privateKeyPem) {
        // Check for duplicate access key
        for (const [, key] of this.keys) {
            if (key.accessKey === accessKey) {
                this.logger.info('ApiKeyStore: re-enabling existing key', {
                    accessKey: accessKey.substring(0, 8) + '...',
                });
                key.enabled = true;
                key.disabledReason = undefined;
                key.disabledAt = undefined;
                key.privateKeyPem = privateKeyPem;
                return key.id;
            }
        }
        const id = `key-${nextKeyId++}`;
        const key = {
            id,
            accessKey,
            privateKeyPem,
            enabled: true,
            addedAt: new Date().toISOString(),
            requestCount: 0,
            errorCount: 0,
            rateLimitedUntilMs: 0,
        };
        this.keys.set(id, key);
        this.logger.info('ApiKeyStore: added key', {
            id,
            accessKey: accessKey.substring(0, 8) + '...',
        });
        return id;
    }
    /** Remove a key by ID. */
    removeKey(id) {
        const deleted = this.keys.delete(id);
        if (deleted) {
            this.logger.info('ApiKeyStore: removed key', { id });
        }
        return deleted;
    }
    /** Disable a key (e.g., on 401). */
    disableKey(id, reason) {
        const key = this.keys.get(id);
        if (!key)
            return;
        key.enabled = false;
        key.disabledReason = reason;
        key.disabledAt = new Date().toISOString();
        this.logger.warn('ApiKeyStore: disabled key', {
            id,
            accessKey: key.accessKey.substring(0, 8) + '...',
            reason,
        });
    }
    /** Re-enable a key. */
    enableKey(id) {
        const key = this.keys.get(id);
        if (!key)
            return false;
        key.enabled = true;
        key.disabledReason = undefined;
        key.disabledAt = undefined;
        return true;
    }
    /** Mark a key as rate-limited until a future timestamp. */
    rateLimitKey(id, retryAfterMs) {
        const key = this.keys.get(id);
        if (!key)
            return;
        key.rateLimitedUntilMs = Date.now() + retryAfterMs;
        this.logger.warn('ApiKeyStore: rate-limited key', {
            id,
            accessKey: key.accessKey.substring(0, 8) + '...',
            retryAfterMs,
        });
    }
    /** Record a request against a key. */
    recordRequest(id) {
        const key = this.keys.get(id);
        if (key)
            key.requestCount++;
    }
    /** Record an error against a key. */
    recordError(id) {
        const key = this.keys.get(id);
        if (key)
            key.errorCount++;
    }
    /**
     * Get the next available key via round-robin.
     * Skips disabled and currently-rate-limited keys.
     * Returns null if no keys are available.
     */
    getNextKey() {
        const allKeys = Array.from(this.keys.values());
        if (allKeys.length === 0)
            return null;
        const now = Date.now();
        const availableKeys = allKeys.filter((k) => k.enabled && k.rateLimitedUntilMs <= now);
        if (availableKeys.length === 0)
            return null;
        this.rotationIndex = this.rotationIndex % availableKeys.length;
        const key = availableKeys[this.rotationIndex];
        this.rotationIndex = (this.rotationIndex + 1) % availableKeys.length;
        return key;
    }
    /** Check if any keys are configured. */
    hasKeys() {
        return this.keys.size > 0;
    }
    /** Check if any keys are currently available (enabled + not rate-limited). */
    hasAvailableKeys() {
        const now = Date.now();
        return Array.from(this.keys.values()).some((k) => k.enabled && k.rateLimitedUntilMs <= now);
    }
    /** List all keys (public info only, no PEMs). */
    listKeys() {
        const now = Date.now();
        return Array.from(this.keys.values()).map((k) => ({
            id: k.id,
            accessKey: k.accessKey.substring(0, 8) + '...' + k.accessKey.slice(-4),
            enabled: k.enabled,
            disabledReason: k.disabledReason,
            disabledAt: k.disabledAt,
            addedAt: k.addedAt,
            requestCount: k.requestCount,
            errorCount: k.errorCount,
            rateLimited: k.rateLimitedUntilMs > now,
        }));
    }
    /** Get count of all / enabled / available keys. */
    getStats() {
        const now = Date.now();
        const all = Array.from(this.keys.values());
        return {
            total: all.length,
            enabled: all.filter((k) => k.enabled).length,
            available: all.filter((k) => k.enabled && k.rateLimitedUntilMs <= now).length,
        };
    }
}
//# sourceMappingURL=ApiKeyStore.js.map