/**
 * PolymarketFetcher — Persistent Polymarket data fetcher.
 *
 * Resolves NBA market token IDs via the Gamma API, then connects to the
 * Polymarket CLOB WebSocket for orderbook streaming. Normalizes bids/asks
 * into the same YES/NO bid-ladder format used by the cache.
 */
import axios from 'axios';
import WebSocket from 'ws';
const GAMMA_API = 'https://gamma-api.polymarket.com';
export class PolymarketFetcher {
    config;
    logger;
    cache;
    ws = null;
    tokenIds = [];
    tokenToTicker = new Map();
    refreshTimer = null;
    reconnectTimer = null;
    reconnectAttempts = 0;
    maxReconnectAttempts = 20;
    stopped = false;
    _isConnected = false;
    constructor(config, logger, cache) {
        this.config = config;
        this.logger = logger;
        this.cache = cache;
    }
    get isConnected() {
        return this._isConnected;
    }
    async start() {
        this.stopped = false;
        await this.discover();
        if (this.tokenIds.length > 0) {
            this.connectWs();
        }
        this.refreshTimer = setInterval(() => this.discover().catch((e) => this.logger.error('Polymarket re-discovery failed', { error: String(e) })), this.config.smartRelayRefreshMs);
    }
    stop() {
        this.stopped = true;
        if (this.refreshTimer) {
            clearInterval(this.refreshTimer);
            this.refreshTimer = null;
        }
        if (this.reconnectTimer) {
            clearTimeout(this.reconnectTimer);
            this.reconnectTimer = null;
        }
        if (this.ws) {
            this.ws.close();
            this.ws = null;
        }
        this._isConnected = false;
    }
    // ── Gamma Discovery ───────────────────────────────────────────────────
    async discover() {
        try {
            // Search for NBA markets via Gamma
            const res = await axios.get(`${GAMMA_API}/markets`, {
                params: { tag: 'nba', closed: false, limit: 200 },
                headers: { Accept: 'application/json', 'User-Agent': 'Mozilla/5.0' },
                timeout: 15000,
            });
            const markets = Array.isArray(res.data) ? res.data : [];
            if (markets.length === 0) {
                this.logger.info('Polymarket: no NBA markets found');
                return;
            }
            const newTokenIds = [];
            for (const m of markets) {
                if (!m.tokens || m.tokens.length < 2)
                    continue;
                for (const token of m.tokens) {
                    const tokenId = token.token_id;
                    if (!tokenId)
                        continue;
                    const ticker = `poly:${tokenId}`;
                    this.tokenToTicker.set(tokenId, ticker);
                    newTokenIds.push(tokenId);
                    // Write initial price from Gamma
                    const cached = {
                        ticker,
                        venue: 'polymarket',
                        price: typeof token.price === 'number' ? token.price : null,
                        yesBidCents: null,
                        yesAskCents: null,
                        noBidCents: null,
                        noAskCents: null,
                        volume: null,
                        updatedAt: new Date().toISOString(),
                    };
                    this.cache.updateMarket(cached);
                }
            }
            this.tokenIds = newTokenIds;
            this.logger.info('Polymarket: discovered tokens', { count: newTokenIds.length });
            // If WS is open, subscribe to new tokens
            if (this.ws && this.ws.readyState === WebSocket.OPEN) {
                this.subscribeAll();
            }
        }
        catch (err) {
            this.logger.error('Polymarket: discovery failed', { error: String(err) });
        }
    }
    // ── WebSocket ─────────────────────────────────────────────────────────
    connectWs() {
        if (this.stopped)
            return;
        try {
            this.ws = new WebSocket(this.config.polymarketWsUrl);
            this.ws.on('open', () => {
                this._isConnected = true;
                this.reconnectAttempts = 0;
                this.logger.info('Polymarket WS: connected');
                this.subscribeAll();
            });
            this.ws.on('message', (data) => {
                try {
                    const msg = JSON.parse(data.toString());
                    this.handleWsMessage(msg);
                }
                catch (err) {
                    this.logger.error('Polymarket WS: parse error', { error: String(err) });
                }
            });
            this.ws.on('close', () => {
                this._isConnected = false;
                this.logger.info('Polymarket WS: disconnected');
                this.scheduleReconnect();
            });
            this.ws.on('error', (err) => {
                this.logger.error('Polymarket WS: error', { error: String(err) });
            });
        }
        catch (err) {
            this.logger.error('Polymarket WS: connect failed', { error: String(err) });
            this.scheduleReconnect();
        }
    }
    subscribeAll() {
        if (!this.ws || this.ws.readyState !== WebSocket.OPEN)
            return;
        if (this.tokenIds.length === 0)
            return;
        // Polymarket CLOB WS subscription
        const msg = {
            type: 'market',
            assets_ids: this.tokenIds,
        };
        this.ws.send(JSON.stringify(msg));
        this.logger.info('Polymarket WS: subscribed', { tokens: this.tokenIds.length });
    }
    handleWsMessage(msg) {
        // Polymarket sends book snapshots/updates with bids/asks arrays
        const assetId = (msg.asset_id ?? msg.market ?? msg.token_id);
        if (!assetId)
            return;
        const ticker = this.tokenToTicker.get(assetId);
        if (!ticker)
            return;
        const bids = msg.bids;
        const asks = msg.asks;
        if (!Array.isArray(bids) && !Array.isArray(asks))
            return;
        // Normalize: bids → YES levels, asks inverted → NO levels
        const yesLevels = polyBidsAsYesLevels(bids ?? []);
        const noLevels = polyAsksAsNoLevels(asks ?? []);
        const cached = {
            ticker,
            venue: 'polymarket',
            yes: yesLevels,
            no: noLevels,
            updatedAt: new Date().toISOString(),
        };
        this.cache.updateOrderbook(cached);
        // Derive market price from orderbook
        const bestYesBid = yesLevels.length > 0 ? yesLevels[0].priceCents : null;
        const bestNoBid = noLevels.length > 0 ? noLevels[0].priceCents : null;
        const yesAsk = bestNoBid !== null ? 100 - bestNoBid : null;
        const price = bestYesBid !== null && yesAsk !== null
            ? (bestYesBid + yesAsk) / 2 / 100
            : bestYesBid !== null
                ? bestYesBid / 100
                : yesAsk !== null
                    ? yesAsk / 100
                    : null;
        const market = {
            ticker,
            venue: 'polymarket',
            price,
            yesBidCents: bestYesBid,
            yesAskCents: yesAsk,
            noBidCents: bestNoBid,
            noAskCents: bestYesBid !== null ? 100 - bestYesBid : null,
            volume: null,
            updatedAt: new Date().toISOString(),
        };
        this.cache.updateMarket(market);
    }
    scheduleReconnect() {
        if (this.stopped)
            return;
        if (this.reconnectAttempts >= this.maxReconnectAttempts) {
            this.logger.error('Polymarket WS: max reconnect attempts reached');
            return;
        }
        const delay = Math.min(1000 * Math.pow(2, this.reconnectAttempts), 60000);
        this.reconnectAttempts++;
        this.logger.info('Polymarket WS: reconnecting', {
            attempt: this.reconnectAttempts,
            delayMs: delay,
        });
        this.reconnectTimer = setTimeout(() => this.connectWs(), delay);
    }
}
// ── Normalization (mirrors dashboard polymarket/normalize.ts) ────────
function polyBidsAsYesLevels(bids) {
    return bids
        .map((lvl) => ({
        priceCents: Math.round(Number(lvl.price) * 100),
        size: Number(lvl.size),
    }))
        .filter((x) => Number.isFinite(x.priceCents) &&
        x.priceCents > 0 &&
        x.priceCents < 100 &&
        Number.isFinite(x.size) &&
        x.size > 0)
        .sort((a, b) => b.priceCents - a.priceCents);
}
function polyAsksAsNoLevels(asks) {
    return asks
        .map((lvl) => ({
        priceCents: 100 - Math.round(Number(lvl.price) * 100),
        size: Number(lvl.size),
    }))
        .filter((x) => Number.isFinite(x.priceCents) &&
        x.priceCents > 0 &&
        x.priceCents < 100 &&
        Number.isFinite(x.size) &&
        x.size > 0)
        .sort((a, b) => b.priceCents - a.priceCents);
}
//# sourceMappingURL=PolymarketFetcher.js.map