/**
 * StreamBroadcaster — Manages dashboard client WebSocket connections on /stream/markets.
 *
 * - On client subscribe: sends current cache snapshot, registers for live updates
 * - Listens to MarketCache events and broadcasts StreamUpdate to all subscribed clients
 * - Supports ticker filtering (client can subscribe to specific tickers or all)
 * - Periodic status heartbeat every 30s
 */
import { WebSocket } from 'ws';
/** Max pending messages before disconnecting a slow client. */
const CLIENT_HIGH_WATER_MARK = 100;
export class StreamBroadcaster {
    logger;
    cache;
    clients = new Map();
    heartbeatTimer = null;
    startTime = Date.now();
    getKalshiConnected;
    getPolymarketConnected;
    constructor(logger, cache, opts) {
        this.logger = logger;
        this.cache = cache;
        this.getKalshiConnected = opts.getKalshiConnected;
        this.getPolymarketConnected = opts.getPolymarketConnected;
        // Listen to cache events
        this.cache.on('market:update', (market) => {
            this.broadcastMarketUpdate(market);
        });
        this.cache.on('orderbook:update', (orderbook) => {
            this.broadcastOrderbookUpdate(orderbook);
        });
    }
    start() {
        this.startTime = Date.now();
        // Heartbeat every 30s
        this.heartbeatTimer = setInterval(() => this.broadcastStatus(), 30000);
    }
    stop() {
        if (this.heartbeatTimer) {
            clearInterval(this.heartbeatTimer);
            this.heartbeatTimer = null;
        }
        // Close all client connections
        for (const [ws] of this.clients) {
            try {
                ws.close(1001, 'Server shutting down');
            }
            catch {
                // ignore
            }
        }
        this.clients.clear();
    }
    /** Handle a new client WebSocket connection. */
    handleClientConnection(ws) {
        const sub = {
            ws,
            marketTickers: null,
            orderbookTickers: null,
            pendingMessages: 0,
        };
        this.clients.set(ws, sub);
        this.logger.info('Stream client connected', { clients: this.clients.size });
        ws.on('message', (data) => {
            try {
                const msg = JSON.parse(data.toString());
                this.handleClientMessage(ws, msg);
            }
            catch (_err) {
                this.sendToClient(ws, {
                    type: 'error',
                    message: 'Invalid message format',
                    timestamp: new Date().toISOString(),
                });
            }
        });
        ws.on('close', () => {
            this.clients.delete(ws);
            this.logger.info('Stream client disconnected', { clients: this.clients.size });
        });
        ws.on('error', (err) => {
            this.logger.error('Stream client error', { error: String(err) });
            this.clients.delete(ws);
        });
    }
    getClientCount() {
        return this.clients.size;
    }
    getStatus() {
        const stats = this.cache.getStats();
        return {
            type: 'status',
            kalshiConnected: this.getKalshiConnected(),
            polymarketConnected: this.getPolymarketConnected(),
            marketCount: stats.marketCount,
            orderbookCount: stats.orderbookCount,
            uptimeMs: Date.now() - this.startTime,
            timestamp: new Date().toISOString(),
        };
    }
    // ── Private ───────────────────────────────────────────────────────────
    handleClientMessage(ws, msg) {
        const sub = this.clients.get(ws);
        if (!sub)
            return;
        if (msg.op === 'subscribe') {
            if (msg.channel === 'markets') {
                if (msg.tickers && msg.tickers.length > 0) {
                    if (!sub.marketTickers)
                        sub.marketTickers = new Set();
                    msg.tickers.forEach((t) => sub.marketTickers.add(t));
                }
                else {
                    sub.marketTickers = null; // null = all
                }
                // Send snapshot
                const markets = this.cache.getMarketSnapshot();
                const filtered = sub.marketTickers
                    ? markets.filter((m) => sub.marketTickers.has(m.ticker))
                    : markets;
                const snapshot = {
                    type: 'snapshot',
                    channel: 'markets',
                    markets: filtered,
                    timestamp: new Date().toISOString(),
                };
                this.sendToClient(ws, snapshot);
            }
            if (msg.channel === 'orderbooks') {
                if (msg.tickers && msg.tickers.length > 0) {
                    if (!sub.orderbookTickers)
                        sub.orderbookTickers = new Set();
                    msg.tickers.forEach((t) => sub.orderbookTickers.add(t));
                }
                else {
                    sub.orderbookTickers = null; // null = all
                }
                const orderbooks = this.cache.getOrderbookSnapshot();
                const filtered = sub.orderbookTickers
                    ? orderbooks.filter((o) => sub.orderbookTickers.has(o.ticker))
                    : orderbooks;
                const snapshot = {
                    type: 'snapshot',
                    channel: 'orderbooks',
                    orderbooks: filtered,
                    timestamp: new Date().toISOString(),
                };
                this.sendToClient(ws, snapshot);
            }
        }
        if (msg.op === 'unsubscribe') {
            if (msg.channel === 'markets') {
                if (msg.tickers && sub.marketTickers) {
                    msg.tickers.forEach((t) => sub.marketTickers.delete(t));
                }
                else {
                    sub.marketTickers = new Set(); // empty set = unsubscribed from all
                }
            }
            if (msg.channel === 'orderbooks') {
                if (msg.tickers && sub.orderbookTickers) {
                    msg.tickers.forEach((t) => sub.orderbookTickers.delete(t));
                }
                else {
                    sub.orderbookTickers = new Set();
                }
            }
        }
    }
    broadcastMarketUpdate(market) {
        const msg = {
            type: 'update',
            channel: 'markets',
            market,
            timestamp: new Date().toISOString(),
        };
        for (const [ws, sub] of this.clients) {
            // null = subscribed to all; Set = filter
            if (sub.marketTickers === null || sub.marketTickers.has(market.ticker)) {
                this.sendToClient(ws, msg);
            }
        }
    }
    broadcastOrderbookUpdate(orderbook) {
        const msg = {
            type: 'update',
            channel: 'orderbooks',
            orderbook,
            timestamp: new Date().toISOString(),
        };
        for (const [ws, sub] of this.clients) {
            if (sub.orderbookTickers === null || sub.orderbookTickers.has(orderbook.ticker)) {
                this.sendToClient(ws, msg);
            }
        }
    }
    broadcastStatus() {
        const status = this.getStatus();
        for (const [ws] of this.clients) {
            this.sendToClient(ws, status);
        }
    }
    sendToClient(ws, msg) {
        if (ws.readyState !== WebSocket.OPEN)
            return;
        const sub = this.clients.get(ws);
        if (!sub)
            return;
        // Backpressure: disconnect slow clients that can't keep up
        if (sub.pendingMessages >= CLIENT_HIGH_WATER_MARK) {
            this.logger.warn('Stream client too slow, disconnecting', {
                pending: sub.pendingMessages,
            });
            this.clients.delete(ws);
            try {
                ws.close(4002, 'Too slow: message backlog exceeded');
            }
            catch {
                // ignore
            }
            return;
        }
        try {
            sub.pendingMessages++;
            ws.send(JSON.stringify(msg), (err) => {
                sub.pendingMessages = Math.max(0, sub.pendingMessages - 1);
                if (err) {
                    this.logger.error('Failed to send to stream client', { error: String(err) });
                }
            });
        }
        catch (err) {
            sub.pendingMessages = Math.max(0, sub.pendingMessages - 1);
            this.logger.error('Failed to send to stream client', { error: String(err) });
        }
    }
}
//# sourceMappingURL=StreamBroadcaster.js.map