/**
 * Response Cache for HTTP Relay
 *
 * Caches GET responses for known Kalshi endpoints that return identical data
 * for all users (market discovery, event metadata, orderbook snapshots).
 * Keyed by URL only — auth headers differ per user but responses are the same.
 */

import { HttpRelayResponse } from '@galactus/shared';

interface CachedEntry {
  status: number;
  headers: Record<string, string>;
  body: string;
  cachedAt: number;
  ttlMs: number;
}

interface CacheStats {
  hits: number;
  misses: number;
  entries: number;
}

/** URL patterns and their TTLs. Order doesn't matter — first match wins. */
const CACHE_RULES: Array<{ pattern: RegExp; ttlMs: number }> = [
  // Market discovery: GET /trade-api/v2/markets?series_ticker=...
  { pattern: /\/trade-api\/v2\/markets\?/, ttlMs: 5 * 60_000 },
  // Event metadata: GET /trade-api/v2/events/
  { pattern: /\/trade-api\/v2\/events\//, ttlMs: 30 * 60_000 },
  // Orderbook snapshots: GET /trade-api/v2/orderbook/v2/
  { pattern: /\/trade-api\/v2\/orderbook\/v2\//, ttlMs: 30_000 },
];

const PRUNE_INTERVAL_MS = 60_000;

export class ResponseCache {
  private cache = new Map<string, CachedEntry>();
  private hits = 0;
  private misses = 0;
  private pruneTimer: ReturnType<typeof setInterval>;

  constructor() {
    this.pruneTimer = setInterval(() => this.prune(), PRUNE_INTERVAL_MS);
    // Don't keep the process alive just for pruning
    this.pruneTimer.unref();
  }

  /**
   * Build a cache key from a URL. Returns null if the URL is not cacheable.
   */
  buildKey(url: string): string | null {
    const ttl = this.getTtl(url);
    if (ttl === null) return null;

    try {
      const parsed = new URL(url);
      // Sort query params for consistent keys
      const params = new URLSearchParams(parsed.searchParams);
      const sorted = new URLSearchParams([...params.entries()].sort());
      return `${parsed.pathname}?${sorted.toString()}`;
    } catch {
      return null;
    }
  }

  /**
   * Get a cached response if it exists and is still fresh.
   */
  get(key: string): Omit<HttpRelayResponse, 'id'> | null {
    const entry = this.cache.get(key);
    if (!entry) {
      this.misses++;
      return null;
    }

    if (Date.now() - entry.cachedAt > entry.ttlMs) {
      this.cache.delete(key);
      this.misses++;
      return null;
    }

    this.hits++;
    return {
      status: entry.status,
      headers: entry.headers,
      body: entry.body,
    };
  }

  /**
   * Store a response if the URL matches a cacheable pattern.
   */
  tryStore(url: string, response: HttpRelayResponse): void {
    const ttl = this.getTtl(url);
    if (ttl === null) return;

    const key = this.buildKey(url);
    if (!key) return;

    this.cache.set(key, {
      status: response.status,
      headers: response.headers,
      body: response.body,
      cachedAt: Date.now(),
      ttlMs: ttl,
    });
  }

  getStats(): CacheStats {
    return {
      hits: this.hits,
      misses: this.misses,
      entries: this.cache.size,
    };
  }

  destroy(): void {
    clearInterval(this.pruneTimer);
    this.cache.clear();
  }

  private getTtl(url: string): number | null {
    for (const rule of CACHE_RULES) {
      if (rule.pattern.test(url)) {
        return rule.ttlMs;
      }
    }
    return null;
  }

  private prune(): void {
    const now = Date.now();
    for (const [key, entry] of this.cache) {
      if (now - entry.cachedAt > entry.ttlMs) {
        this.cache.delete(key);
      }
    }
  }
}
