/**
 * KalshiFetcher — Persistent Kalshi REST + WebSocket data fetcher.
 *
 * On startup:
 *  1. REST: discover NBA markets via /trade-api/v2/events (public, no auth)
 *  2. REST: fetch initial orderbooks for all discovered tickers
 *  3. WS: connect to Kalshi WebSocket and subscribe to orderbook_delta + ticker + trade channels
 *
 * Features:
 *  - API key rotation via ApiKeyStore (round-robin across healthy keys)
 *  - 429 rate-limit handling with Retry-After parsing and adaptive backoff
 *  - Auto-disables keys on 401 Unauthorized
 *  - Writes all updates into the shared MarketCache
 *  - Periodic re-discovery every smartRelayRefreshMs to pick up new games
 *  - WebSocket reconnection with exponential backoff
 */

import axios, { AxiosRequestConfig } from 'axios';
import WebSocket from 'ws';
import type { ServerConfig } from '../config.js';
import type { Logger } from '../logger.js';
import type { MarketCache } from './MarketCache.js';
import type { ApiKeyStore } from './ApiKeyStore.js';
import type { CachedMarket, CachedOrderbook } from '@galactus/shared';

interface KalshiMarketRaw {
  ticker: string;
  event_ticker: string;
  title?: string;
  yes_sub_title?: string;
  no_sub_title?: string;
  status?: string;
  last_price?: number;
  yes_bid?: number;
  yes_ask?: number;
  no_bid?: number;
  no_ask?: number;
  volume?: number;
  volume_24h?: number;
}

interface KalshiOrderbookRaw {
  yes?: Array<[number, number]>;
  no?: Array<[number, number]>;
}

/** Default retry-after in ms when header is missing */
const DEFAULT_RETRY_AFTER_MS = 10000;
/** Max retries per request on 429 */
const MAX_429_RETRIES = 3;

export class KalshiFetcher {
  private config: ServerConfig;
  private logger: Logger;
  private cache: MarketCache;
  private keyStore: ApiKeyStore;
  private ws: WebSocket | null = null;
  private discoveredTickers: string[] = [];
  private refreshTimer: ReturnType<typeof setInterval> | null = null;
  private reconnectTimer: ReturnType<typeof setTimeout> | null = null;
  private reconnectAttempts = 0;
  private maxReconnectAttempts = 20;
  private stopped = false;
  private _isConnected = false;
  private nextMsgId = 1;

  constructor(config: ServerConfig, logger: Logger, cache: MarketCache, keyStore: ApiKeyStore) {
    this.config = config;
    this.logger = logger;
    this.cache = cache;
    this.keyStore = keyStore;
  }

  get isConnected(): boolean {
    return this._isConnected;
  }

  async start(): Promise<void> {
    this.stopped = false;
    await this.discover();
    this.connectWs();

    // Periodic re-discovery
    this.refreshTimer = setInterval(
      () =>
        this.discover().catch((e) =>
          this.logger.error('Kalshi re-discovery failed', { error: String(e) })
        ),
      this.config.smartRelayRefreshMs
    );
  }

  stop(): void {
    this.stopped = true;
    if (this.refreshTimer) {
      clearInterval(this.refreshTimer);
      this.refreshTimer = null;
    }
    if (this.reconnectTimer) {
      clearTimeout(this.reconnectTimer);
      this.reconnectTimer = null;
    }
    if (this.ws) {
      this.ws.close();
      this.ws = null;
    }
    this._isConnected = false;
  }

  // ── Authenticated HTTP with key rotation + rate limit handling ────────

  /**
   * Make an HTTP GET to Kalshi with optional API key auth.
   * Uses key rotation if keys are available, falls back to unauthenticated.
   * Handles 429 with retry-after and 401 with key disabling.
   */
  private async kalshiGet(
    path: string,
    params?: Record<string, unknown>,
    retryCount = 0
  ): Promise<{ data: unknown; status: number }> {
    const baseUrl = this.config.kalshiBaseUrl;
    const url = `${baseUrl}${path}`;

    const axiosConfig: AxiosRequestConfig = {
      method: 'GET',
      url,
      params,
      timeout: 15000,
      validateStatus: () => true, // handle all statuses ourselves
    };

    // Attach API key headers if available
    const key = this.keyStore.getNextKey();
    if (key) {
      axiosConfig.headers = {
        'KALSHI-ACCESS-KEY': key.accessKey,
        // Kalshi REST auth also needs a signature, but for public endpoints
        // the access key alone may work as a rate-limit identifier.
        // If full signing is needed, it can be added here.
      };
      this.keyStore.recordRequest(key.id);
    }

    try {
      const res = await axios(axiosConfig);

      // 429 — Rate limited
      if (res.status === 429) {
        const retryAfterMs = parseRetryAfter(res.headers['retry-after']);
        this.logger.warn('Kalshi: rate limited (429)', { path, retryAfterMs, keyId: key?.id });

        if (key) {
          this.keyStore.rateLimitKey(key.id, retryAfterMs);
        }

        if (retryCount < MAX_429_RETRIES) {
          await sleep(retryAfterMs);
          return this.kalshiGet(path, params, retryCount + 1);
        }

        throw new Error(`Kalshi 429: max retries exceeded for ${path}`);
      }

      // 401 — Unauthorized (bad key)
      if (res.status === 401 && key) {
        this.keyStore.disableKey(key.id, `401 Unauthorized on ${path}`);
        this.logger.warn('Kalshi: key disabled due to 401', { keyId: key.id, path });

        // Retry without this key (will pick next in rotation)
        if (retryCount < MAX_429_RETRIES && this.keyStore.hasAvailableKeys()) {
          return this.kalshiGet(path, params, retryCount + 1);
        }
      }

      // 5xx — Server error
      if (res.status >= 500 && retryCount < 2) {
        if (key) this.keyStore.recordError(key.id);
        await sleep(2000 * (retryCount + 1));
        return this.kalshiGet(path, params, retryCount + 1);
      }

      if (res.status >= 400 && key) {
        this.keyStore.recordError(key.id);
      }

      return { data: res.data, status: res.status };
    } catch (err) {
      if (key) this.keyStore.recordError(key.id);

      // Network errors — retry once
      if (retryCount < 1) {
        await sleep(2000);
        return this.kalshiGet(path, params, retryCount + 1);
      }

      throw err;
    }
  }

  // ── REST Discovery ────────────────────────────────────────────────────

  private async discover(): Promise<void> {
    try {
      // Discover NBA events (public endpoint)
      const eventsRes = await this.kalshiGet('/trade-api/v2/events', {
        series_ticker: 'KXNBA',
        status: 'open',
        limit: 200,
      });

      const events: Array<{ event_ticker: string }> =
        ((eventsRes.data as Record<string, unknown>)?.events as Array<{ event_ticker: string }>) ??
        [];
      if (events.length === 0) {
        this.logger.info('Kalshi: no open NBA events found');
        return;
      }

      // Discover markets for these events
      const allMarkets: KalshiMarketRaw[] = [];
      for (const evt of events) {
        try {
          const marketsRes = await this.kalshiGet('/trade-api/v2/markets', {
            event_ticker: evt.event_ticker,
            limit: 200,
          });
          const markets: KalshiMarketRaw[] =
            ((marketsRes.data as Record<string, unknown>)?.markets as KalshiMarketRaw[]) ?? [];
          allMarkets.push(...markets);
        } catch (err) {
          this.logger.error('Kalshi: market fetch failed for event', {
            event: evt.event_ticker,
            error: String(err),
          });
        }
      }

      this.logger.info('Kalshi: discovered markets', { count: allMarkets.length });

      // Write initial market data into cache
      for (const m of allMarkets) {
        const cached: CachedMarket = {
          ticker: m.ticker,
          venue: 'kalshi',
          price: normalizePriceTo01(m.last_price ?? m.yes_bid ?? null),
          yesBidCents: toCents(m.yes_bid),
          yesAskCents: toCents(m.yes_ask),
          noBidCents: toCents(m.no_bid),
          noAskCents: toCents(m.no_ask),
          volume: m.volume_24h ?? m.volume ?? null,
          updatedAt: new Date().toISOString(),
        };
        this.cache.updateMarket(cached);
      }

      // Fetch orderbooks
      const tickers = allMarkets.map((m) => m.ticker);
      this.discoveredTickers = tickers;

      // Fetch orderbooks in batches (avoid hammering)
      const BATCH = 10;
      for (let i = 0; i < tickers.length; i += BATCH) {
        const batch = tickers.slice(i, i + BATCH);
        await Promise.all(batch.map((t) => this.fetchOrderbook(t)));
      }

      this.logger.info('Kalshi: initial orderbooks loaded', { count: tickers.length });
    } catch (err) {
      this.logger.error('Kalshi: discovery failed', { error: String(err) });
    }
  }

  private async fetchOrderbook(ticker: string): Promise<void> {
    try {
      const res = await this.kalshiGet(`/trade-api/v2/orderbook/v2/${ticker}`);
      const raw = res.data as Record<string, unknown>;
      const ob: KalshiOrderbookRaw = (raw?.orderbook ?? raw ?? {}) as KalshiOrderbookRaw;

      const cached: CachedOrderbook = {
        ticker,
        venue: 'kalshi',
        yes: toLevels(ob.yes),
        no: toLevels(ob.no),
        updatedAt: new Date().toISOString(),
      };
      this.cache.updateOrderbook(cached);
    } catch (err) {
      this.logger.error('Kalshi: orderbook fetch failed', { ticker, error: String(err) });
    }
  }

  // ── WebSocket ─────────────────────────────────────────────────────────

  private connectWs(): void {
    if (this.stopped) return;

    try {
      this.ws = new WebSocket(this.config.kalshiWsUrl);

      this.ws.on('open', () => {
        this._isConnected = true;
        this.reconnectAttempts = 0;
        this.logger.info('Kalshi WS: connected');

        // Subscribe to all discovered tickers
        this.subscribeAll();
      });

      this.ws.on('message', (data: WebSocket.Data) => {
        try {
          const msg = JSON.parse(data.toString());
          this.handleWsMessage(msg);
        } catch (err) {
          this.logger.error('Kalshi WS: parse error', { error: String(err) });
        }
      });

      this.ws.on('close', () => {
        this._isConnected = false;
        this.logger.info('Kalshi WS: disconnected');
        this.scheduleReconnect();
      });

      this.ws.on('error', (err) => {
        this.logger.error('Kalshi WS: error', { error: String(err) });
      });
    } catch (err) {
      this.logger.error('Kalshi WS: connect failed', { error: String(err) });
      this.scheduleReconnect();
    }
  }

  private subscribeAll(): void {
    if (!this.ws || this.ws.readyState !== WebSocket.OPEN) return;

    for (const ticker of this.discoveredTickers) {
      const msg = {
        id: this.nextMsgId++,
        cmd: 'subscribe',
        params: {
          channels: ['orderbook_delta', 'ticker', 'trade'],
          market_ticker: ticker,
          market_tickers: [ticker],
        },
      };
      this.ws.send(JSON.stringify(msg));
    }

    this.logger.info('Kalshi WS: subscribed', { tickers: this.discoveredTickers.length });
  }

  private handleWsMessage(msg: Record<string, unknown>): void {
    const channel =
      (typeof msg.channel === 'string' ? msg.channel : undefined) ??
      (typeof msg.type === 'string' ? msg.type : undefined);

    if (!channel) return;

    if (
      channel === 'orderbook' ||
      channel === 'orderbook_snapshot' ||
      channel === 'orderbook_delta' ||
      channel === 'orderbook_update'
    ) {
      this.handleOrderbookMessage(msg);
      return;
    }

    if (channel === 'ticker') {
      this.handleTickerMessage(msg);
      return;
    }

    if (channel === 'trade' || channel === 'trades' || channel === 'public_trades') {
      this.handleTradeMessage(msg);
      return;
    }
  }

  private handleOrderbookMessage(msg: Record<string, unknown>): void {
    const payload = (msg.msg ?? msg.data ?? msg.payload ?? msg) as Record<string, unknown>;
    const ticker =
      (payload.market_ticker as string) ??
      (payload.ticker as string) ??
      (payload.marketTicker as string);
    if (!ticker) return;

    // Snapshot has yes/no arrays
    const yesRaw = payload.yes ?? (payload.orderbook as Record<string, unknown>)?.yes;
    const noRaw = payload.no ?? (payload.orderbook as Record<string, unknown>)?.no;

    if (Array.isArray(yesRaw) || Array.isArray(noRaw)) {
      const cached: CachedOrderbook = {
        ticker,
        venue: 'kalshi',
        yes: toLevels(yesRaw as Array<[number, number]> | undefined),
        no: toLevels(noRaw as Array<[number, number]> | undefined),
        updatedAt: new Date().toISOString(),
      };
      this.cache.updateOrderbook(cached);
      this.deriveMarketPriceFromOrderbook(ticker, cached);
      return;
    }

    // Delta: { side: "yes"|"no", price: 62, delta: 100 }
    const existing = this.cache.getOrderbook(ticker);
    if (!existing) return;

    const side = typeof payload.side === 'string' ? payload.side.toLowerCase() : '';
    const price = typeof payload.price === 'number' ? Math.round(payload.price) : null;
    const delta = typeof payload.delta === 'number' ? payload.delta : null;
    if ((side !== 'yes' && side !== 'no') || price === null || delta === null) return;

    const book = side === 'yes' ? [...existing.yes] : [...existing.no];
    const idx = book.findIndex((l) => l.priceCents === price);
    if (idx >= 0) {
      const newSize = book[idx].size + delta;
      if (newSize <= 0) {
        book.splice(idx, 1);
      } else {
        book[idx] = { priceCents: price, size: newSize };
      }
    } else if (delta > 0) {
      book.push({ priceCents: price, size: delta });
      book.sort((a, b) => b.priceCents - a.priceCents);
    }

    const updated: CachedOrderbook = {
      ...existing,
      [side]: book,
      updatedAt: new Date().toISOString(),
    };
    this.cache.updateOrderbook(updated);
    this.deriveMarketPriceFromOrderbook(ticker, updated);
  }

  private handleTickerMessage(msg: Record<string, unknown>): void {
    const payload = (msg.msg ?? msg.data ?? msg.payload ?? msg) as Record<string, unknown>;
    const ticker =
      (payload.market_ticker as string) ??
      (payload.ticker as string) ??
      (payload.marketTicker as string);
    if (!ticker) return;

    const yesBid = toCents(payload.yes_bid ?? payload.best_yes_bid ?? payload.bid);
    const yesAsk = toCents(payload.yes_ask ?? payload.best_yes_ask ?? payload.ask);
    const noBid = toCents(payload.no_bid ?? payload.best_no_bid);
    const noAsk = toCents(payload.no_ask ?? payload.best_no_ask);

    const price =
      computeMid01(yesBid, yesAsk) ??
      normalizePriceTo01(payload.yes_price ?? payload.last_price ?? payload.price);

    const cached: CachedMarket = {
      ticker,
      venue: 'kalshi',
      price,
      yesBidCents: yesBid,
      yesAskCents: yesAsk,
      noBidCents: noBid,
      noAskCents: noAsk,
      volume: typeof payload.volume === 'number' ? payload.volume : null,
      updatedAt: new Date().toISOString(),
    };
    this.cache.updateMarket(cached);
  }

  private handleTradeMessage(msg: Record<string, unknown>): void {
    const payload = msg.msg ?? msg.trade ?? msg.trades ?? msg.data ?? msg.payload;
    const trades = Array.isArray(payload) ? payload : payload ? [payload] : [];

    for (const trade of trades) {
      const rec = trade && typeof trade === 'object' ? (trade as Record<string, unknown>) : null;
      if (!rec) continue;

      const ticker =
        (rec.market_ticker as string) ?? (rec.ticker as string) ?? (rec.marketTicker as string);
      if (!ticker) continue;

      const price = normalizePriceTo01(rec.price ?? rec.yes_price ?? rec.price_dollars);
      if (price === null) continue;

      const existing = this.cache.getMarket(ticker);
      const cached: CachedMarket = {
        ticker,
        venue: 'kalshi',
        price,
        yesBidCents: existing?.yesBidCents ?? null,
        yesAskCents: existing?.yesAskCents ?? null,
        noBidCents: existing?.noBidCents ?? null,
        noAskCents: existing?.noAskCents ?? null,
        volume: typeof rec.volume === 'number' ? rec.volume : (existing?.volume ?? null),
        updatedAt: new Date().toISOString(),
      };
      this.cache.updateMarket(cached);
    }
  }

  private deriveMarketPriceFromOrderbook(ticker: string, ob: CachedOrderbook): void {
    const bestYesBid = ob.yes.length > 0 ? Math.max(...ob.yes.map((l) => l.priceCents)) : null;
    const bestNoBid = ob.no.length > 0 ? Math.max(...ob.no.map((l) => l.priceCents)) : null;
    const yesAsk = bestNoBid !== null ? 100 - bestNoBid : null;

    const mid = computeMid01(bestYesBid, yesAsk);
    if (mid === null) return;

    const existing = this.cache.getMarket(ticker);
    const cached: CachedMarket = {
      ticker,
      venue: 'kalshi',
      price: mid,
      yesBidCents: bestYesBid,
      yesAskCents: yesAsk,
      noBidCents: bestNoBid,
      noAskCents: bestYesBid !== null ? 100 - bestYesBid : null,
      volume: existing?.volume ?? null,
      updatedAt: new Date().toISOString(),
    };
    this.cache.updateMarket(cached);
  }

  private scheduleReconnect(): void {
    if (this.stopped) return;
    if (this.reconnectAttempts >= this.maxReconnectAttempts) {
      this.logger.error('Kalshi WS: max reconnect attempts reached');
      return;
    }

    const delay = Math.min(1000 * Math.pow(2, this.reconnectAttempts), 60000);
    this.reconnectAttempts++;
    this.logger.info('Kalshi WS: reconnecting', {
      attempt: this.reconnectAttempts,
      delayMs: delay,
    });

    this.reconnectTimer = setTimeout(() => this.connectWs(), delay);
  }
}

// ── Helpers ───────────────────────────────────────────────────────────

function normalizePriceTo01(raw: unknown): number | null {
  if (raw === null || raw === undefined) return null;
  const n = typeof raw === 'number' ? raw : typeof raw === 'string' ? Number(raw) : NaN;
  if (!Number.isFinite(n)) return null;
  if (n >= 0 && n <= 1) return n;
  if (n >= 0 && n <= 100) return n / 100;
  return null;
}

function toCents(raw: unknown): number | null {
  if (raw === null || raw === undefined) return null;
  const n = typeof raw === 'number' ? raw : typeof raw === 'string' ? Number(raw) : NaN;
  if (!Number.isFinite(n)) return null;
  if (n >= 0 && n <= 1) return Math.round(n * 100);
  if (n >= 0 && n <= 100) return Math.round(n);
  return null;
}

function computeMid01(bidCents: number | null, askCents: number | null): number | null {
  if (bidCents === null && askCents === null) return null;
  if (bidCents !== null && askCents !== null) return (bidCents + askCents) / 2 / 100;
  return bidCents !== null ? bidCents / 100 : askCents !== null ? askCents / 100 : null;
}

function toLevels(
  raw: Array<[number, number]> | undefined | null
): Array<{ priceCents: number; size: number }> {
  if (!Array.isArray(raw)) return [];
  const out: Array<{ priceCents: number; size: number }> = [];
  for (const row of raw) {
    if (!Array.isArray(row) || row.length < 2) continue;
    const priceCents = Math.round(Number(row[0]));
    const size = Number(row[1]);
    if (!Number.isFinite(priceCents) || !Number.isFinite(size) || size <= 0) continue;
    out.push({ priceCents, size });
  }
  out.sort((a, b) => b.priceCents - a.priceCents);
  return out;
}

/** Parse Retry-After header (seconds or HTTP-date) into milliseconds. */
function parseRetryAfter(header: string | undefined): number {
  if (!header) return DEFAULT_RETRY_AFTER_MS;
  const seconds = Number(header);
  if (Number.isFinite(seconds) && seconds > 0) {
    return Math.min(seconds * 1000, 120000); // cap at 2min
  }
  // Try HTTP-date format
  const date = new Date(header);
  if (!isNaN(date.getTime())) {
    const ms = date.getTime() - Date.now();
    return Math.max(ms, 1000);
  }
  return DEFAULT_RETRY_AFTER_MS;
}

function sleep(ms: number): Promise<void> {
  return new Promise((resolve) => setTimeout(resolve, ms));
}
