/**
 * MarketCache — In-memory cache for market data with event emission.
 *
 * Stores normalized market prices and orderbooks, emitting events when data changes
 * so downstream consumers (StreamBroadcaster) can push updates to clients.
 *
 * Supports TTL-based pruning: entries older than `cacheTtlMs` are automatically
 * removed on a periodic sweep (every 60 seconds).
 */

import { EventEmitter } from 'events';
import type { CachedMarket, CachedOrderbook } from '@galactus/shared';

export interface MarketCacheEvents {
  'market:update': (market: CachedMarket) => void;
  'orderbook:update': (orderbook: CachedOrderbook) => void;
}

export class MarketCache extends EventEmitter {
  private markets: Map<string, CachedMarket> = new Map();
  private orderbooks: Map<string, CachedOrderbook> = new Map();
  private cacheTtlMs: number;
  private pruneTimer: ReturnType<typeof setInterval> | null = null;

  constructor(cacheTtlMs: number) {
    super();
    this.cacheTtlMs = cacheTtlMs;

    // Prune stale entries every 60 seconds
    this.pruneTimer = setInterval(() => this.prune(), 60000);
  }

  /** Update or insert a market entry. Emits 'market:update' on change. */
  updateMarket(market: CachedMarket): void {
    this.markets.set(market.ticker, market);
    this.emit('market:update', market);
  }

  /** Update or insert an orderbook entry. Emits 'orderbook:update' on change. */
  updateOrderbook(orderbook: CachedOrderbook): void {
    this.orderbooks.set(orderbook.ticker, orderbook);
    this.emit('orderbook:update', orderbook);
  }

  /** Get a full snapshot of all cached markets. */
  getMarketSnapshot(): CachedMarket[] {
    return Array.from(this.markets.values());
  }

  /** Get a full snapshot of all cached orderbooks. */
  getOrderbookSnapshot(): CachedOrderbook[] {
    return Array.from(this.orderbooks.values());
  }

  /** Get a single market by ticker. */
  getMarket(ticker: string): CachedMarket | undefined {
    return this.markets.get(ticker);
  }

  /** Get a single orderbook by ticker. */
  getOrderbook(ticker: string): CachedOrderbook | undefined {
    return this.orderbooks.get(ticker);
  }

  /** Cache stats for health endpoint. */
  getStats(): { marketCount: number; orderbookCount: number } {
    return {
      marketCount: this.markets.size,
      orderbookCount: this.orderbooks.size,
    };
  }

  /** Clear all cached data and stop pruning. */
  clear(): void {
    this.markets.clear();
    this.orderbooks.clear();
    if (this.pruneTimer) {
      clearInterval(this.pruneTimer);
      this.pruneTimer = null;
    }
  }

  /** Remove entries whose updatedAt is older than cacheTtlMs. */
  private prune(): void {
    const cutoff = Date.now() - this.cacheTtlMs;
    let prunedMarkets = 0;
    let prunedOrderbooks = 0;

    for (const [ticker, market] of this.markets) {
      if (new Date(market.updatedAt).getTime() < cutoff) {
        this.markets.delete(ticker);
        prunedMarkets++;
      }
    }

    for (const [ticker, orderbook] of this.orderbooks) {
      if (new Date(orderbook.updatedAt).getTime() < cutoff) {
        this.orderbooks.delete(ticker);
        prunedOrderbooks++;
      }
    }

    if (prunedMarkets > 0 || prunedOrderbooks > 0) {
      this.emit('prune', { prunedMarkets, prunedOrderbooks });
    }
  }
}
